/*====================================================================*
 -  Copyright (C) 2001 Leptonica.  All rights reserved.
 -  This software is distributed in the hope that it will be
 -  useful, but with NO WARRANTY OF ANY KIND.
 -  No author or distributor accepts responsibility to anyone for the
 -  consequences of using this software, or for whether it serves any
 -  particular purpose or works at all, unless he or she says so in
 -  writing.  Everyone is granted permission to copy, modify and
 -  redistribute this source code, for commercial or non-commercial
 -  purposes, with the following restrictions: (1) the origin of this
 -  source code must not be misrepresented; (2) modified versions must
 -  be plainly marked as such; and (3) this notice may not be removed
 -  or altered from any source or modified source distribution.
 *====================================================================*/


/*
 *  rotateorth.c
 *
 *      180-degree rotation
 *            PIX     *pixRotate180()
 *
 *      90-degree rotation (both directions)
 *            PIX     *pixRotate90()
 *
 *      LR rotation
 *            PIX     *pixRotateLR()
 *
 *      TB rotation
 *            PIX     *pixRotateTB()
 */

#include <stdio.h>
#include <stdlib.h>

#include "allheaders.h"



/*!
 *  pixRotate180()
 *
 *      Input:  pixd  (<optional>; can be equal to pixs)
 *              pixs
 *      Return: pixd
 *
 *  This prepares for the low-level operation, which is done in-place.
 */
PIX *
pixRotate180(PIX  *pixd,
             PIX  *pixs)
{
l_int32  d;

    PROCNAME("pixRotate180");

    if (!pixs)
	return (PIX *)ERROR_PTR("pixs not defined", procName, pixd);

    d = pixGetDepth(pixs);
    if (d != 1 && d != 2 && d != 4 && d != 8 && d != 16 && d != 32)
	return (PIX *)ERROR_PTR("pixs not in {1,2,4,8,16,32} bpp",
	       procName, pixd);

    if (pixs != pixd) {
	if ((pixd = pixCopy(pixd, pixs)) == NULL)
	    return (PIX *)ERROR_PTR("copy fail", procName, pixd);
    }

    pixRotateLR(pixd, pixd);
    pixRotateTB(pixd, pixd);
    return pixd;
}
    

/*!
 *  pixRotate90()
 *
 *      Input:  pixs
 *              direction (1 = clockwise,  -1 = counter-clockwise)
 *      Return: pixd, or null on error
 */
PIX *
pixRotate90(PIX     *pixs,
            l_int32  direction)
{
l_int32    wd, hd, d, wpls, wpld;
l_uint32  *datas, *datad;
PIX       *pixd;

    PROCNAME("pixRotate90");

    if (!pixs)
	return (PIX *)ERROR_PTR("pixs not defined", procName, NULL);

    d = pixGetDepth(pixs);
    if (d != 1 && d != 2 && d != 4 && d != 8 && d != 16 && d != 32)
	return (PIX *)ERROR_PTR("pixs not in {1,2,4,8,16,32} bpp",
	       procName, NULL);

    hd = pixGetWidth(pixs);
    wd = pixGetHeight(pixs);
    if ((pixd = pixCreate(wd, hd, d)) == NULL)
	return (PIX *)ERROR_PTR("pixd not made", procName, NULL);
    pixCopyColormap(pixd, pixs);
    pixCopyResolution(pixd, pixs);
    pixCopyInputFormat(pixd, pixs);

    datas = pixGetData(pixs);
    wpls = pixGetWpl(pixs);
    datad = pixGetData(pixd);
    wpld = pixGetWpl(pixd);

    rotate90Low(datad, wd, hd, d, wpld, datas, wpls, direction);

    return pixd;
}


/*!
 *  pixRotateLR()
 *
 *      Input:  pixd  (<optional>; can be equal to pixs)
 *              pixs
 *      Return: pixd
 *
 *  This prepares for the low-level operation, which is done in-place.
 */
PIX *
pixRotateLR(PIX  *pixd,
            PIX  *pixs)
{
l_uint8   *tab;
l_int32    w, h, d, wpld;
l_uint32  *datad, *buffer;

    PROCNAME("pixRotateLR");

    if (!pixs)
	return (PIX *)ERROR_PTR("pixs not defined", procName, pixd);

    d = pixGetDepth(pixs);
    if (d != 1 && d != 2 && d != 4 && d != 8 && d != 16 && d != 32)
	return (PIX *)ERROR_PTR("pixs not in {1,2,4,8,16,32} bpp",
	       procName, pixd);

    if (pixs != pixd) {
	if ((pixd = pixCopy(pixd, pixs)) == NULL)
	    return (PIX *)ERROR_PTR("copy fail", procName, pixd);
    }

    w = pixGetWidth(pixd);
    h = pixGetHeight(pixd);
    datad = pixGetData(pixd);
    wpld = pixGetWpl(pixd);

    switch (d)
    {
    case 1:
        if ((tab = makeReverseByteTab1()) == NULL)
  	    return (PIX *)ERROR_PTR("tab not made", procName, pixd);
        break;
    case 2:
        if ((tab = makeReverseByteTab2()) == NULL)
  	    return (PIX *)ERROR_PTR("tab not made", procName, pixd);
        break;
    case 4:
        if ((tab = makeReverseByteTab4()) == NULL)
  	    return (PIX *)ERROR_PTR("tab not made", procName, pixd);
        break;
    default:
        tab = NULL;
        break;
    }

    if ((buffer = (l_uint32 *)CALLOC(wpld, sizeof(l_uint32))) == NULL)
	return (PIX *)ERROR_PTR("buffer not made", procName, pixd);

    rotateLRLow(datad, w, h, d, wpld, tab, buffer);

    FREE((void *)buffer);
    if (tab) FREE((void *)tab);
    return pixd;
}


/*!
 *  pixRotateTB()
 *
 *      Input:  pixd  (<optional>; can be equal to pixs)
 *              pixs
 *      Return: pixd
 *
 *  This prepares for the low-level operation, which is done in-place.
 */
PIX *
pixRotateTB(PIX  *pixd,
            PIX  *pixs)
{
l_int32    h, d, wpld;
l_uint32  *datad, *buffer;

    PROCNAME("pixRotateTB");

    if (!pixs)
	return (PIX *)ERROR_PTR("pixs not defined", procName, pixd);

    d = pixGetDepth(pixs);
    if (d != 1 && d != 2 && d != 4 && d != 8 && d != 16 && d != 32)
	return (PIX *)ERROR_PTR("pixs not in {1,2,4,8,16,32} bpp",
	       procName, pixd);

    if (pixs != pixd) {
	if ((pixd = pixCopy(pixd, pixs)) == NULL)
	    return (PIX *)ERROR_PTR("copy fail", procName, pixd);
    }

    h = pixGetHeight(pixd);
    datad = pixGetData(pixd);
    wpld = pixGetWpl(pixd);

    if ((buffer = (l_uint32 *)CALLOC(wpld, sizeof(l_uint32))) == NULL)
	return (PIX *)ERROR_PTR("buffer not made", procName, pixd);

    rotateTBLow(datad, h, wpld, buffer);

    FREE((void *)buffer);
    return pixd;
}

