/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
/*
**  File  : menu.c
**  Datum : 9-4-92
**  Doel  : plaatsen van group-window en buttons maken voor de verschillende
**          functie en windows
*/

#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "sources.h"
#include "keymap.h"
#include "message.h"
#include "button.h"
#include "symbol.h"
#include "edit.h"
#include "buffer.h"
#include "default.h"
#include "notatype.h"
#include "notation.h"
#include "output.h"
#include "find.h"
#include "editor.h"
#include "menu.h"
#include "popup.h"
#include "fileselc.h"
#include "helpfile.h"
#ifdef PARSER
#include "match.h"
#endif

#define MENUNAME  "MathSpad 0.60"
#define ICONNAME  "MathSpad"
#define POPUPSIZE 1000
#define MAXLINES 50

enum button { WINDOWBUTTON, EDITOPBUTTON, STRUCTUREBUTTON, SELECTIONBUTTON,
	      MISCBUTTON, VERSIONBUTTON, QUITBUTTON, NR_BUTTON };

#define NR_ROWS    1

static
char *menubutton[NR_BUTTON] =
       { "Window", "EditOp", "Structure", "Selection",
	 "Misc.",  "Version","Quit"
       };
static
int menuhelp[NR_BUTTON] =
     { CONSOLEWINDOWHELP, CONSOLEEDITOPHELP, CONSOLESTRUCTUREHELP,
       CONSOLESELECTIONHELP, CONSOLEMISCHELP, CONSOLEVERSIONHELP,
       CONSOLEQUITHELP };


static int rows[NR_ROWS] = { NR_BUTTON };
static MENU mainmenu[NR_BUTTON+1];
static int mainfunc[NR_BUTTON+1];

static void popup_window(void *data, int n);
static void popup_structure(void *data, int n);
static void popup_selection(void *data, int n);
static void popup_join(void *data, int n);
static void popup_latex(void *data, int n);
static void popup_spacing(void *data, int n);
static void popup_editop(void *data, int n);
static void popup_misc(void *data, int n);
static void popup_ispell(void *data, int n);
static void popup_latex_ex(void *data, int n);
static void popup_maple(void *data, int n);
static void popup_version(void *data, int n);
static void popup_quit(void *data, int n);
static void create_popup(void *data, int n);

typedef struct ITEMDESC { 
    char *descr;
    int nr;
    int submenu;
    void *data;
    BTFUNC func;
} ITEMDESC;
static ITEMDESC itemdesc[] = {
    {"Document", 0, 0, NULL, popup_window},
    {"Stencil", 1, 0, NULL, popup_window},
    {"Symbol", 2, 0, NULL, popup_window},
    {"Find", 3, 0, NULL, popup_window},
    {"Buffer", 4, 0, NULL, popup_window},
    {"Default", 5, 0, NULL, popup_window},
    {NULL, 0, CONSOLEWINDOWPINUP, NULL, create_popup},
    {"Insert Expr.", 0, 0, NULL, popup_editop},
    {"Insert Disp.", 1, 0, NULL, popup_editop},
    {"Copy", 2, 0, NULL, popup_editop},
    {"Swap", 3, 0, NULL, popup_editop},
    {"Delete", 4, 0, NULL, popup_editop},
    {"Insert Id.", 5, 0, NULL, popup_editop},
    {"Insert Op.", 6, 0, NULL, popup_editop},
    {"Insert Var.", 7, 0, NULL, popup_editop},
    {"Insert Text.", 8, 0, NULL, popup_editop},
    {NULL, 1, CONSOLEEDITOPPINUP, NULL, create_popup},
    {"Add/Rem ()", 0, 0, NULL, popup_structure},
    {"Reverse", 1, 0, NULL, popup_structure},
    {"Distribute", 2, 0, NULL, popup_structure},
    {"Factorise", 3, 0, NULL, popup_structure},
    {"Apply", 4, 0, NULL, popup_structure},
    {"Group", 5, 0, NULL, popup_structure},
    {"Ungroup", 6, 0, NULL, popup_structure},
    {"Rename", 7, 0, NULL, popup_structure},
    {NULL, 2, CONSOLESTRUCTUREPINUP, NULL, create_popup},
    {"Swap", 0, 0, NULL, popup_selection},
    {"Clear", 1, 0, NULL, popup_selection},
    {"Join", 2, 1, NULL, NULL},
    {NULL, 3, CONSOLESELECTIONPINUP, NULL, create_popup},
    {"T+A->T", 0, 0, NULL, popup_join},
    {"S+A->S", 1, 0, NULL, popup_join},
    {"A+S->A", 2, 0, NULL, popup_join},
    {NULL, 0, CONSOLEJOINPINUP, NULL, NULL},
    {"Word Wrap", 0, 0, NULL, popup_misc},
    {"Output", 1, 1, NULL, popup_misc},
    {"Paste", 2, 0, NULL, popup_misc},
    {"Goto Line", 3, 0, NULL, popup_misc},
    {"Toggle Dots", 4, 0, NULL, popup_misc},
    {"Display Left", 5, 0, NULL, popup_misc},
    {"Display Right", 6, 0, NULL, popup_misc},
    {"Spacing", 7, 1, NULL, popup_misc},
    {"Ispell",9,1,NULL,popup_misc},
    {"Maple",8,1,NULL,popup_misc},
#ifdef PARSER
    {"Parse",10,0,NULL,popup_misc},
#endif
    {"LaTeX Last File", 12, 1, NULL, popup_misc},
    {NULL, 4, CONSOLEMISCPINUP, NULL, create_popup},
    {"LaTeX", 1, 0, NULL, popup_latex},
    {"Plain", 2, 0, NULL, popup_latex},
    {"Ascii", 3, 0, NULL, popup_latex},
    {NULL, 0, 0, NULL, NULL},
    {"Increase", 1, 0, NULL, popup_spacing},
    {"Decrease", 2, 0, NULL, popup_spacing},
    {"Reset", 3, 0, NULL, popup_spacing},
    {NULL, 0, 0, NULL, NULL},
    {"Start Ispell",1,0,NULL,popup_ispell},
    {"Check Target",2,0,NULL,popup_ispell},
    {"Accept Word",3,0,NULL,popup_ispell},
    {"Add Word",4,0,NULL,popup_ispell},
    {"TeX Mode",5,0,NULL,popup_ispell},
    {"Normal Mode",6,0,NULL,popup_ispell},
    {"Save Dictionary",7,0,NULL,popup_ispell},
    {NULL, 0, 0, NULL, NULL},
    {"Start Maple", 1,0,NULL,popup_maple},
    {"Evaluate", 2,0,NULL,popup_maple},
    {"Simplify", 3,0,NULL,popup_maple},
    {"Normalize", 4,0,NULL,popup_maple},
    {"Raw Command", 5,0,NULL,popup_maple},
    {"Plot", 8,0,NULL,popup_maple},
    {NULL, 0, 0, NULL, NULL},
    {"LaTeX", 1, 0, NULL, popup_latex_ex},
    {"Preview", 2, 0, NULL, popup_latex_ex},
    {"Print", 3, 0, NULL, popup_latex_ex},
    {NULL,0,0,NULL,NULL},
    {NULL, 0, CONSOLEVERSIONPINUP, NULL, popup_version},
    {"Quit", 0, 0, NULL, popup_quit},
    {"Fast Quit", 1, 0, NULL, popup_quit},
    {"New Project", 2, 0, NULL, popup_quit},
    {"Save Project", 3, 0, NULL, popup_quit},
    {NULL, 6, CONSOLEQUITPINUP, NULL, create_popup}
};

static Window menuwin, miniwin;
static void *mini_info;
static int win_xpos=0, win_ypos=0;
static int win_width=0, win_height=0, window_width;
static int miniwin_width, miniwin_height;
static int line_y;
static Bool searching = False;
static char popuptext[POPUPSIZE];
static int key_map = 0, search_map=0, print_map = 0, y_n_map=0;

#define sub_width(A)   (A) - INTERSPACE*2
#define sub_height(A)  line_height()
#define window_height  NR_ROWS*(button_height+BINTERSPACE) + line_height() \
                       + INTERSPACE*2

int quit_sequence = False;

static void set_name(char *projectname)
{
    XTextProperty prop_name;
    char name[500];
    char *filename;
    int i;

    if (projectname) {
	filename = strrchr(projectname,'/');
	if (!filename) filename=projectname; else filename++;
	sprintf(name, "%s : %s", MENUNAME, filename);
	i=strlen(name);
	if (i>4 && !strcmp(name+i-4,".mpj")) name[i-4]='\0';
    } else
	strcpy(name, MENUNAME);
    filename=name;
    if (!XStringListToTextProperty(&filename, 1, &prop_name)) return;
    XSetWMName(display, menuwin, &prop_name);
    filename=ICONNAME;
    if (!XStringListToTextProperty(&filename, 1, &prop_name)) return;
    XSetWMIconName(display, menuwin, &prop_name);
}

static void notation_make_backups(void *data, int dumps)
{
    int i=0;
    char *c;
    Bool opened;

    while ((i = get_next_filename(i, &c, &opened))>=0)
	auto_save_window(i, dumps);
}

static void draw_search(void);

static void menu_draw(void *data)
{
    Window win = *((Window*)data);
    if (win == menuwin)
	XDrawLine(display, menuwin, get_GC(Normal,0),
		  0, line_y, win_width, line_y);
    else
	if (searching)
	    draw_search();
	else
	    draw_message();
}

static void menu_layout_change(void *data)
{
    XSizeHints hints;
    clear_tab_positions();
    push_fontgroup(POPUPFONT);
    win_height = window_height;
    miniwin_height = sub_height(window_height);
    XResizeWindow(display, menuwin, win_width, win_height);
    hints.flags = (PMinSize | PMaxSize);
    hints.min_height = hints.max_height = win_height;
    hints.min_width = window_width;
    hints.max_width = display_width;
    XSetWMNormalHints(display, menuwin, &hints);
    XResizeWindow(display, miniwin, miniwin_width, miniwin_height);
    XClearArea(display, miniwin, 0, 0, 0, 0, True);
    resize_window(mini_info, miniwin_width, miniwin_height);
    if (searching) draw_search();
    pop_fontgroup();
}

static void menu_state(void *data, int *posx, int *posy, int *width,
		       int *height, int *as_icon, int *sbpos, char **string)
{
    *posx = win_xpos;
    *posy = win_ypos;
    *height = win_height;
    *width = win_width;
    *as_icon = menu_iconized;
    *sbpos = 0;
    *string = NULL;
}

static void menu_resize(void *data, XConfigureEvent *event)
{
    if (event->window == menuwin) {
	int x,y;
	window_manager_added(menuwin, &x,&y);
	win_xpos = event->x-x;
	win_ypos = event->y-y;
	win_width = event->width;
	push_fontgroup(POPUPFONT);
	miniwin_width = sub_width(win_width);
	win_height = event->height;
	XResizeWindow(display, miniwin, miniwin_width, miniwin_height);
	resize_window(mini_info, miniwin_width, miniwin_height);
	menu_draw(data);
	pop_fontgroup();
    }
}

static void handle_new_version(void *data,  int ivnr)
{
    if (ivnr>=0) new_version(ivnr);
}

static void handle_new_id_font(void *data,  int nfnr)
{
    new_id_font(nfnr);
}

static Bool make_id_popup(void)
{
    int nr,i,k;
    MENU idfont;

    nr = ps_id_font();
    if (nr<0) return False;
    for (i=1,k=1; i<256;i++)
	if (font_opentexttex(i,0) || font_openmathtex(i,0)) k++;
    idfont.line = (MENULINE*) malloc(sizeof(MENULINE)*k);
    for (i=0; i<k; i++) {
	idfont.line[i].func=handle_new_id_font;
	idfont.line[i].fdata=NULL;
	idfont.line[i].submenu=NULL;
    }
    idfont.selline=idfont.x=idfont.y=-1;
    idfont.defline=0;
    idfont.stickopt=idfont.freesub=True;
    idfont.endfunc=NULL;
    idfont.help=NULL;
    idfont.sticky=False;
    idfont.nr=k;
    idfont.title="Identifier";
    idfont.transwin=menuwin;
    idfont.line[0].txt = char2Char("Default");
    idfont.line[0].len = strlen("Default");
    idfont.line[0].fint=0;
    k=1;
    for (i=1;i<256; i++) {
	if (font_opentexttex(i,0) || font_openmathtex(i,0)) {
	    char *h=font_name(i);
	    if (nr==i) idfont.defline=k;
	    idfont.line[k].len = strlen(h);
	    idfont.line[k].txt = char2Char(h);
	    idfont.line[k].fint = i;
	    k++;
	}
    }
    return (popup_make(&idfont)!=NULL);
}

static void popup_editop(void *data, int n)
{
    switch (n) {
    case 0: insert_expr(0,1); break;
    case 1: insert_disp(0,1); break;
    case 2: copy_region(0,0); break;
    case 3: swap_region(0,0); break;
    case 4: remove_region(0,1); break;
    case 5: insert_id(0,1); break;
    case 6: insert_op(0,1); break;
    case 7: insert_var(0,1); break;
    case 8: insert_text(0,1); break;
    default: break;
    }
}

static void popup_structure(void *data, int n)
{
    switch (n) {
    case 0: switch_parens(0,0); break;
    case 1: commute(0,0); break;
    case 2: distribute(0,0); break;
    case 3: factorise(0,0); break;
    case 4: apply(0,0); break;
    case 5: lower_region(0,0);  break;
    case 6: raise_node(0,0);    break;
    case 7: rename_id(0,0); break;
    default: break;
    }
}

static void popup_window(void *data, int n)
{
    switch (n) {
    case 0: if (can_open_edit) edit_open(); break;
    case 1: if (can_open_notation) notation_open(); break;
    case 2: if (can_open_symbol) symbol_open(); break;
    case 3: if (can_close_find) find_close();
	    if (can_open_find) find_open();
	    break;
    case 4: if (can_close_buffer) buffer_close();
	    if (can_open_buffer) buffer_open();
	    break;
    case 5: if (can_close_default) default_close();
	    if (can_open_default) default_open();
	    break;
    default: break;
    }
}

static void popup_version(void *data, int n)
{
    if (mouse_button == Button3) {
	unsigned int vnr = 0;
	int nr = ps_notation(&vnr);
	if (nr>=0) {
	    if (!make_notation_popup(nr, vnr, handle_new_version,
				     menubutton[VERSIONBUTTON], True))
		new_version(-1);
	} else {
	    if (!make_id_popup())
		message(MESSAGE, "No template or identifier selected.");
	}
    } else
	new_version(-1);
}

static void popup_selection(void *data, int n)
{
    switch (n) {
    case 0: swap_selections(0,mouse_button); break;
    case 1: unset_select(0,mouse_button); break;
    case 2: join_selections(0,mouse_button); break;
    default: break;
    }
}

static void popup_join(void *data, int n)
{
    switch (n) {
    case 0: join_selections(0,1); break;
    case 1: join_selections(0,2); break;
    case 2: join_selections(0,3); break;
    default: break;
    }
}

static void latex_select(KEYCODE keycode, Index arg);
static void ask_selection(KEYCODE keycode, Index arg);
static void ask_latex_line(KEYCODE keycode, Index arg);
static void switch_textdots(KEYCODE keycode, Index arg);

static void popup_misc(void *data, int n)
{
    switch (n) {
    case 0: word_wrap_selection(1); break;
    case 1: latex_select(0,1); break;
    case 2: ask_selection(0,1); break;
    case 3: ask_latex_line(0,1); break;
    case 4: switch_textdots(0,1); break;
    case 5: display_left(0,1); break;
    case 6: display_right(0,1); break;
    case 7: increase_spacing(0,1); break;
    case 8: open_program("maplescript %i &", "Maple Shell"); break;
    case 9: open_program("ispellscript %i &", "Ispell Shell"); break;
#ifdef PARSER
    case 10:
	{
	    char *psel=NULL;
	    tex_set_string(&psel);
	    tex_placeholders(ON);
	    tex_mode(MPTEX);
	    latex_selection(1);
	    tex_unset();
	    if (psel && parse_text(psel)) include_selection();
	    cleanup_nodestack();
	    free(psel);
	}
	break;	
#endif
    case 12: open_program("latexscript %i &", "LaTeX Shell"); break;
    default: break;
    }
}

static void popup_latex_ex(void *data, int n)
{
    /* these functions should open a fileselector */
    switch (n) {
    case 1: open_program("latexscript %i &", "LaTeX Shell"); break;
    case 2: system("previewscript &"); break;
    case 3: system("printscript &"); break;
    default: break;
    }
}

static void popup_maple(void *data, int n)
{
    switch (n) {
    case 1: open_program("maplescript %i &", "Maple Shell"); break;
    case 2: edit_string_to_proces("eval(%1);\n", "Maple Shell"); break;
    case 3: edit_string_to_proces("simplify(%1);\n", "Maple Shell"); break;
    case 4: edit_string_to_proces("normal(%1);\n", "Maple Shell"); break;
    case 5: edit_string_to_proces("%1;\n", "Maple Shell"); break;
    case 8: edit_string_to_proces("plotsetup(gif);\nplot(%1,%2);\n!mapleshow plot.gif ;\n", "Maple Shell"); break;
    default: break;
    }
}

static void popup_ispell(void *data, int n)
{
    switch (n) {
    case 1: open_program("ispellscript %i &", "Ispell Shell"); break;
    case 2: edit_string_to_proces("^%P^%T%1\n", "Ispell Shell"); break;
    case 3: edit_string_to_proces("@%P@%T%1\n", "Ispell Shell"); break;
    case 4: edit_string_to_proces("*%P*%T%1\n", "Ispell Shell"); break;
    case 5: edit_string_to_proces("+\n", "Ispell Shell"); break;
    case 6: edit_string_to_proces("-\n", "Ispell Shell"); break;
    case 7: edit_string_to_proces("#\n", "Ispell Shell"); break;
    default: break;
    }
}

static void popup_latex(void *data, int n)
{
    latex_select(0,n);
}

static void popup_spacing(void *data, int n)
{
    switch (n) {
    case 1: increase_spacing(0,1); break;
    case 2: decrease_spacing(0,1); break;
    case 3: reset_spacing(0,1); break;
    default: break;
    }
}

static void save_project_name(void *data, char *c)
{
    if (!strcmp(".mpj", c+strlen(c)-4)) c[strlen(c)-4]='\0';
    save_project(c);
    set_name(project_name);
}

static void popup_quit(void *data, int n)
{
    char *c;
    switch (n) {
    case 0: menu_close(); break;
    case 1: menu_bad_end(NULL); break;
    case 2:
	message(CLICKREMARK, "Sorry, not yet implemented.");
	break;
    case 3:
	c = concat(homedir, "mathspad/");
	fileselc_open(save_project_name, NULL,
		      "Save the current state in a project file.\n"
		      "To be able to find it when you start again,\n"
		      "it will be placed in your mathspad directory.",
		      c, "*.mpj", project_name, menuwin);
	break;
    default: break;
    }
}

static void create_popup(void *data, int n)
{
    if (mouse_button == Button3) {
	(void) popup_make(mainmenu+n);
    } else {
	MENULINE *mp = mainmenu[n].line+mainmenu[n].defline;
	(*(mp->func))(mp->fdata, mp->fint);
    }
}


static void menu_press(void *data, XButtonEvent *event)
{
    get_motion_hints(miniwin, -1);
    other_window(mini_info);
}

static void menu_release(void *data, XButtonEvent *event)
{
    stop_motion_hints();
    call_func(K_escape);
    use_map(0, key_map);
}

static void menu_motion(void *data, int x, int y)
{
}

static void menu_iconize(void *data)
{
    menu_iconized = True;
}

static void menu_deiconize(void *data)
{
    menu_iconized = False;
}

static int menu_last_pos(int *x, int *y, int *w, int *h)
{
    *x = win_xpos;
    *y = win_ypos;
    *h = win_height;
    *w = win_width;
    return False;
}

static void menu_set_last_pos(int x, int y, int w, int h)
{
    win_xpos = x;
    win_ypos = y;
    win_height = h;
    win_width = w;
}    

FUNCTIONS menufuncs = {
    menu_bad_end, menu_draw, menu_resize, menu_press, menu_release,
    menu_motion, menu_iconize, menu_deiconize, NULL, NULL, menu_layout_change,
    notation_make_backups, menu_open, menu_state, NULL, NULL, menu_last_pos,
    menu_set_last_pos };

void menu_set_command(void)
{
    char *newarg[3];
    newarg[0] = arguments[0];
    newarg[1] = "-project";
    newarg[2] = project_name;
    XSetCommand(display, menuwin, newarg, 3);
}

void menu_init()
{
}

void menu_open(int x, int y, int w, int h, int icon, int s, char *str)
{
    XSetWindowAttributes menu_attr;
    unsigned long menu_mask;
    int miniwin_x, miniwin_y;
    int i,j,mnleft,b_x, b_y;
    int row_width;
    XSizeHints size_hints;
    MENU makemenu[50];
    int makefunc[50];

    j=0;
    i=0;
    mnleft=0;
    while (i<NR_BUTTON || mnleft) {
	int l=0,k=0,n=0,omnl;
	Char *c;
	MENULINE *mp;
	omnl=mnleft;
	while (itemdesc[j+k].descr) {
	    l+=strlen(itemdesc[j+k].descr)+1;
	    k++;
	}
	n=j+k;
	makefunc[i]=j+k;
	makemenu[i].title=NULL;
	makemenu[i].help= helpname[itemdesc[j+k].submenu];
	makemenu[i].nr=k;
	makemenu[i].defline=0;
	makemenu[i].selline=-1;
	makemenu[i].stickopt=True;
	makemenu[i].freesub=False;
	makemenu[i].sticky=False;
	makemenu[i].width=makemenu[i].height=makemenu[i].x=makemenu[i].y=-1;
	makemenu[i].endint=0;
	makemenu[i].endfunc=NULL;
	makemenu[i].mainmenu=NULL;
	makemenu[i].submenu=NULL;
	if (k) {
	    makemenu[i].line = (MENULINE*) malloc(sizeof(MENULINE)*k);
	    c = (Char*) malloc(sizeof(Char)*l);
	    mp = makemenu[i].line;
	    while (itemdesc[j].descr) {
		int dl=0;
		mp->txt = c;
		while ((*c++ = itemdesc[j].descr[dl++]));
		mp->len = dl;
		mp->height= (itemdesc[j].submenu?j:0);
		mp->width = itemdesc[j].submenu;
		mp->fint = itemdesc[j].nr;
		mp->fdata = itemdesc[j].data;
		mp->func = itemdesc[j].func;
		mp->submenu=NULL;
		if (itemdesc[j].submenu) {
		    mnleft++;
		    makemenu[i].endint++;
		}
		mp++;
		j++;
	    }
	}
	j++;
	while (i && !makemenu[i].endint && makemenu[i-1].endint) {
	    MENULINE *ml;
	    ml= makemenu[i-1].line;
	    makemenu[i-1].endint--;
	    mnleft--;
	    while (!ml->width || ml->submenu) ml++;
	    ml->width=0;
	    ml->submenu = (MENU*) malloc(sizeof(MENU));
	    *ml->submenu=makemenu[i];
	    ml->submenu->title=itemdesc[ml->height].descr;
	    ml->height=0;
	    i--;
	}
	i++;
    }
    for (i=0; i<NR_BUTTON; i++) {
	mainmenu[i]=makemenu[i];
	mainmenu[i].title=menubutton[i];
	mainfunc[i]=makefunc[i];
    }
    j=0;
    window_width = 0;
    set_change_function(update_selections);
    set_change_function(refresh_all);
    push_fontgroup(POPUPFONT);
    for (i=0; i<NR_ROWS; i++) {
	row_width = 0;
	while (j<rows[i]) {
	    row_width += button_width(menubutton[j])+BINTERSPACE;
	    j++;
	}
	if (window_width<row_width) window_width=row_width;
    }
    if (win_width<window_width) win_width = window_width;
    if (win_height<window_height) win_height = window_height;
    size_hints.min_height = size_hints.max_height = window_height;
    size_hints.min_width  = window_width;
    size_hints.max_width  = display_width;
    if (w>win_width) win_width = w;
    if (h>win_height) win_height = h;
    if (w) {
	win_xpos = x;
	win_ypos = y;
    } else if (!win_xpos && !win_ypos) {
	win_xpos = (display_width - win_width)/2;
	win_ypos = (display_height - win_height)/2;
    }
    line_y = NR_ROWS * (button_height+BINTERSPACE);
    miniwin_width = sub_width(win_width);
    miniwin_height = sub_height(win_height);
    miniwin_x = INTERSPACE;
    miniwin_y = line_y+ INTERSPACE;
    menu_mask = (CWBackPixel | CWBorderPixel | CWBitGravity | CWEventMask |
		 CWColormap);
    menu_attr.background_pixel = white_pixel;
    menu_attr.colormap=colormap;
    menu_attr.border_pixel = black_pixel;
    menu_attr.bit_gravity = NorthWestGravity;
    menu_attr.event_mask = (ExposureMask | ButtonPressMask |
			    ButtonReleaseMask | KeyPressMask |
			    PropertyChangeMask | StructureNotifyMask |
			    VisibilityChangeMask);
    menuwin = XCreateWindow(display, root_window,
			    win_xpos, win_ypos, win_width, win_height,
			    BORDERWIDTH, CopyFromParent, InputOutput,
			    visual,
			    menu_mask, &menu_attr);
    miniwin = XCreateWindow(display, menuwin, miniwin_x, miniwin_y,
			    miniwin_width, miniwin_height,
                            0, CopyFromParent, InputOutput,
			    visual,
                            menu_mask, &menu_attr);
    if (w)
	size_hints.flags = USPosition | USSize | PMinSize | PMaxSize;
    else
	size_hints.flags = PPosition | PSize | PMinSize | PMaxSize;
    wm_hints.initial_state = ((iconic || icon) ? IconicState : NormalState);
    wm_hints.input = True;
    wm_hints.icon_pixmap = icon_pixmap;
    wm_hints.window_group = menuwin;
    wm_hints.flags = StateHint | IconPixmapHint | InputHint | WindowGroupHint;

    class_hints.res_name = progname;
    class_hints.res_class = "MathEdit";

    XSetWMProperties(display, menuwin, NULL, NULL,
		     arguments, number_of_arguments,
		     &size_hints, &wm_hints, &class_hints);
    set_name(project_name);
    XSetWMProtocols(display, menuwin, protocol, 2);

    for (j=0; j<NR_BUTTON; j++) mainmenu[j].transwin=menuwin;
    i=0;
    j=0;
    b_x = b_y = BINTERSPACE/2;
    if (add_window(menuwin, MENUWINDOW, root_window,
		   NULL, helpname[CONSOLEHELP])) {
	while (j<NR_BUTTON &&
	       button_make(itemdesc[mainfunc[j]].nr,menuwin,menubutton[j],
			   &b_x,b_y,1,NULL, helpname[menuhelp[j]], NULL,NULL,
			   itemdesc[mainfunc[j]].func,
			   itemdesc[mainfunc[j]].func,
			   itemdesc[mainfunc[j]].func, NULL)) {
	    j++;
	    b_x+=BINTERSPACE;
	    if (j==rows[i]) {
		b_x = BINTERSPACE/2;
		b_y += button_height+BINTERSPACE;
		i++;
	    }
	}
	if (add_window(miniwin, MENUWINDOW, menuwin, NULL,
		       helpname[CONSOLEHELP])) j++;
    }
    if (j<NR_BUTTON+1) {
	XDestroyWindow(display, menuwin);
	XDestroyWindow(display, miniwin);
	message(EXIT -1, "Unable to open menu window.");
    }
    set_selection_window(menuwin);
    mini_info = open_miniwindow( &miniwin, miniwin_width, miniwin_height);
    menu_is_open = True;
    set_message_window(&miniwin);
    XMapSubwindows(display, menuwin);
    XMapWindow(display, menuwin);
    pop_fontgroup();
}

void menu_close(void)
{
    void *data, *unsaved;
    int i;

    i=0;
    unsaved = NULL;
    quit_sequence = True;
    while ((data = next_data_with_type(MAINEDITWINDOW, &i)))
	if (edit_saved(data))
	    edit_close(data);
	else {
	    unsaved = data;
	    i++;
	}
    if (unsaved==NULL) {
	/* there are no unsave documents. */
	if ((i=notation_not_saved(0)))
	    notation_confirm_backup(i);
	else
	    server_close();
    } else
	edit_close(unsaved);
}

void menu_bad_end(void *data)
{
    void *dt;
    int i;

    i=0;
    menu_is_open = False;
    while ((dt = next_data_with_type(MAINEDITWINDOW, &i)))
	edit_bad_end(dt);
    notation_make_backups(NULL,0);
    server_close();
}

static void menu_keysymbol(KEYCODE keycode, Index arg)
{
    Char selchar;

    selchar = symbol_last();
    if (selchar) {
	insert_symbol(0, selchar);
    }
}

static void menu_notation_shortcut(KEYCODE keycode, Index arg)
{
    char *name = get_string_arg(arg);
    int usenota;
    int i=0,l;

    if (!name) return;
    l = strlen(name);
    if (l>2 && name[l-2]==':' && name[l-1]>='1' && name[l-1]<='9') {
	i = name[l-1]-'0';
	name[l-2] = '\0';
    }
    usenota = notation_with_name(name);
    if (usenota>=0) {
	usenota = nnr_vnr2innr(usenota, 0);
	if (i>0 && i<= which_notation(usenota)->versions)
	    usenota = which_notation(usenota)->vers[i-1].ivnr;
	insert_notation(usenota);
    }
    if (i) name[l-2]=':';
}

static void menu_notation(KEYCODE keycode, Index arg)
{
    int usenota = notation_last();

    if (usenota >=0) insert_notation(usenota);
}

static void ask_selection(KEYCODE keycode, Index arg)
{
    get_wm_selection();
}

#define SEARCH 0
#define NOT_FOUND 1
#define WRAP_SEARCH 2
#define WRAP_NOT_FOUND 3

#define MAXLEN 500

static Char searchstr[MAXLEN], replacestr[MAXLEN], oldsearchstr[MAXLEN];
static int search_status = SEARCH;
static int stackkind[256];
static int stackd=0;
static Bool find_notation = False;
static Bool backward_search = False;
static Bool in_search = True;
static Bool is_search = True;
static Bool y_n_option = False;
static int searchlen = 0, replacelen = 0, oldsearchlen = 0;

#define stackcode(A) ((search_status<<11) + (backward_search<<10)+(A))
#define statusfromstack(A) (stackkind[A]>>11)
#define backfromstack(A) ((stackkind[A]>>10)&0x1)
#define numberfromstack(A) (stackkind[A]&0x3FF)

static void draw_search(void)
{
    int i=0;

    push_fontgroup(POPUPFONT);
    set_output_window(&miniwin);
    out_clear();
    switch (search_status) {
    case NOT_FOUND:
	out_string("Failing ");
	break;
    case WRAP_SEARCH:
	out_string("Wrapped ");
	break;
    case WRAP_NOT_FOUND:
	out_string("Failing wrapped ");
	break;
    default:
	break;
    }
    if (is_search) {
	Char *c;
	if (backward_search)
	    out_string("I-search backward :");
	else
	    out_string("I-search :");
	if (find_notation)
	    c = stencil_screen(searchstr[0]);
	else {
	    c = searchstr;
	    searchstr[searchlen]=0;
	}
	while (c[i]) {
	    if  (!IsNewline(searchstr[i]))
		out_char(searchstr[i]);
	    else
		out_string("^J");
	    i++;
	}
	out_cursor(CURSOR);
    } else {
	Char *c;
	out_string("Query replace");
	if (in_search) out_char(':');
	out_char(' ');
	if (find_notation)
	    c = stencil_screen(searchstr[0]);
	else {
	    c = searchstr;
	    searchstr[searchlen]=0;
	}
	while (c[i]) {
	    if  (!IsNewline(c[i]))
		out_char(c[i]);
	    else
		out_string("^J");
	    i++;
	}
	if (in_search)
	    out_cursor(CURSOR);
	else {
	    out_string(" with");
	    if (!y_n_option) out_char(':');
	    out_char(' ');
	    if (find_notation)
		c = stencil_screen(replacestr[0]);
	    else {
		c = replacestr;
		replacestr[replacelen]=0;
	    }
	    i=0;
	    while (c[i]) {
		if  (!IsNewline(c[i]))
		    out_char(c[i]);
		else
		    out_string("^J");
		i++;
	    }
	    if (!y_n_option)
		out_cursor(CURSOR);
	    else {
		out_string(": y,n,q or ! ");
		out_cursor(CURSOR);
	    }
	}
    }
    unset_output_window();
    pop_fontgroup();
}

static void end_search(KEYCODE code, Index count)
{
    int i=0;

    clear_message(True);
    reset_map(0,0);
    searching = False;
    in_search = is_search = True;
    oldsearchlen = 0;
    while (i<searchlen)
	oldsearchstr[oldsearchlen++] = searchstr[i++];
    oldsearchstr[i] = 0;
    oldsearchstr[i+1] = find_notation;
    find_notation = y_n_option = False;
    search_status = SEARCH;
    backward_search = False;
    searchstr[0] = searchlen = 0;
    replacestr[0] = replacelen = 0;
    clear_stack();
    stackd=0;
}

static void end_search_oldpos(KEYCODE code, Index count)
{
    clear_stack_and_use();
    end_search(code,count);
}

static void add_search_char(Char c, Index count)
{
    int i = count;

    if (!IsNewline(c) || is_search) {
	while (i && searchlen<MAXLEN-1) {
	    searchstr[searchlen++] = c;
	    i--;
	}
	searchstr[searchlen]=0;
	if (is_search) {
	    stack_position();
	    stackkind[stackd++] = stackcode(count-i);
	    switch (search_status) {
	    case SEARCH:
	    case WRAP_SEARCH:
		if ((!backward_search && !find_string(searchstr)) ||
		    (backward_search && !find_backward_string(searchstr)))
		    search_status++;
		break;
	    default:
		break;
	    }
	}
    } else {
	in_search = False;
	replacestr[0] = replacelen = 0;
    }
    draw_search();
}

static void add_replace_char(Char c, Index count)
{
    int i = count;

    if (!IsNewline(c)) {
	while (i && replacelen<MAXLEN-1) {
	    replacestr[replacelen++] = c;
	    i--;
	}
	replacestr[replacelen] = 0;
	draw_search();
    } else {
	if (!find_replace(searchstr)) {
	    end_search(0,0);
	    message(MESSAGE, "Done.");
	} else {
	    y_n_option = True;
	    use_map(0, y_n_map);
	    draw_search();
	}
    }
}

static void add_char(KEYCODE code, Index count)
{
    if (!find_notation)
	if (in_search)
	    add_search_char(code, count);
	else
	    add_replace_char(code, count);
}

static void add_tab(KEYCODE code, Index count)
{
    add_char(Rtab, count);
}

static void add_return(KEYCODE code, Index count)
{
    add_char(Newline, 1);
}

static void add_sym(KEYCODE code, Index count)
{
    Char selchar;

    selchar = symbol_last();
    if (selchar && !find_notation)
	if (in_search)
	    add_search_char( selchar, 1);
	else
	    add_replace_char(selchar, 1);
}

static void add_search_notation(KEYCODE code, Index count)
{
    int usenota = notation_last();

    if (usenota>=0 && ((in_search && !searchlen) ||
		      (find_notation && !is_search && !replacelen))) {
	if (in_search) {
	    searchstr[0] = usenota;
	    searchlen = 1;
	    find_notation = True;
	    replacestr[0] = replacelen = 0;
	    in_search = is_search;
	    if (is_search) {
		stack_position();
		stackkind[stackd++] = stackcode(1);
		switch (search_status) {
		case SEARCH:
		case WRAP_SEARCH:
		    if ((!backward_search && !find_stencil(searchstr[0])) ||
			(backward_search && !find_backward_stencil(searchstr[0])))
			search_status++;
		    break;
		default:
		    break;
		}
	    }
	    draw_search();
	} else {
	    replacestr[0]=usenota;
	    replacelen = 1;
	    if (!find_replace_stencil(searchstr[0])) {
		end_search(0,0);
		message(MESSAGE, "Done.");
	    } else {
		y_n_option = True;
		use_map(0, y_n_map);
		draw_search();
	    }
	}
    }
}

static void remove_search_char(KEYCODE code, Index count)
{
    if (is_search) {
	use_stack();
	if (stackd) {
	    stackd--;
	    count = numberfromstack(stackd);
	    search_status = statusfromstack(stackd);
	    backward_search = backfromstack(stackd);
	}
    }
    if (in_search) {
	if (find_notation) {
	    searchlen = 0;
	    find_notation = False;
	} else
	    if (count>searchlen)
		searchlen = 0;
	    else
		searchlen -= count;
	searchstr[searchlen]=0;
    } else {
	if (count>replacelen)
	    replacelen = 0;
	else
	    replacelen -= count;
	replacestr[replacelen] = 0;
    }
    draw_search();
}

static void positive_yn(KEYCODE code, Index count)
{
    if (find_notation)
	replace_notation(searchstr[0], replacestr[0]);
    else
	replace_string(searchstr, replacestr);
    if ((find_notation && !findnext_replace_stencil(searchstr[0])) ||
	(!find_notation && !findnext_replace(searchstr)))
	end_search(0,0);
    else
	draw_search();
}

static void negative_yn(KEYCODE code, Index count)
{
    if ((find_notation && !findnext_replace_stencil(searchstr[0])) ||
	(!find_notation && !findnext_replace(searchstr)))
	end_search(0,0);
    else
	draw_search();
}

static void replace_all_yn(KEYCODE code, Index count)
{
    if (find_notation)
	replace_all_notation(searchstr[0], replacestr[0]);
    else
	replace_all(searchstr, replacestr);
    end_search(0,0);
}

static void start_find(KEYCODE c, Index count)
{
    searching = True;
    stack_position();
    draw_search();
    use_map(0, search_map);
}

static void start_replace(KEYCODE c, Index count)
{
    stack_position();
    is_search = False;
    draw_search();
    use_map(0, search_map);
}

static void start_backward_find(KEYCODE c, Index count)
{
    searching = True;
    backward_search = True;
    stack_position();
    draw_search();
    use_map(0,search_map);
}

static void do_find(KEYCODE c, Index count)
{
    int i=0;

    if (is_search) {
	if (!searchlen) {
	    while (i<oldsearchlen) {
		searchstr[searchlen++] = oldsearchstr[i++];
	    }
	    searchstr[i]=0;
	    find_notation = oldsearchstr[i+1];
	}
	stack_position();
	stackkind[stackd++] = stackcode(i);
	if (backward_search) {
	    search_status = SEARCH;
	    backward_search=False;
	}
	switch (search_status) {
	case SEARCH:
	case WRAP_SEARCH:
	    if ((find_notation && !findnext_stencil(searchstr[0])) ||
		(!find_notation && !findnext_string(searchstr)))
		search_status++;
	    break;
	case NOT_FOUND:
	    if ((find_notation && !findwrap_stencil(searchstr[0])) ||
		(!find_notation && !findwrap_string(searchstr)))
		search_status = WRAP_NOT_FOUND;
	    else
		search_status++;
	    break;
	case WRAP_NOT_FOUND:
	    if ((find_notation && findwrap_stencil(searchstr[0])) ||
		(!find_notation && findwrap_string(searchstr)))
		search_status--;
	    break;
	default:
	    break;
	}
	draw_search();
    }
}

static void do_find_backward(KEYCODE c, Index count)
{
    int i=0;

    if (is_search) {
	if (!searchlen) {
	    while (i<oldsearchlen) {
		searchstr[searchlen++] = oldsearchstr[i++];
	    }
	    searchstr[i]=0;
	    find_notation = oldsearchstr[i+1];
	}
	stack_position();
	stackkind[stackd++] = stackcode(i);
	if (!backward_search) {
	    search_status = SEARCH;
	    backward_search = True;
	}
	switch (search_status) {
	case SEARCH:
	case WRAP_SEARCH:
	    if ((find_notation && !findprev_stencil(searchstr[0])) ||
		(!find_notation && !findprev_string(searchstr)))
		search_status++;
	    break;
	case NOT_FOUND:
	    if ((find_notation && !findwrap_backward_stencil(searchstr[0])) ||
		(!find_notation && !findwrap_backward_string(searchstr)))
		search_status = WRAP_NOT_FOUND;
	    else
		search_status++;
	    break;
	case WRAP_NOT_FOUND:
	    if ((find_notation && !findwrap_backward_stencil(searchstr[0])) ||
		(!find_notation && !findwrap_backward_string(searchstr)))
		search_status--;
	    break;
	default:
	    break;
	}
	draw_search();
    }
}

static void latex_select(KEYCODE keycode, Index arg)
{
    if (set_wm_selection()) {
	tex_set_string(&latexselection);
	tex_placeholders(ON);
	switch (arg) {
	case 2: tex_mode(PLAINTEX); break;
	case 3: tex_mode(ASCII);    break;
	case 1: tex_mode(MPTEX);    break;
	default:break;
	}
	latex_selection(1);
	tex_unset();
	set_clipboard();
	message(MESSAGE, "Selection converted to LaTeX.");
    } else
	message(ERROR, "Unable to obtain the selection.");
}

static void make_project(KEYCODE keycode, Index arg)
{
    popup_quit(NULL, 3);
}

static void filter_linenr(char *name)
{
    int n;

    if (sscanf(name, "%i", &n)) {
	goto_latex_line(0, n);
    }
    message(MESSAGE, "");
}

static void ask_latex_line(KEYCODE keycode, Index arg)
{
    construct_string("Go to LaTeX-line: ", "", filter_linenr);
}

static void filter_savetime(char *name)
{
    int n;

    if (sscanf(name, "%i", &n)) {
	set_save_period(n);
    }
    message(MESSAGE, "");
}

static void ask_save_time(KEYCODE keycode, Index arg)
{
    sprintf(popuptext, "%i", save_minute);
    construct_string("Minutes between automatic save: ",
		     popuptext, filter_savetime);
}

static void menu_insert_string(KEYCODE keycode, Index arg)
{
    if (wmselection)
	insert_string(wmselection);
}

static int argnr=0,bi=0,fi=0, cpos=-1;
static int bp[20] = {0};
static int bpd = 0;
static char *args[10] = {NULL};
static char buffer[5000];
static char *format=NULL;

static void use_argument(char *name)
{
    int i=0;
    if (argnr) {
	args[argnr] = concat(name,"");
	if (args[argnr][0]) {
	    while (name[i])
		buffer[bi++] = name[i++];
	} else if (bpd && bp[bpd-1]>=0) {
	    Bool stop=False;
	    bi=bp[bpd-1];
	    if (cpos>bi) cpos = bi;
	    i = bpd;
	    while (!stop) {
		while (format[fi] && format[fi]!='%') fi++;
		fi++;
		if (format[fi]==']') bpd--;
		if (format[fi]=='[') bpd++;
		stop = (!format[fi] || bpd <i);
		if (!stop && format[fi]>'0' && format[fi]<='9') {
		    i= format[fi]-'0';
		    if (!args[i]) args[i]= concat("","");
		}
	    }
	    if (format[fi]) fi++;
	}
    }
    while (format[fi]) {
	if (format[fi]=='%') {
	    fi++;
	    switch (format[fi]) {
	    case 'c': cpos = bi; break;
	    case '[': bp[bpd++] = bi; break;
	    case ']': bpd--;          break;
	    case 'n': buffer[bi++]='\n'; break;
	    case 't': buffer[bi++]='\t'; break;
	    case '%': buffer[bi++]='%';  break;
	    case '1': case '2': case '3': case '4': case '5':
	    case '6': case '7': case '8': case '9':
		i = format[fi]-'0';
		if (args[i]) {
		    int j;
		    if (args[i][0]) {
			for (j=0; args[i][j]; j++,bi++)
			    buffer[bi]=args[i][j];
		    } else if (bpd && bp[bpd-1]>=0) {
			Bool stop=False;
			bi=bp[bpd-1];
			if (cpos>bi) cpos = bi;
			j = bpd;
			while (!stop) {
			    while (format[fi] && format[fi]!='%') fi++;
			    fi++;
			    if (format[fi]==']') bpd--;
			    if (format[fi]=='[') bpd++;
			    stop = (!format[fi] ||  bpd<j);
			    if (!stop && format[fi]>'0' && format[fi]<='9') {
				j= format[fi]-'0';
				if (!args[j]) args[j]= concat("","");
			    }
			}
			if (!format[fi]) fi--; 
		    }
		} else {
		    argnr = i;
		    fi++;
		    construct_string("Argument: ", "", use_argument);
		    return;
		}
		break;
	    case '\0': fi--;
	    default: return;
	    }
	    fi++;
	} else
	    buffer[bi++] = format[fi++];
    }
    buffer[bi]='\0';
    insert_string(buffer);
    if (cpos>=0) {
	backward_char(0, 1);
	forward_char(0,cpos);
    }
}


static void menu_filter_string(KEYCODE keycode, Index arg)
{
    int i;
    for (i=0; i<10; i++)
	if (args[i]) {
	    myfree(args[i]);
	    args[i]= NULL;
	}
    format = get_string_arg(arg);
    buffer[0]='\0';
    bi = fi = 0;
    cpos = -1;
    bpd=0;
    argnr=0;
    if (!format) return;
    use_argument("");
}

static void start_print(KEYCODE keycode, Index arg)
{
    use_map(0, print_map);
    print_key(keycode, 1);
}

static void switch_textdots(KEYCODE keycode, Index arg)
{
    int i;
    void *data;

    textdots= !textdots;
    clear_tab_positions();
    i=0;
    while ((data = next_data_with_type(MAINEDITWINDOW, &i))) {
	i++;
	(*(eventfunc[EDITWINDOW]->draw))(data);
    }
    if (notadef_is_open && !notadef_iconized)
	(*(eventfunc[NOTATIONDEFWINDOW]->draw))(NULL);
    if (buffer_is_open && !buffer_iconized)
	(*(eventfunc[BUFFERWINDOW]->draw))(NULL);
    if (find_is_open && !find_iconized)
	(*(eventfunc[FINDWINDOW]->layout_change))(NULL);
}

typedef struct {
    void (*keyfunc)(KEYCODE,Index);
    char *namefunc;
    KEYCODE key;
} FUNCTIONTYPE;

static FUNCTIONTYPE keysfuncs[] = 
{
    { switch_textdots, "switch-textdots" },
    { menu_filter_string, "insert-string" },
    { use_map_string, "use-map" },
    { insert_expr, "insert-expression" },
    { insert_disp, "insert-display" },
    { insert_var, "insert-variable" },
    { insert_id, "insert-identifier" },
    { insert_op, "insert-operator" },
    { insert_text, "insert-text" },
    { raise_node, "ungroup-region" },
    { lower_region, "group-region" },
    { rename_id, "rename-identifier" },
    { commute, "commute-expression" },
    { factorise, "factorise-function" },
    { distribute, "distribute-function" },
    { apply, "apply-function" },
    { clear_parens, "remove-parens" },
    { set_parens, "add-parens" },
    { copy_region, "copy-region" },
    { swap_region, "swap-region" },
    { latex_select, "LaTeX-region" },
    { up, "expression-up" },
    { down, "expression-down" },
    { recenter, "recenter" },
    { move_to_center, "move-to-center" },
    { scroll_up, "scroll-up" },
    { scroll_down, "scroll-down" },
    { transpose_chars, "transpose-chars" },
    { transpose_words, "transpose-words" },
    { ask_latex_line, "goto-latex-line" },
    { begin_of_line, "beginning-of-line" },
    { end_of_line, "end-of-line" },
    { backward_line, "previous-line" },
    { forward_line, "next-line" },
    { backward_char, "backward-char" },
    { forward_char, "forward-char" },
    { begin_of_buffer, "beginning-of-buffer" },
    { end_of_buffer, "end-of-buffer" },
    { forward_word, "forward-word" },
    { backward_word, "backward-word" },
    { forward_remove_char, "delete-char" },
    { backward_remove_char, "backward-delete-char" },
    { remove_double_chars, "just-one-space" },
    { display_left, "display-left" },
    { display_right, "display-right" },
    { kill_line, "kill-line" },
    { make_project, "make-project" },
    { backward_kill_line, "backward-kill-line" },
    { kill_word, "kill-word" },
    { backward_kill_word, "backward-kill-word" },
    { kill_region, "kill-region" },
    { yank, "yank" },
    { append_next_kill, "append-next-kill" },
    { insert_newline, "soft-newline" },
    { insert_rtab, "insert-rtab" },
    { insert_hard_newline, "newline" },
    { construct_argument, "universal-argument" },
    { increase_spacing, "increase-spacing" },
    { decrease_spacing, "decrease-spacing" },
    { set_index_nr, "set-index-nr" },
    { reset_spacing, "reset-spacing" },
    { menu_notation_shortcut, "insert-template" },
    { ask_selection, "insert-selection" },
    { menu_insert_string, NULL, WMSELECT },
    { menu_keysymbol, "symbol-click" },
    { menu_notation, "template-click" },
    { ask_save_time, "set-save-time" },
    { start_replace, "query-replace" },
    { insert_char, "self-insert" },
    { next_node_or_insert, "next-node-or-insert" },
    { next_node_insert, "next-node-insert" },
    { make_list_insert, "insert-list-element" },
    { openparen_insert, "open-parenthesis" },
    { closeparen_insert, "close-parenthesis" },
    { insert_symbol, "insert-symbol" },
    { start_find, "isearch-forward" },
    { start_backward_find, "isearch-backward" },
    { start_print, "start-print" },
    { reset_map, "reset-map" },
    { print_key, "print-key" },
    { do_find, "search-next" },
    { do_find_backward, "search-previous" },
    { add_char, "search-self-insert" },
    { add_return, "search-newline" },
    { add_tab, "search-tab" },
    { add_sym, "search-symbol" },
    { add_search_notation, "search-template" },
    { remove_search_char, "search-remove-char" },
    { end_search_oldpos, "search-cancel" },
    { end_search, "search-stop" },
    { positive_yn, "answer-yes" },
    { negative_yn, "answer-no" },
    { end_search, "answer-stop" },
    { end_search_oldpos, "answer-cancel" },
    { replace_all_yn, "answer-all" },
    { NULL, NULL }
};

void menu_keyboard(void)
{
    int i,j;
    key_map = new_map("Global");
    print_map = new_map("print");
    search_map = new_map("search");
    y_n_map = new_map("answer");
    for (j=0; keysfuncs[j].keyfunc!=NULL; j++) {
	i = add_func(keysfuncs[j].keyfunc, keysfuncs[j].namefunc);
	if (keysfuncs[j].key==WMSELECT)
	    add_key(key_map, WMSELECT, i, 1);
    }
    use_map(0, key_map);
}
