/*
 * misc.c
 *
 *     -- miscellaneous utility functions
 *
 * Copyright  1999 Anselm Lingnau <lingnau@tm.informatik.uni-frankfurt.de>
 * See file COPYING for conditions on use and distribution.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "dvi.h"
#include "dviInt.h"

#ifndef lint
static char rcsid[] VAR_UNUSED = "$Id: misc.c,v 1.1.1.1 1999/06/10 12:36:50 lingnau Exp $";
#endif /* lint */


/*
 * ------------------------------------------------------------------------
 *
 * DviSaveStr --
 *
 *     Given a string, save a copy of the string in dynamic memory.
 *
 * Results:
 *     A pointer to the saved copy of the string. If there isn't enough
 *     dynamic memory left, the program is aborted.
 *
 * ------------------------------------------------------------------------
 */

char *
DviSaveStr (string)
    const char *string;
{
    char *copy = ckalloc(strlen(string) + 1);

    if (copy == 0) {
	fprintf(stderr, "Not enough memory\n");
	exit(1);
    }
    return strcpy(copy, string);
}


/*
 * ------------------------------------------------------------------------
 *
 * DviSaveStrN --
 *
 *     Given a string, save a copy of the string in dynamic memory
 *     up to a given length.
 *
 * Results:
 *     A pointer to the saved copy of the string. If there isn't enough
 *     dynamic memory left, the program is aborted.
 *
 * ------------------------------------------------------------------------
 */

char *
DviSaveStrN (string, length)
    const char *string;
    const size_t length;
{
    char *copy = ckalloc(length + 1);

    if (copy == 0) {
	fprintf(stderr, "Not enough memory\n");
	exit(1);
    }
    strncpy(copy, string, length);
    copy[length] = '\0';
    return copy;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_GetPixels --
 *
 *     Calculate the number of pixels for a given dimension (scale factor
 *     and unit) at a given resolution.
 *
 * Results:
 *     A standard Tcl result. If the result is TCL_OK, the number of
 *     pixels is in the variable pointed to by `result'. If the result
 *     is TCL_ERROR, an error has occurred. If TCL_LEAVE_ERR_MSG is set
 *     in `flags', a more detailed error message is put into the Tcl
 *     result string.
 *
 * ------------------------------------------------------------------------
 */

static struct units {
    char *name;
    double conv;
} units[] = {
    { "px", 1.0 },		/* This must be the first entry */
    { "in", 1.0 },
    { "cm", 2.54 },
    { "mm", 25.4 },
    { "pt", 72.27 },
    { "bp", 72 },
    { "pc", 72.27 / 12 },
    { "dd", 72.27 * 1157.0 / 1238 },
    { "cc", 72.27 * 1157.0 / 1238 / 12 },
    { "sp", 72.27 * 65536 },
    { (char *)0, 0 }
};

int
Dvi_GetPixels (interp, resolution, string, result, flags)
    Tcl_Interp *interp;
    const int resolution;
    const char *string;
    int *result;
    const int flags;
{
    double scale = 0.0;
    char *unit;
    struct units *u;

    *result = 0;
    if ((scale = strtod(string, &unit)) == HUGE_VAL || scale < 0) {
	if (flags & TCL_LEAVE_ERR_MSG) {
	    Tcl_SetResult(interp, "scale factor out of range", TCL_STATIC);
	}
	return TCL_ERROR;
    }
    
    if (unit == string) {
	if (flags & TCL_LEAVE_ERR_MSG) {
	    Tcl_SetResult(interp, "invalid scale factor", TCL_STATIC);
	}
	return TCL_ERROR;
    }

    if (*unit == '\0') {
	*result = (int)ceil(scale);
	return TCL_OK;
    }

    units[0].conv = resolution;
    for (u = units; u->name && strcmp(unit, u->name) != 0; u++)
	;
    if (u->name) {
	*result = (int)ceil(scale / u->conv * resolution);
	return TCL_OK;
    }
    if (flags & TCL_LEAVE_ERR_MSG) {
	Tcl_SetResult(interp, "unknown unit", TCL_STATIC);
    }
    return TCL_ERROR;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_GetDistance --
 *
 *     Calculate a distance (in a specified unit) from a number of pixels
 *     at a given resolution.
 *
 * Results:
 *     A standard Tcl result. If the result is TCL_OK, the
 *     distance is left in the variable pointed to by `result',
 *     If the result is TCL_ERROR, an error has occurred. In this case,
 *     if TCL_LEAVE_ERR_MSG is set in `flags', a more detailed error
 *     message is put into the Tcl result string.
 *
 * ------------------------------------------------------------------------
 */

int
Dvi_GetDistance (interp, resolution, pixels, unit, result, flags)
    Tcl_Interp *interp;
    const int resolution;
    const double pixels;
    const char *unit;
    double *result;
    const int flags;
{
    struct units *u;

    units[0].conv = resolution;
    for (u = units; u->name && strcmp(unit, u->name) != 0; u++)
	;
    if (u->name == 0) {
	if (flags & TCL_LEAVE_ERR_MSG) {
	    Tcl_SetResult(interp, "unknown unit", TCL_STATIC);
	}
	return TCL_ERROR;
    }

    *result = u->conv * pixels / resolution;
    return TCL_OK;
}
