/*
  This is the file ttf2pk.c of the CJK macro package ver. 4.1.3
  (20-Jun-1997).
*/


/*
  TTF2PK adapted from ttf2bmp and pbmtopk by Yaw-Jen Lin
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "ttf.h"
#include "coding.h"


#define MAXPKCHAR 256
#define MAXOPTLINE 200
#define MAXWIDTHTAB 256
#define MAXHEIGHTTAB 16
#define MAXDEPTHTAB 16
#define MAXITALICTAB 64
#define MAXPARAMS 30
#define NAMELENGTH 80

#define DESIGNSIZE 10.0

/* we must explicitly set these values for rotated glyphs */
#define ASCENDER_ROTATED 0.75
#define DESCENDER_ROTATED 0.25

#define true (1)
#define false (0)
#define chr(a) (a)

#define fixword(d) ((int)((double)(d) * 1048576))
#define unfixword(f) ((double)(f) / 1048576)
#define fixrange(f) ((f) < 16777216 && (f) > -16777216)
#define designunits(p) \
          ((p) * 72.27 / (double)x_resolution / unfixword(designsize))

/* character flags: in order of appearance in option files. */
#define XOFFSET     1
#define YOFFSET     2
#define HORZESC     4
#define VERTESC     8
#define TFMWIDTH   16
#define TFMHEIGHT  32
#define TFMDEPTH   64
#define TFMITALIC 128

typedef int integer;
typedef char quarterword;
typedef char boolean;
typedef quarterword ASCIIcode;
typedef quarterword eightbits;
typedef FILE *bytefile;
typedef unsigned char byte;

unsigned char mask_pixel[8] = {0x80, 0x40, 0x20, 0x10, 8, 4, 2, 1};
unsigned char mask_from[8] = {0xff, 0x7f, 0x3f, 0x1f, 0xf, 7, 3, 1};
unsigned char mask_to[8] = {0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff};
unsigned char b[2];
double point; /* added by y. R. Li */

integer x_resolution, y_resolution, designsize;
char *filename[MAXPKCHAR];

integer xoffset[MAXPKCHAR];
integer yoffset[MAXPKCHAR];
integer horzesc[MAXPKCHAR];
integer vertesc[MAXPKCHAR];

byte tfmindex[MAXPKCHAR];
byte hgtindex[MAXPKCHAR];
byte depindex[MAXPKCHAR];
byte italindex[MAXPKCHAR];
byte charflags[MAXPKCHAR];

TTF *fn;
BITMAP *bm;
unsigned int schar, nchar;
/* added by Y. R. Li */
int enc;
boolean translate_to_SJIS;
integer rotation;
integer smallestch, largestch;
integer emwidth;
integer checksum;
char *codingscheme = "CJK";
char *familyname = "TTF2PK";

integer widthtab[MAXWIDTHTAB]; /* TFM widths */
integer numwidth; /* number of entries in width table */
integer heighttab[MAXHEIGHTTAB];
integer numheight;
integer depthtab[MAXDEPTHTAB];
integer numdepth;
integer italictab[MAXITALICTAB];
integer numitalic;
integer parameters[MAXPARAMS];
integer numparam;

ASCIIcode xord[128];
char xchr[256];
bytefile tfmfile, pkfile;
char tfmname[NAMELENGTH + 1], pkname[NAMELENGTH + 1];
integer pkloc;
integer bitweight;
integer outputbyte;
integer car;
integer hppp, vppp;
integer width;
integer bwidth;
integer height;
integer power[32];


void main(int, char *[]);
static void jumpout(void);
static byte add_tfmtable(integer *, integer *, integer, integer, char *);
static void add_suffix(char *, char *);
static void initialize(void);
static void writepreamble(void);
static void writepostamble(void);
static void writetfmfile(void);
static void rotate_BMP(void);
static void BMP_shipcharacter(void);

static void pkbyte(integer);
static void pkhalfword(integer);
static void pkthreebytes(integer);
static void pkword(integer);
static void pknyb(integer);

static void tfmbyte(integer);
static void tfmhalfword(integer);
static void tfmword(integer);

static int equal(BYTE *, BYTE *);
static void usage(void);
static void dialog(int, char **);


/*@*/

#if 0
integer compute_checksum()
{
  /*
    begin
      c0:=bc; c1:=ec; c2:=bc; c3:=ec;
      for c:=bc to ec do if char_wd[c]>0 then
        begin
          temp_width:=memory[char_wd[c]];
          if design_units<>unity then
            temp_width:=round((temp_width/design_units)*1048576.0);
          temp_width:=temp_width + (c+4)*@'20000000; {this should be positive}
          c0:=(c0+c0+temp_width) mod 255;
          c1:=(c1+c1+temp_width) mod 253;
          c2:=(c2+c2+temp_width) mod 251;
          c3:=(c3+c3+temp_width) mod 247;
        end;
      header_bytes[check_sum_loc]:=c0;
      header_bytes[check_sum_loc+1]:=c1;
      header_bytes[check_sum_loc+2]:=c2;
      header_bytes[check_sum_loc+3]:=c3;
    end
  */ 

  return 0;
}
#endif


/*@*/

#define add_tfmwidth(v) \
          (add_tfmtable(widthtab, &numwidth, v, MAXWIDTHTAB, "TFM width"))
#define add_tfmheight(v) \
          (add_tfmtable(heighttab, &numheight, v, MAXHEIGHTTAB, "TFM height"))
#define add_tfmdepth(v) \
          (add_tfmtable(depthtab, &numdepth, v, MAXDEPTHTAB, "TFM depth"))
#define add_tfmitalic(v) \
         (add_tfmtable(italictab, &numitalic, v, MAXITALICTAB, \
          "Italic correction"))


/*@@*/
          
static byte add_tfmtable(table, count, value, maxv, name)
  integer *table, *count, value, maxv;
  char *name;
{
  integer i;
  
  for (i = 0; i < *count; i++) /* search for value in tfm table */
    if (table[i] == value)
      return (byte)i;
  if (*count >= maxv)
  {
    fprintf(stderr, "Too many values in %s table!\n", name);
    jumpout();
  }
  if (!fixrange(value))
  {
    fprintf(stderr, " %s %f for char %d out of range!\n",
            name, unfixword(value), car);
    jumpout();
  }
  table[*count] = value;
  return (*count)++;
}


static void add_suffix(name, suffix)
  char *name, *suffix;
{
  int haveext = 0;
  if (name && strcmp(name, "-"))
  {
    while (*name)
    {
      if (*name == '/')
        haveext = 0;
      else if (*name == '.')
        haveext = 1;
      name++;
    }
    if (!haveext)
    {
      *name++ = '.';
      strcpy(name,suffix) ;
    }
  }
}


static void initialize()
{
  integer i;
  
  fprintf(stderr, "This is TTFtoPK, CJK Version 4.1.3\n");
  for (i = 0; i <= 31; i++)
    xchr[i] = '?';
    
  xchr[32] = ' ';
  xchr[33] = '!';
  xchr[34] = '"';
  xchr[35] = '#';
  xchr[36] = '$';
  xchr[37] = '%';
  xchr[38] = '&';
  xchr[39] = '\'';
  xchr[40] = '(';
  xchr[41] = ')';
  xchr[42] = '*';
  xchr[43] = '+';
  xchr[44] = ',';
  xchr[45] = '-';
  xchr[46] = '.';
  xchr[47] = '/';
  xchr[48] = '0';
  xchr[49] = '1';
  xchr[50] = '2';
  xchr[51] = '3';
  xchr[52] = '4';
  xchr[53] = '5';
  xchr[54] = '6';
  xchr[55] = '7';
  xchr[56] = '8';
  xchr[57] = '9';
  xchr[58] = ':';
  xchr[59] = ';';
  xchr[60] = '<';
  xchr[61] = '=';
  xchr[62] = '>';
  xchr[63] = '?';
  xchr[64] = '@';
  xchr[65] = 'A';
  xchr[66] = 'B';
  xchr[67] = 'C';
  xchr[68] = 'D';
  xchr[69] = 'E';
  xchr[70] = 'F';
  xchr[71] = 'G';
  xchr[72] = 'H';
  xchr[73] = 'I';
  xchr[74] = 'J';
  xchr[75] = 'K';
  xchr[76] = 'L';
  xchr[77] = 'M';
  xchr[78] = 'N';
  xchr[79] = 'O';
  xchr[80] = 'P';
  xchr[81] = 'Q';
  xchr[82] = 'R';
  xchr[83] = 'S';
  xchr[84] = 'T';
  xchr[85] = 'U';
  xchr[86] = 'V';
  xchr[87] = 'W';
  xchr[88] = 'X';
  xchr[89] = 'Y';
  xchr[90] = 'Z';
  xchr[91] = '[';
  xchr[92] = '\\';
  xchr[93] = ']';
  xchr[94] = '^';
  xchr[95] = '_';
  xchr[96] = '`';
  xchr[97] = 'a';
  xchr[98] = 'b';
  xchr[99] = 'c';
  xchr[100] = 'd';
  xchr[101] = 'e';
  xchr[102] = 'f';
  xchr[103] = 'g';
  xchr[104] = 'h';
  xchr[105] = 'i';
  xchr[106] = 'j';
  xchr[107] = 'k';
  xchr[108] = 'l';
  xchr[109] = 'm';
  xchr[110] = 'n';
  xchr[111] = 'o';
  xchr[112] = 'p';
  xchr[113] = 'q';
  xchr[114] = 'r';
  xchr[115] = 's';
  xchr[116] = 't';
  xchr[117] = 'u';
  xchr[118] = 'v';
  xchr[119] = 'w';
  xchr[120] = 'x';
  xchr[121] = 'y';
  xchr[122] = 'z';
  xchr[123] = '{';
  xchr[124] = '|';
  xchr[125] = '}';
  xchr[126] = '~';
  for (i = 127; i <= 255; i ++)
    xchr[i] = '?';
  for (i = 0; i <= 127; i ++)
    xord[chr(i)] = 32;
  for (i = 32; i <= 126; i ++)
    xord[(integer)(xchr[i])] = i;
  for (i = 0; i < 32; i++)
    power[i] = 1 << i;
  for (i = 0; i < MAXPKCHAR; i++)
  {
#if 0
    filename[i] = NULL;
#endif
    charflags[i] = 0;
  }
  pkloc = 0;
  designsize = fixword(DESIGNSIZE);
  point = DESIGNSIZE;
  numparam = 0;
  widthtab[0] = heighttab[0] = depthtab[0] = italictab[0] = 0;
  numwidth = numheight = numdepth = numitalic = 1;
  smallestch = MAXPKCHAR;
  largestch = -1;
  emwidth = 0;
  enc = BIG5; /* default encoding for backward compatibility */
  translate_to_SJIS = 0;
  rotation = 0;
}


static void jumpout()
{
  exit(1);
}


static void pkbyte(b)
  integer b;
{
  if (b < 0)
    b = b + 256;
  putc(b, pkfile);
  pkloc++;
}


static void pkhalfword(a)
  integer a;
{
  if (a < 0)
    a = a + 65536;
  pkbyte(a >> 8);
  pkbyte(a & 255);
}


static void pkthreebytes(a)
  integer a;
{
  pkbyte((a >> 16) & 255);
  pkbyte((a >> 8) & 255);
  pkbyte(a & 255);
}


static void pkword(a)
  integer a;
{
  pkbyte((a >> 24) & 255);
  pkbyte((a >> 16) & 255);
  pkbyte((a >> 8) & 255);
  pkbyte(a & 255);
}


static void pknyb(a)
  integer a;
{
  if (bitweight == 16)
  {
    outputbyte = (a << 4);
    bitweight = 1;
  } 
  else
  {
    pkbyte(outputbyte + a);
    bitweight = 16;
  }
}


static void writepreamble()
{
  integer i;
  char *comment = "ttf2pk CJK version 4.1.3 output";
  
  pkbyte(247);
  pkbyte(89);
  pkbyte(strlen(comment));
  for (i = 0; i < strlen(comment); i++)
    pkbyte(xord[(integer)(comment[i])]);
  pkword(designsize);
  pkword(checksum); /* checksum; calculate if possible */
  pkword(hppp);
  pkword(vppp);
}


static void writepostamble()
{
  pkbyte(245);
  while ((pkloc % 4 != 0))
    pkbyte(246);
  fprintf(stderr, "%d bytes written to packed file.\n", pkloc);
}


static void tfmbyte(b)
  integer b;
{
  if (b < 0)
    b = b + 256;
  putc(b, tfmfile);
}


static void tfmhalfword(a)
  integer a;
{
  if (a < 0)
    a = a + 65536;
  tfmbyte(a >> 8);
  tfmbyte(a & 255);
}


static void tfmword(a)
  integer a;
{
  tfmbyte((a >> 24) & 255);
  tfmbyte((a >> 16) & 255);
  tfmbyte((a >> 8) & 255);
  tfmbyte(a & 255);
}


static void writetfmfile()
{
  integer totallength;
  integer headersize = 17;
  integer i;

  if ((largestch - smallestch) < 0)
  {
    largestch = 0;
    smallestch = 1;
  }
  if (numparam < 7) /* set default parameters */
    switch (numparam)
    {
    case 0: /* slant */
      parameters[numparam++] = 0;
    case 1: /* space */
      parameters[numparam++] = 0;
    case 2: /* space_stretch */
      parameters[numparam++] = 0;
    case 3: /* space_shrink */
      parameters[numparam++] = 0;
    case 4: /* x_height */
      parameters[numparam++] = fixword(1);
    case 5: /* quad */
      parameters[numparam++] = fixword(1);
    case 6: /* extra_space */
      parameters[numparam++] = 0;
    }
  totallength = 6 + headersize + (largestch + 1 - smallestch) +
    numwidth + numheight + numdepth + numitalic + numparam;
  /* lengths */
  tfmhalfword(totallength);
  tfmhalfword(headersize);
  tfmhalfword(smallestch);
  tfmhalfword(largestch);
  tfmhalfword(numwidth);
  tfmhalfword(numheight);
  tfmhalfword(numdepth);
  tfmhalfword(numitalic);
  tfmhalfword(0); /* lig/kern table */
  tfmhalfword(0); /* kern table */
  tfmhalfword(0); /* extensible char table */
  tfmhalfword(numparam);
  /* header */
  tfmword(checksum);
  tfmword(designsize);
  if (strlen(codingscheme) > 39)
    tfmbyte(39);
  else
    tfmbyte(strlen(codingscheme));
  for (i = 0; i < 39; i++)
    if (*codingscheme)
      tfmbyte(xord[(integer)(*codingscheme++)]);
    else
      tfmbyte(0);
  if (strlen(familyname) > 19)
    tfmbyte(19);
  else
    tfmbyte(strlen(familyname));
  for (i = 0; i < 19; i++)
    if (*familyname)
      tfmbyte(xord[(integer)(*familyname++)]);
    else
      tfmbyte(0);
  /* char_info */
  for (car = smallestch; car <= largestch; car++)
    if (filename[car])
    {
      tfmbyte(tfmindex[car]);
      tfmbyte((hgtindex[car]<<4) + depindex[car]);
      tfmbyte(italindex[car]<<2);
      tfmbyte(0);
    }
    else
      tfmword(0);
  /* width */
  for (i = 0; i < numwidth; i++)
    tfmword(widthtab[i]);
  /* height */
  for (i = 0; i < numheight; i++)
    tfmword(heighttab[i]);
  /* depth */
  for (i = 0; i < numdepth; i++)
    tfmword(depthtab[i]);
  /* italic */
  for (i = 0; i < numitalic; i++)
    tfmword(italictab[i]);
  /* no lig_kern, kern, or exten */
  /* param */
  for (i = 0; i < numparam; i++)
    if (i && (!fixrange(parameters[i])))
    {
      fprintf(stderr, " Parameter %d out of range (-p)!\n", i);
      jumpout();
    }
    else
      tfmword(parameters[i]) ;
  fprintf(stderr, "%d bytes written to tfm file.\n", totallength * 4);
}


static int equal(row1, row2)
  BYTE *row1, *row2;
{
  integer i;
  
  for (i = 0; i < bwidth; i++)
    if (row1[i] != row2[i])
      return 0;
  return 1;
}


static void rotate_BMP()
{
  char *bitmap, *source, *bm_end, *pos;
  int y;
  int imask, omask;
  int c;
  int size, bheight;

  source = bm->map;
  bheight = (height + 7) / 8;
  size = bheight * width;
  bitmap = TTF_ALLOC(char, size, "rotate_BMP");
  bm_end = bitmap + bheight;
  memset(bitmap, 0, size);

  for (y = 0; y < height; y++)
  {
    pos = bitmap + size - bheight + y / 8;
    omask = 1 << (7 - y % 8);
    /* y = height - 1 */
    for (;;)
    {
      c = *source++;
      for (imask = 0x80; imask != 0; imask >>= 1)
      {
        if ((c & imask) != 0)
          *pos |= omask;
        if (pos < bm_end)
          goto end_column;
        pos -= bheight;
      }
    }
end_column:
    ;
  }

  TTF_FREE(bm->map, "rotate_BMP");
  bm->map = bitmap;

  bm->xpoint = height;
  bm->ysize = width;
  bm->xsize = bheight;
  width = bm->xpoint;
  height = bm->ysize;
  bwidth = bm->xsize;
}


static void BMP_shipcharacter()
{
  integer compsize;
  integer i, j, k;
  BYTE *zerorow, *onesrow;
  integer *repeatptr, *bitcounts;
  integer count;
  integer test;
  integer curptr, rowptr;
  integer bitval;
  integer repeatflag;
  integer colptr;
  integer currepeat;
  integer dynf;
  integer deriv[14];
  integer bcompsize;
  boolean firston;
  integer flagbyte;
  boolean state;
  boolean on;
  integer hbit;
  integer pbit;
  boolean ron = 0, son = 0;
  integer rcount = 0, scount = 0;
  integer ri = 0, si = 0;
  integer max2;
  integer predpkloc;
  integer buff;

  integer tfwid = widthtab[tfmindex[car]];
  integer hesc = horzesc[car];
  integer vesc = vertesc[car];
  integer xoff = xoffset[car];
  integer yoff = yoffset[car];

  zerorow = TTF_ALLOC(BYTE, bwidth, "BMP_shipcharacter");
  onesrow = TTF_ALLOC(BYTE, bwidth, "BMP_shipcharacter");
  repeatptr =
    TTF_ALLOC(integer, height + 1, "BMP_shipcharacter");
  bitcounts =
    TTF_ALLOC(integer, height * width, "BMP_shipcharacter");
  for (i = 0; i < bwidth - 1; i++)
  {
    zerorow[i] = 0x00;
    onesrow[i] = 0xff;
  }
  zerorow[bwidth - 1] = 0x00;
  onesrow[bwidth - 1] = mask_to[(width - 1) & 7];
  for (i = 0; i < height; i++)
  {
    if (equal(&((bm->map)[i * bwidth]), zerorow))
      repeatptr[i] = 0;
    else if (equal(&((bm->map)[i * bwidth]), onesrow)) 
      repeatptr[i] = 0;
    else if (i + 1 < height &&
             equal(&((bm->map)[i * bwidth]), &((bm->map)[(i + 1) * bwidth])))
      repeatptr[i] = 1;
    else
      repeatptr[i] = 0;
  }
  i = 0;
  while (i < height)
  {
    k = i;
    while (repeatptr[k] == 1)
      k++;
    repeatptr[i] = k - i;
    i = k + 1;
  }
  repeatptr[i] = 0;
  colptr = width - 1;
  repeatflag = currepeat = curptr = count = rowptr = 0;
  test = 0;
  do
  {
    colptr++;
    if (colptr == width)
    {
      colptr = 0;
      rowptr = currepeat;
      if (repeatptr[currepeat] > 0)
      {
        repeatflag = repeatptr[currepeat];
        currepeat += repeatflag;
        rowptr += repeatflag;
      }
      currepeat++;
    }
    if (rowptr >= height)
      bitval = -1;
    else
      bitval = ((bm->map)[rowptr * bwidth + colptr / 8] &
               mask_pixel[colptr & 7]) != 0;
    if (bitval == test)
      count++;
    else
    {
      bitcounts[curptr++] = count;
      if (curptr + 3 >= height * width)
      {
        fprintf(stderr, " Out of memory while saving character counts!\n");
        jumpout();
      }
      count = 1;
      test = bitval;
      if (repeatflag > 0)
      {
        bitcounts[curptr++] = -repeatflag;
        repeatflag = 0;
      }
    }
  } while (test != -1);
  bitcounts[curptr] = 0;
  bitcounts[curptr + 1] = 0;
  for (i = 1; i <= 13; i ++)
    deriv[i] = 0;
  i = firston = (bitcounts[0] == 0);
  compsize = 0;
  while (bitcounts[i] != 0)
  {
    j = bitcounts[i];
    if (j == -1)
      compsize++;
    else 
    {
      if (j < 0) 
      {
        compsize++;
        j = -j;
      }
      if (j < 209)
        compsize += 2;
      else
      {
        k = j - 193;
        while (k >= 16)
        {
          k >>= 4;
          compsize += 2;
        }
        compsize++;
      }
      if (j < 14)
        (deriv[j])--;
      else if (j < 209)
        (deriv[(223 - j) / 15])++;
      else
      {
        k = 16;
        while (((k << 4) < j + 3))
          k <<= 4;
        if (j - k <= 192)
          deriv[(207 - j + k) / 15] += 2;
      }
    }
    i++;
  }
  bcompsize = compsize;
  dynf = 0;
  for (i = 1; i <= 13; i ++)
  {
    compsize += deriv[i];
    if (compsize <= bcompsize)
    {
      bcompsize = compsize;
      dynf = i;
    }
  }
  compsize = ((bcompsize + 1) >> 1);
  if ((compsize > ((height * width + 7) >> 3)) || (height * width == 0))
  {
    compsize = ((height * width + 7) >> 3);
    dynf = 14;
  }
  flagbyte = (dynf << 4);
  if (firston)
    flagbyte |= 8;
  if ((tfwid > 16777215) || (tfwid < 0) || (hesc < 0) || (vesc != 0) ||
      (compsize > 196579) || (width > 65535) || (height > 65535) ||
      (xoff > 32767) || (yoff > 32767) || (xoff < -32768) || (yoff < -32768))
  {
    flagbyte |= 7;
    pkbyte(flagbyte);
    compsize += 28;
    pkword(compsize);
    pkword(car);
    predpkloc = pkloc + compsize;
    pkword(tfwid);
    pkword(hesc << 16);
    pkword(vesc << 16);
    pkword(width);
    pkword(height);
    pkword(xoff);
    pkword(yoff);
  }
  else if ((hesc > 255) || (width > 255) || (height > 255) ||
           (xoff > 127) || (yoff > 127) || (xoff < -128) ||
           (yoff < -128) || (compsize > 1015))
  {
    compsize += 13;
    flagbyte += (compsize >> 16) + 4;
    pkbyte(flagbyte);
    pkhalfword(compsize & 65535);
    pkbyte(car);
    predpkloc = pkloc + compsize;
    pkthreebytes(tfwid);
    pkhalfword(hesc);
    pkhalfword(width);
    pkhalfword(height);
    pkhalfword(xoff);
    pkhalfword(yoff);
  }
  else
  {
    compsize += 8;
    flagbyte = flagbyte + (compsize >> 8);
    pkbyte(flagbyte);
    pkbyte(compsize & 255);
    pkbyte(car);
    predpkloc = pkloc + compsize;
    pkthreebytes(tfwid);
    pkbyte(hesc);
    pkbyte(width);
    pkbyte(height);
    pkbyte(xoff);
    pkbyte(yoff);
  }
  if (dynf != 14)
  {
    bitweight = 16;
    max2 = 208 - 15 * dynf;
    i = firston;
    while (bitcounts[i] != 0)
    {
      j = bitcounts[i];
      if (j == - 1)
        pknyb(15);
      else
      {
        if (j < 0)
        {
          pknyb(14);
          j = -j;
        }
        if (j <= dynf)
          pknyb(j);
        else if (j <= max2)
        {
          j -= dynf + 1;
          pknyb((j >> 4) + dynf + 1);
          pknyb((j & 15));
        }
        else
        {
          j -= max2 - 15;
          k = 16;
          while (k <= j)
          {
            k <<= 4;
            pknyb(0);
          }
          while (k > 1)
          {
            k >>= 4;
            pknyb(j / k);
            j = j % k;
          }
        }
      }
      i++;
    }
    if (bitweight != 16)
      pkbyte(outputbyte);
  }
  else
  {
    buff = 0;
    pbit = 8;
    i = firston;
    hbit = width;
    on = ! firston;
    state = false;
    count = repeatflag = 0;
    while ((bitcounts[i] != 0) || state || (count > 0))
    {
      if (state)
      {
        count = rcount;
        i = ri;
        on = ron;
        repeatflag--;
      }
      else
      {
        rcount = count;
        ri = i;
        ron = on;
      }
      do
      {
        if (count == 0)
        {
          if (bitcounts[i] < 0)
          {
            if (! state)
              repeatflag = -bitcounts[i];
            i++;
          }
          count = bitcounts[i];
          i++;
          on = !on;
        }
        if ((count >= pbit) && (pbit < hbit))
        {
          if (on)
            buff += power[pbit] - 1;
          pkbyte(buff);
          buff = 0;
          hbit -= pbit;
          count -= pbit;
          pbit = 8;
        }
        else if ((count < pbit) && (count < hbit))
        {
          if (on)
            buff += power[pbit] - power[pbit - count];
          pbit -=  count;
          hbit -= count;
          count = 0;
        }
        else
        {
          if (on)
            buff += power[pbit] - power[pbit - hbit];
          count -= hbit;
          pbit -= hbit;
          hbit = width;
          if (pbit == 0)
          {
            pkbyte(buff);
            buff = 0;
            pbit = 8;
          }
        }
      } while (hbit != width);
      if (state && (repeatflag == 0))
      {
        count = scount;
        i = si;
        on = son;
        state = false;
      }
      else if (! state && (repeatflag > 0))
      {
        scount = count;
        si = i;
        son = on;
        state = true;
      }
    }
    if (pbit != 8)
      pkbyte(buff);
  }
  if (predpkloc != pkloc)
  {
    fprintf(stderr, " Bad predicted character length: character %d!\n", car);
    jumpout();
  }
  TTF_FREE(zerorow, "BMP_shipcharacter");
  TTF_FREE(onesrow, "BMP_shipcharacter");
  TTF_FREE(repeatptr, "BMP_shipcharacter");
  TTF_FREE(bitcounts, "BMP_shipcharacter");
}


static void usage()
{
  fprintf(stderr, " Usage: ttf2pk  pkfile[.pk] tfmfile[.tfm] dpi_x y_scale\n");
  fprintf(stderr, "                start_char nmb_of_char\n");
#ifdef DEBUG
  fprintf(stderr, "                [-d debuglevel]\n");
#endif
  fprintf(stderr, "                [-p nmbpar param1 param2 ...]\n");
  fprintf(stderr, "                [-C codingscheme ] [-F family]\n");
  fprintf(stderr, "                [-W tfmwidth] [-H tfmheight] [-D tfmdepth]\n");
  fprintf(stderr, "                [-I ital_corr] [-h horiz] [-v vert]\n");
  fprintf(stderr, "                [-x xoffset] [-y yoffset]\n");
  fprintf(stderr, "                [-r rotation] [-e encoding] ttf_file\n");
  jumpout();
}


/*@*/

#if 0
static void optionfile(name)
  char *name;
{
  FILE *fp;
  char buffer[MAXOPTLINE];
  
  if (!strcmp(name, "-"))
    fp = stdin;
  else if ((fp = fopen(name, "r")) == NULL)
  {
    fprintf(stderr, " Can't open option file %s!\n", name);
    jumpout();
  }
  while (!feof(fp))
  {
    char *here = buffer;

    if (fgets(buffer, MAXOPTLINE, fp) == NULL)
      break;
    while (isspace(*here))
      here++;
    if (*here && *here == '=')
    {
      if (sscanf(here + 1, "%d", &car) != 1)
      {
        fprintf(stderr, "Bad option file line %s!\n", buffer);
        jumpout();
      }
    }
    else if (*here && *here != '%' && *here != '#')
    {
      char str[NAMELENGTH];
      integer i, n;
      
      if (sscanf(here, "%s%n", str, &n) != 1)
      {
        fprintf(stderr, "Bad option file line %s!\n", buffer);
        jumpout();
      }
      filename[car] = TTF_ALLOC(char, strlen(str) + 1, "optionfile");
      strcpy(filename[car], str);
      for (i = 1; i < 256; i<<=1)
      {
        here += n;
        if (sscanf(here, "%s%n", str, &n) != 1)
          break;
        if (strcmp(str, "*"))
        {
          charflags[car] |= i;
          switch (i)
          {
          case XOFFSET:
            xoffset[car] = atoi(str);
            break;
          case YOFFSET:
            yoffset[car] = atoi(str);
            break;
          case HORZESC:
            horzesc[car] = atoi(str);
            break;
          case VERTESC:
            vertesc[car] = atoi(str);
            break;
          case TFMWIDTH:
            tfmindex[car] = add_tfmwidth(fixword(atof(str)));
            break;
          case TFMHEIGHT:
            hgtindex[car] = add_tfmheight(fixword(atof(str)));
            break;
          case TFMDEPTH:
            depindex[car] = add_tfmdepth(fixword(atof(str)));
            break;
          case TFMITALIC:
            italindex[car] = add_tfmitalic(fixword(atof(str)));
            break;
          }
        }
      }
      car++;
    }
  }
  if (fp != stdin)
    fclose(fp);
}
#endif


/*@@*/

static void dialog(gargc, gargv)
  int gargc;
  char **gargv;
{
  integer i, tfwid, tfdep = 0, tfhgt = 0, tfital = 0;
  double y_scale, hesc, vesc, xoff, yoff;
  double hppd, vppd;        /* pixels per designsize */
  byte flags;
  
  if (--gargc < 1)
    usage();
  strcpy(pkname, *++gargv);
  add_suffix(pkname, "pk");
  
  if (--gargc < 1)
    usage();
  strcpy(tfmname, *++gargv);
  add_suffix(tfmname, "tfm");
  
  if (--gargc < 1)
    usage();
  x_resolution = atoi(*++gargv);
  if (x_resolution < 1 || x_resolution > 32767)
  {
    fprintf(stderr, " Unlikely horizontal resolution %d dpi!\n", x_resolution);
    jumpout();
  }

  if (--gargc < 1)
    usage();
  y_scale = atof(*++gargv);
  y_resolution = x_resolution * y_scale;
  if (y_resolution < 1 || y_resolution > 32767)
  {
    fprintf(stderr, " Unlikely vertical scaling %f dpi!\n", y_scale);
    jumpout();
  }

  if (--gargc < 1)
    usage();
  schar = strtol(*++gargv, 0, 0) & 0xFFFF;
  if (--gargc < 1)
    usage();
  nchar = atoi(*++gargv);
  if (nchar > 256)
    nchar = 256;

  car = flags = hesc = vesc = xoff = yoff = tfwid = 0;
  while (++gargv, --gargc)
  {
    if (gargv[0][0] == '-' && gargv[0][1])
    {
      char c, *p;
      p = *gargv;
      c = gargv[0][1];
      if (gargv[0][2])
        p = *gargv + 2;
      else if (++gargv, --gargc)
        p = *gargv;
      else
        usage();
        
      switch (c)
      {
#ifdef DEBUG
      case 'd':
        ttf_debug = atoi(p);
        break;
#endif
      case 'C':
        codingscheme = p;
        break;
      case 'F':
        familyname = p;
        break;
      case 'h':
        hesc = atof(p);
        flags |= HORZESC;
        break;
      case 'v':
        vesc = atof(p);
        flags |= VERTESC;
        break;
      case 'x':
        xoff = -atof(p);
        flags |= XOFFSET;
        break;
      case 'y':
        yoff = atof(p);
        flags |= YOFFSET;
        break;
      case 'W':
        tfwid = fixword(atof(p));
        flags |= TFMWIDTH;
        break;
      case 'H':
        tfhgt = fixword(atof(p));
        flags |= TFMHEIGHT;
        break;
      case 'D':
        tfdep = fixword(atof(p));
        flags |= TFMDEPTH;
        break;
      case 'I':
        tfital = fixword(atof(p));
        flags |= TFMITALIC;
        break;
#if 0
      case 'f':
        optionfile(p);
        break;
#endif
      case 'e':
        switch (*p)
        {
        case 'e':
        case 'E':
          enc = EUC;
          break;
        case 'b':
        case 'B':
          enc = BIG5;
          break;
        case 's':
        case 'S':
          enc = SJIS;
          break;
        case 'x':
        case 'X':
          enc = SJIS;
          translate_to_SJIS = 1;
          break;
        case 'u':
        case 'U':
          enc = Unicode;
          break;
        default:
          fprintf(stderr, " Unknown encoding: %s\n", p);
          jumpout();
        }
        break;
        
      case 'p':
        numparam = atoi(p);
        if (numparam < 1 || numparam > 30)
        {
          fprintf(stderr, " Parameter count %d out of range!\n", numparam);
          jumpout();
        }
        for (i = 0; i < numparam; i++)
          if (++gargv,--gargc)
            parameters[i] = fixword(atof(*gargv));
          else
          {
            fprintf(stderr, " Not enough parameters (-p)!\n");
            jumpout();
          }
        break;

      case 'r':
        rotation = atoi(p);
        break;

      default:
        usage();
      }
    }
    else
    {
      /* designsize is always 10pt */
      hppd = (x_resolution / 72.27) * DESIGNSIZE;
      vppd = (y_resolution / 72.27) * DESIGNSIZE;

      /* car = 0 */
      if (flags & TFMWIDTH)
        tfmindex[car] = add_tfmwidth(tfwid);
      if (flags & TFMDEPTH)
        depindex[car] = add_tfmdepth(tfdep);
      if (flags & TFMHEIGHT)
        hgtindex[car] = add_tfmheight(tfhgt);
      if (flags & TFMITALIC)
        italindex[car] = add_tfmitalic(tfital);
      horzesc[car] = hesc * hppd;
      vertesc[car] = vesc * vppd;
      xoffset[car] = xoff * hppd;
      yoffset[car] = yoff * vppd;
      filename[car] = *gargv;
      charflags[car] = flags;
      flags = 0;
    }
  }
}


void main(argc, argv)
  int argc;
  char *argv[];
{
  int glyphindex;
  USHORT EM, x_ppem, y_ppem;
  double x_scaling, y_scaling;

  initialize();
  dialog(argc, argv);

  /* command line parameters act globally on all characters */
  for (car = 1; car < MAXPKCHAR; car++)
  {
    tfmindex[car] = tfmindex[0];
    depindex[car] = depindex[0];
    hgtindex[car] = hgtindex[0];
    italindex[car] = italindex[0];
    horzesc[car] = horzesc[0];
    vertesc[car] = vertesc[0];
    xoffset[car] = xoffset[0];
    yoffset[car] = yoffset[0];
    charflags[car] = charflags[0];
    filename[car] = filename[0];
  }

  hppp = round((x_resolution << 16) / 72.27);
  vppp = round((y_resolution << 16) / 72.27);

  add_suffix(pkname, "pk");
  add_suffix(tfmname, "tfm");
  if (!strcmp(pkname, "-"))
    pkfile = stdout;
  else if ((pkfile = fopen(pkname, WRITE_BIN)) == NULL)
  {
    fprintf(stderr, " Can't open PK file %s!\n", pkname);
    jumpout();
  }
  if (!strcmp(tfmname, "-"))
    tfmfile = stdout;
  else if ((tfmfile = fopen(tfmname, WRITE_BIN)) == NULL)
  {
    fprintf(stderr, " Can't open TFM file %s!\n", tfmname);
    jumpout();
  }

  writepreamble();
  RENDER_Init();
  fn = ttf_init(filename[0]);
  if (enc == Unicode)
    ttf_selectCmap(fn, 3, 1);
  else
    ttf_selectCmap(fn, 3, 4);

  EM = fn->head->unitsPerEm;
  x_ppem = round(point * x_resolution / 72.0);
  y_ppem = round(point * y_resolution / 72.0);
  x_scaling = (double)x_ppem / EM;
  y_scaling = (double)y_ppem / EM;

  b[0] = (schar >> 8) & 0xff;
  b[1] = schar & 0xff;
  if (translate_to_SJIS)
    JIS_to_SJIS(b);

  for (car = 0; car < nchar; car++)
  {
    bm = ttf_renderChar(fn, b, point, x_resolution, y_resolution);
    glyphindex = ttf_getFontIndex(fn, b);

    if (glyphindex >= 0)
    {
      height = bm->ysize;
      width = bm->xpoint;
      bwidth = bm->xsize;

      if (rotation)
      {
        if ((charflags[car] & HORZESC) == 0)
          horzesc[car] = round((fn->horz->ascender - fn->horz->descender)
                         * x_scaling);
        if ((charflags[car] & VERTESC) == 0)
          vertesc[car] = 0;
        if ((charflags[car] & XOFFSET) == 0)
          xoffset[car] = round(-(EM + fn->horz->descender - fn->glyf->yMax)
                         * x_scaling);
        else
          xoffset[car] -= round((EM - fn->glyf->yMax) * x_scaling);
        if ((charflags[car] & YOFFSET) == 0)
          yoffset[car] = round((fn->glyf->xMax - DESCENDER_ROTATED * EM)
                         * y_scaling);
        else
          yoffset[car] += round(fn->glyf->xMax * y_scaling);
        if ((charflags[car] & TFMWIDTH) == 0)
          tfmindex[car] = add_tfmwidth(
           (int)((double)(fn->horz->ascender - fn->horz->descender)
           / EM * 1048756));
        if ((charflags[car] & TFMHEIGHT) == 0)
          hgtindex[car] = add_tfmheight((int)(ASCENDER_ROTATED * 1048576));
        if ((charflags[car] & TFMDEPTH) == 0)
          depindex[car] = add_tfmdepth((int)(DESCENDER_ROTATED * 1048576));
        if ((charflags[car] & TFMITALIC) == 0)
          italindex[car] = 0;
      }
      else
      {
        if ((charflags[car] & HORZESC) == 0)
          horzesc[car] = round(ttf_getAdvanceWidth(fn, glyphindex)
                         * x_scaling);
        if ((charflags[car] & VERTESC) == 0)
          vertesc[car] = 0;
        /* we want to shift right with positive xoffset values */
        if ((charflags[car] & XOFFSET) == 0)
          xoffset[car] = round(-(ttf_getLsb(fn, glyphindex) + fn->glyf->xMin)
                         * x_scaling);
        else
          xoffset[car] -= round((ttf_getLsb(fn, glyphindex) + fn->glyf->xMin)
                          * x_scaling);
        /*
          ttf2pk is heavily based on pbmtopk; as a result we must use
          yMax instead of yMin because pbm bitmaps have the upper left
          corner as the origin.
        */
        if ((charflags[car] & YOFFSET) == 0)
          yoffset[car] = round(fn->glyf->yMax * y_scaling);
        else
          yoffset[car] += round(fn->glyf->yMax * y_scaling);
        if ((charflags[car] & TFMWIDTH) == 0)
          tfmindex[car] = add_tfmwidth(
           (int)((double)ttf_getAdvanceWidth(fn, glyphindex) / EM * 1048576));
        if ((charflags[car] & TFMHEIGHT) == 0)
          hgtindex[car] = add_tfmheight(
           (int)((double)fn->horz->ascender / EM * 1048576));
        if ((charflags[car] & TFMDEPTH) == 0)
          depindex[car] = add_tfmdepth(
           (int)(-(double)fn->horz->descender / EM * 1048576));
        if ((charflags[car] & TFMITALIC) == 0)
          italindex[car] = 0;
      }

      if (car < smallestch)
        smallestch = car;
      if (car > largestch)
        largestch = car;

      if (rotation)
        rotate_BMP();
      if (width > emwidth)
        emwidth = width;
      BMP_shipcharacter();
    }
    else
      filename[car] = NULL;

    BITMAP_done(bm);
  
    get_next_char(b, enc);
    if (car && ((car % 10) == 0))
    {
      fprintf(stderr, "[%d-%d] ", car - 9, car);
      fflush(stderr);
    }
  }

  if (((car - 1) % 10) == 0)
    fprintf(stderr, "\n");
  else
    fprintf(stderr, "[%d-%d]\n", car/10 * 10 + 1, car);
  
  writepostamble();
  writetfmfile();

  if (pkfile != stdout)
    fclose(pkfile);
  if (tfmfile != stdout)
    fclose(tfmfile);
  exit(0);
}


/* end of ttf2pk.c */
