// $Id: TeXpk.cc,v 1.2 2000/10/20 02:25:24 yotam Exp $
// Class TeXpk - implementation

#include "TeXpk.h"
#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <algorithm>
#include "BifStream.h"
#include "Rect.h"
#include "verbose.h"
#include "debug.h"

using namespace std;

#define VerboseMessage(vflag, el) _VerboseMessage(TeXpk::vflag, el)


class PixelRun
{
 public:
   PixelRun(US32 _bitmapWidth) :
           bitmapWidth(_bitmapWidth),
           currRow(0),
           currCol(0),
           nextCol(0),
           runCount(0),
           repeatCount(0)
           {}
   const US32   bitmapWidth;
   US32         currRow,
                currCol,
                nextCol,
                runCount,
                repeatCount;
}; // PixelRun


// for real implementation hiding
class _TeXpkImplement
{
 private:
   enum {MaxChars = 256};
   class Char
   {
    public:
      class LoadData
      {
       public:
         LoadData(BifStream& pkf, US8 f, bool vLd, US32 vFlgs, ostream* vstrm) :
            pkfile(pkf), 
            flag(f), 
            verboseLoad(vLd), 
            verbFlags(vFlgs), 
            verbStream(vstrm)
            {}
         BifStream&  pkfile;
         US8         flag;
         bool        verboseLoad;
         US32        verbFlags;
         ostream*    verbStream;
         bool        bitmapped;
         US8         dyn_f;
         US32        pl;
         streampos   ccPos;
      }; // LoadData
      Char();
      ~Char() {}

      bool load(_TeXpkImplement&  parent,
                BifStream&        pkfile,
                US8               flag,
                bool              verboseLoad);
      US32              cc;  // convenient - although mostly the index
      US32              tfmWidth;
      Int32             dx;
      Int32             dy;
      US32              bitmapWidth;
      US32              bitmapHeight;
      Int32             hoff;
      Int32             voff;
      vector<Rect>      rects;
    private:
      bool  loadHeader(LoadData&);
      bool  loadRaster(LoadData&, bool blackMode);
      void  loadBitmap(BifStream&);
      bool  blackRun(LoadData&, PixelRun&);
      bool  whiteRun(LoadData&, PixelRun&);
   }; // Char

 public:
   _TeXpkImplement();
   _TeXpkImplement(const _TeXpkImplement&);
   ~_TeXpkImplement();

   _TeXpkImplement&  operator=(const _TeXpkImplement&);

   void copyDynPart(const _TeXpkImplement&);

   void load(BifStream &pkfile, const char* fn);
   void clear();

   // retrieval members
   int           active()   const { return(_active);};
   const string& fileName() const { return(_fileName);}; 
   US8           id()       const { return(_id);};
   const string& comment()  const { return(_comment);};
   US32          ds()       const { return(_ds);};
   US32          cs()       const { return(_cs);};
   US32          hppp()     const { return(_hppp);};
   US32          vppp()     const { return(_vppp);};
   US16          nChars()   const { return(_nChars);};
   const Char * const *  
                 charData() const { return(_charData);};
   
   US8           charFirst() const {return _charFirst;}
   US8           charLast()  const {return _charLast;}
   US32           maxOffset(TeXpk::Direction di) const {return _maxOffset[di];}

   friend class  _TeXpkImplement::Char; 

 private:
   void           calcOffsets();
   int            _active;
   string         _fileName; 

   // From preamble
   US8            _id;       // id=89 is currently supported
   string         _comment;
   US32           _ds;
   US32           _cs;
   US32           _hppp;
   US32           _vppp;

   US16           _nChars;  // But only upto 256 supported.
   Char*          _charData[MaxChars];

   US8            _charFirst;
   US8            _charLast;
   US32           _maxOffset[TeXpk::NDir];

 public:
   US32           verbFlags;
   ostream*       verbStream;
}; // _TeXpkImplement


////////////////////////////////////////////////////////////////////////
_TeXpkImplement::Char::Char() :
   bitmapHeight(0)
{
} // _TeXpkImplement::Char::Char


////////////////////////////////////////////////////////////////////////
_TeXpkImplement::_TeXpkImplement() :
   _active(0),
   verbFlags(0),
   verbStream(&cerr)
{
   DHERE;
   for (int i = MaxChars;  i--;  _charData[i] = 0);
   for (int j = 4;  j--;  _maxOffset[j] = 0);
} // _TeXpkImplement


////////////////////////////////////////////////////////////////////////
void
_TeXpkImplement::clear()
{
   if (verbFlags & TeXpk::memory)
   {
      (*verbStream) << "TeXpk: clear" << endl;
   }
   // Yes, it could be optimized with link-list
   for (int i = MaxChars;  i--;)
   {
      if (_charData[i])
      {
         delete _charData[i];
         _charData[i] = 0;
      }
   }
   _comment = "";
   _active = 0;
} // clear


////////////////////////////////////////////////////////////////////////
_TeXpkImplement::~_TeXpkImplement()
{
   DHERE;
   clear();
} // ~_TeXpkImplement


////////////////////////////////////////////////////////////////////////
// Inspired from GFtoPK
static US32
pk_packed_num(BifStream &pkfile, US32 dyn_f, US32 i)
{
   US32  j;

   DASSERT(i < 14); // starting nybble not a repeat count
   if (i == 0)
   {
      while ((j = pkfile.get4bits()) == 0)
      {
         i++;
      }
      for (i++;  i--;  j = (j << 4) | (US32)pkfile.get4bits());
      i = (j + 16*(13 - dyn_f) + dyn_f) - 15;
   }
   else if (i > dyn_f)
   {
      i = 16*(i - dyn_f - 1) + pkfile.get4bits() + dyn_f + 1;
   }
   return(i);
} // pk_packed_num


////////////////////////////////////////////////////////////////////////
void _TeXpkImplement::Char::loadBitmap(BifStream& pkfile)
{
   Rect   currRect;
   for (US32  currRow = 0;  currRow != bitmapHeight;  ++currRow)
   {
      US8  black = 2; // undefined,
      for (US32  currCol = 0;  currCol <= bitmapWidth;  ++currCol)
      {
         // we fake a white right padded column
         US8   pixel = (currCol < bitmapWidth ? pkfile.getByteBits(1) : 0);
         DASSERT(pixel <= 1);
         if (black != pixel)
         {
            if (pixel)
            {            // initialize rectangle
               currRect.x = currCol;
               currRect.y = currRow;
               currRect.h = 1;
            }
            else if (black == 1)
            {            // add rectangle
               currRect.w = currCol - currRect.x;
               rects.push_back(currRect);
            }
            black = pixel;
         }
      }
   }
} // _TeXpkImplement::Char::loadBitmap


////////////////////////////////////////////////////////////////////////
bool _TeXpkImplement::Char::blackRun(LoadData& loader, PixelRun&  pr)
{
   bool  ok = true;
   US32  lastCol = pr.nextCol - 1;
   US32  endCol = min(lastCol, pr.bitmapWidth - 1);
   rects.push_back(Rect(pr.currCol, pr.currRow,
                        endCol - pr.currCol + 1, pr.repeatCount + 1));
   if (pr.nextCol >= pr.bitmapWidth)
   {
      pr.nextCol -= pr.bitmapWidth;
      pr.currRow += (pr.repeatCount + 1);
      pr.repeatCount = 0;
      US32  nFullRows = pr.nextCol / pr.bitmapWidth;

      // possible overflow was checked by nPixels
      lastCol %= pr.bitmapWidth;
      if (nFullRows)
      {
         rects.push_back(Rect(0, pr.currRow, pr.bitmapWidth, nFullRows));
         pr.currRow += nFullRows;
      }
      if (lastCol != pr.bitmapWidth - 1)
      {
         rects.push_back(Rect(0, pr.currRow, lastCol + 1, 1));
      }
   }
   return(ok);
} // _TeXpkImplement::Char::blackRun


////////////////////////////////////////////////////////////////////////
bool _TeXpkImplement::Char::whiteRun(LoadData& loader, PixelRun&  pr)
{
   bool  ok = true;
   if (pr.nextCol >= pr.bitmapWidth)
   {
      pr.currRow += (pr.repeatCount + 1);
      pr.repeatCount = 0;
      pr.nextCol -= pr.bitmapWidth;
      US32  nEmptyRows = pr.nextCol / pr.bitmapWidth;
      pr.currRow += nEmptyRows;
   }
   return(ok);
} // whiteRun


////////////////////////////////////////////////////////////////////////
bool
_TeXpkImplement::Char::load
(
   _TeXpkImplement& parent,
   BifStream &      pkfile,
   US8              flag,
   bool             verboseLoad
)
{
   // For the Error macro to work
   US32      verbFlags  = parent.verbFlags;
   ostream*  verbStream = parent.verbStream;
   LoadData  loadData(pkfile, flag, verboseLoad, verbFlags, verbStream);

   bool blackMode = (flag & 8);
   bool ok = loadHeader(loadData);
   ok = ok && loadRaster(loadData, blackMode);
   return(ok);
} // TeXpkImplement::Char::load


#undef  VerboseLoad
#define VerboseLoad(el) if (loader.verboseLoad) {(*verbStream) << el << endl;}

////////////////////////////////////////////////////////////////////////
bool  _TeXpkImplement::Char::loadHeader(LoadData& loader)
{
   bool       ok = true;
   US32       pl, rasterNbytes;
   loader.dyn_f     = (loader.flag >> 4) & 0xf;  DASSERT(loader.dyn_f != 0xf);
   loader.bitmapped = (loader.dyn_f == 14);
   US8        low3bits  = loader.flag & 7;
   US32       low2bits  = low3bits & 3;

   BifStream& pkfile = loader.pkfile;
   dy = 0;
   if (low3bits <= 3)
   {  // Short form
      pl           = (low2bits << 8) | (US32)pkfile.get1ubyte();
      cc           = pkfile.get1ubyte();   loader.ccPos = pkfile.tellg();
      tfmWidth     = pkfile.get3ubyte();
      dx           = pkfile.get1ubyte();
      bitmapWidth  = pkfile.get1ubyte();
      bitmapHeight = pkfile.get1ubyte();
      hoff         = pkfile.get1byte();
      voff         = pkfile.get1byte();
      rasterNbytes = pl - (3+1+1+1+1+1);
   }
   else if (low3bits != 7)
   {  // Extended short form
      pl           = (low2bits << 16) | (US32)pkfile.get2ubyte();
      cc           = pkfile.get1ubyte();   loader.ccPos = pkfile.tellg();
      tfmWidth     = pkfile.get3ubyte();
      dx           = pkfile.get2ubyte();
      bitmapWidth  = pkfile.get2ubyte();
      bitmapHeight = pkfile.get2ubyte();
      hoff         = pkfile.get2byte();
      voff         = pkfile.get2byte();
      rasterNbytes = pl - (3+2+2+2+2+2);
   }
   else
   {  // Long form
      pl           = pkfile.get4ubyte();
      cc           = pkfile.get4ubyte();   loader.ccPos = pkfile.tellg();
      tfmWidth     = pkfile.get4ubyte();
      dx           = pkfile.get4ubyte();
      dy           = pkfile.get4ubyte();
      bitmapWidth  = pkfile.get4ubyte();
      bitmapHeight = pkfile.get4ubyte();
      hoff         = pkfile.get4byte();
      voff         = pkfile.get4byte();
      rasterNbytes = pl - (3+4+4+4+4+4);
      if (cc >= 256)
      {
         US32&     verbFlags = loader.verbFlags;
         ostream*& verbStream = loader.verbStream;
         Error("Unsupported character code " << cc << " >= 256");
         ok = 0;
      }
   }
   loader.pl = pl;
   return ok;
} // _TeXpkImplement::Char::loadHeader


////////////////////////////////////////////////////////////////////////
bool  _TeXpkImplement::Char::loadRaster(LoadData& loader, bool blackMode)
{
   bool        ok = true;
   BifStream&  pkfile = loader.pkfile;
   US32&       verbFlags = loader.verbFlags;
   ostream*&   verbStream = loader.verbStream;
   pkfile.bitscanFlush();
   PixelRun  pixelRun(bitmapWidth);
   if (loader.bitmapped)
   {
      VerboseLoad("Loading bitmap for cc=0x" << hex << cc);
      loadBitmap(pkfile);
   }
   else
   {  // Check unrealistic overflow
      ok = (bitmapHeight == 0) || (bitmapWidth + 1 < ULONG_MAX / bitmapHeight);
      US32  rasterSize (ok ? bitmapWidth * bitmapHeight : 0);
      US32  nPixels = 0;
      while (ok && nPixels < rasterSize)
      {
         US8  nybble = pkfile.get4bits();
         if (nybble >= 14)
         { // repeat count
            pixelRun.repeatCount = ((nybble == 15)
               ? 1
               : pk_packed_num(pkfile, loader.dyn_f, pkfile.get4bits()));
            nPixels += pixelRun.repeatCount * bitmapWidth;;
            if (nPixels <= rasterSize)
            {
               DASSERT(pixelRun.currRow + pixelRun.repeatCount <
                       bitmapHeight);
               if (!rects.empty() && (rects.back().y == pixelRun.currRow))
               {  // fix 1st rectangle on this row
                  DASSERT(rects.back().h == 1);
                  rects.back().h = pixelRun.repeatCount + 1;
               }
            }
            else
            {
               ok = false;
               Error("Repeat count " << pixelRun.repeatCount <<
                     " bitmap row overflow");
            }
         }
         else
         {
            typedef bool (Char::*PFRun)(LoadData&, PixelRun&);
            static const PFRun  WBrun[2] = {&Char::whiteRun, &Char::blackRun};
            pixelRun.runCount = pk_packed_num(pkfile, loader.dyn_f, nybble);
            pixelRun.nextCol = pixelRun.currCol + pixelRun.runCount;
            nPixels += pixelRun.runCount;
            if (nPixels <= rasterSize)
            {
               (this->*(WBrun[blackMode]))(loader, pixelRun);
               pixelRun.currCol = pixelRun.nextCol % bitmapWidth;
               blackMode = !blackMode;
            }
            else
            {
               ok = false;
               Error("Run count " << pixelRun.runCount <<
                     " bitmap row overflow");
            }
         }
      }
      if (ok && ((pixelRun.currCol != 0) ||
                 (pixelRun.currRow != bitmapHeight) ||
                 (nPixels != rasterSize)))
      {
         ok = false;
         Error("Raster data not synchronized");
      }
   }
   if (ok)
   {
      streampos  endRasterPos = pkfile.tellg();
      if ((streampos)(loader.ccPos + loader.pl) != endRasterPos)
      {
         ok = false;
         Error("cc=0x" << hex << cc << dec <<
               " (begin-postion=" << loader.ccPos << ") + (pl=" << loader.pl <<
               ") != (end-postion=" << endRasterPos << ")");
      }
      // relloacate to save space
      vector<Rect> newRects(rects);
      rects.swap(newRects);
   }
   return ok;
} // _TeXpkImplement::Char::loadRaster


////////////////////////////////////////////////////////////////////////
static void  maxby(US32 &var, Int32 val)
{
   if ((Int32)var < val)
   {
      var = val;
   }
} // maxby


////////////////////////////////////////////////////////////////////////
void
_TeXpkImplement::calcOffsets()
{
   int  i;
   for (i = 4;  i--;  _maxOffset[i] = 0);
   for (i = 256, _charLast = 0;  i--;)
   {
      US8  c = (US8)i;
      Char*  pc = _charData[c];
      if (pc != 0)
      {
         Int32   w = pc->bitmapWidth;
         Int32   h = pc->bitmapHeight;
         Int32   hoff = pc->hoff;
         Int32   voff = pc->voff;
         maxby(_maxOffset[TeXpk::Left], hoff);
         maxby(_maxOffset[TeXpk::Right], w - hoff);
         maxby(_maxOffset[TeXpk::Up], voff);
         maxby(_maxOffset[TeXpk::Down], h - voff);
         if (_charLast == 0)  _charLast = c;
         _charFirst = c;         
      }
   }

} // calcOffsets


////////////////////////////////////////////////////////////////////////
void
_TeXpkImplement::load(BifStream &pkfile, const char* fn)
{
   enum pkOpcode
   {  // original naming as in GFtoPK weaved document
      pk_xxx1 = 0xf0,
      pk_xxx2,
      pk_xxx3,
      pk_xxx4,
      pk_yyy,
      pk_post,
      pk_no_op,
      pk_pre
   };

   bool  ok = true;
   _fileName = fn;
   US8   code;
   bool  preSeen = false;
   bool  postSeen = false;
   bool  anyCharYet = false;

   while (ok && !postSeen)
   {
      code = pkfile.get1ubyte();
      ok = (preSeen != (code == pk_pre));  // pk_pre iff the beginning
      if (!ok)
      {
         Error("Preamble not in the beginning");
      }
      else
      {
         if (code >= 0xf0)
         {
            switch (code)
            {
               case pk_xxx1:
                  pkfile.seekg( pkfile.get1ubyte(), ios::cur );
                  break;
               case pk_xxx2:
                  pkfile.seekg( pkfile.get2ubyte(), ios::cur );
                  break;
               case pk_xxx3:
                  pkfile.seekg( pkfile.get3ubyte(), ios::cur );
                  break;
               case pk_xxx4:
                  pkfile.seekg( pkfile.get4ubyte(), ios::cur );
                  break;
               case pk_yyy:
                  pkfile.seekg( 4, ios::cur );
                  break;
               case pk_post:
                  postSeen = true;
                  break;
               case pk_no_op:
                  break;
               case pk_pre:
                  preSeen = true;
                  _id = pkfile.get1ubyte();
                  if (_id != 89)
                  {
                     ok = 0;
                     Error("PK-id=" << _id << "!=89");
                  }
                  {
                     unsigned  commLen = pkfile.get1ubyte();
                     char*     cs = new char[commLen + 1]; 
                     pkfile.read(cs, commLen);
                     cs[commLen] = '\0';
                     _comment = cs;
                     delete [] cs;
                  }
                  _ds = pkfile.get4ubyte();
                  _cs = pkfile.get4ubyte();
                  _hppp = pkfile.get4ubyte();
                  _vppp = pkfile.get4ubyte();
                  VerbosePreamble("Preamble: " << endl <<
                   "  id=" << (int)_id <<
                   ", comment=<" << _comment << '>' << endl <<
                   "ds=" << _ds << ", cs=" << _cs <<
                   ", hppp=" << _hppp << ", vppp=" << _vppp);
                  break;
               default: Error("Bad PK opcode" << hex);
            }
         }
         else // character data
         {
            Char*  pChar = new(Char);   VerboseNew(pChar);
            if (pChar == 0)
            {
               ok = false;
               Error("Failed to allocate memory for Character");
            }
            else
            {
               bool verboseLoad = (verbFlags & TeXpk::allChars) ||
                              ((verbFlags & TeXpk::firstChar) && !anyCharYet);
               ok = pChar->load(*this, pkfile, code, verboseLoad);
               anyCharYet = true;
               DASSERT(!ok || pChar->cc < MaxChars);
               if (ok)
               {
                  _charData[pChar->cc] = pChar;
               }
               else
               {
                  delete pChar;   VerboseDelete(pChar);
               }
            }
         }
      }
   }
   _active = ok;
   if (ok)
   {
      calcOffsets();
   }
   else
   {
      clear(); // also sets:  _active = 0
   }
} // _TeXpkImplement::load


////////////////////////////////////////////////////////////////////////
void
_TeXpkImplement::copyDynPart(const _TeXpkImplement &x)
{
   bool  ok = true;
   for (int i = _nChars;  i--;)
   {
      _charData[i] = (x._charData[i] ? new Char(*x._charData[i]) : 0);
   }
   for (int k = 4;  k--;  _maxOffset[k] = x._maxOffset[k]);
   if (!ok)
   {
      clear();
   }
} // _TeXpkImplement::copyDynPart(const _TeXpkImplement &)


////////////////////////////////////////////////////////////////////////
_TeXpkImplement::_TeXpkImplement(const _TeXpkImplement &x) :
   _active(x._active),
   _fileName(x._fileName),
   _id(x._id),
   _comment(x._comment),
   _ds(x._ds),
   _cs(x._cs),
   _hppp(x._hppp),
   _vppp(x._vppp),
   _nChars(x._nChars),
   verbFlags(x.verbFlags),
   verbStream(x.verbStream)
{
   if (_active)
   {
      copyDynPart(x);
   }
   _charFirst = x._charFirst;
   _charLast  = x._charLast;
   for (int k = 4;  k--;  _maxOffset[k] = x._maxOffset[k]);
} // _TeXpkImplement::_TeXpkImplement(const _TeXpkImplement &)


////////////////////////////////////////////////////////////////////////
// We alerady know that  this != &x
_TeXpkImplement&
_TeXpkImplement::operator=(const _TeXpkImplement &x)
{
   clear();

   _active = x._active;
   verbStream = x.verbStream;
   verbFlags = x.verbFlags;

   if (_active)
   {
      _fileName  = x._fileName;
      _id        = x._id;
      _comment   = x._comment;
      _ds        = x._ds;
      _cs        = x._cs;
      _hppp      = x._hppp;
      _vppp      = x._vppp;
      _nChars    = x._nChars;
      _charFirst = x._charFirst;
      _charLast  = x._charLast;
      for (int k = 4;  k--;  _maxOffset[k] = x._maxOffset[k]);

      copyDynPart(x);
   }

   return(*this);
} // _TeXpkImplement &  operator = (const _TeXpkImplement &);



////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
//                     TeXpk  -  public members                       //
//                                                                    //


////////////////////////////////////////////////////////////////////////
TeXpk::TeXpk() :
   _impl(0)
{
} // TeXpk


////////////////////////////////////////////////////////////////////////
TeXpk::TeXpk(const char* fn, US32 verboseFlag, ostream& verbStream) :
   _impl(0)
{
   open(fn, verboseFlag, verbStream);
} // TeXpk


////////////////////////////////////////////////////////////////////////
TeXpk::TeXpk(const TeXpk& x) : _impl(0)
{
   if (x._impl)
   {
      ostream*  verbStream = x._impl->verbStream;
      US32      verbFlags  = x._impl->verbFlags;
      _impl = new _TeXpkImplement(*x._impl);    VerboseNew(_impl);
      if (_impl == 0)
      {
         Error("No memory for copy constructor");
      }
   }
} // TeXpk::TeXpk(const TeXpk &)


////////////////////////////////////////////////////////////////////////
TeXpk::~TeXpk()
{
   if (_impl)
   {
      US32    verbFlags = _impl->verbFlags;
      ostream* verbStream = _impl->verbStream;
      delete _impl;   VerboseDelete(_impl);
   }
} // ~TeXpk


////////////////////////////////////////////////////////////////////////
TeXpk&  TeXpk::operator=(const TeXpk& rhs)
{
   if (this != &rhs)
   {
      if (_impl)
      {
         *_impl = *rhs._impl;
      }
   }
   return(*this);
} //TeXpk::operator =


////////////////////////////////////////////////////////////////////////
void  TeXpk::open(const char* fn, US32 verbFlags, ostream& rVerbStream)
{
   ostream* verbStream = &rVerbStream;
   int  ok = 1;
   if (_impl)
   {
      _impl->clear();
   }
   else
   {
      _impl = new _TeXpkImplement;  VerboseNew(_impl);
      if (_impl == 0)
      {
         ok = 0;
         Error("Failed to allocate _impl");
      }
   }
   BifStream pkfile(fn);        VerboseFclopen("Opening " << fn);
   if (!pkfile)
   {
      ok = 0;
      Error("Failed to open " << fn);
   }
   if (ok)
   {                            VerboseFclopen(fn << " opened");
      _impl->verbFlags = verbFlags;
      _impl->verbStream = verbStream;
      _impl->load(pkfile, fn);
      VerboseFclopen(fn << " closed");
   }
} // TeXpk::open


////////////////////////////////////////////////////////////////////////
int  TeXpk::active() const
{
   int  rc = (_impl ? _impl->active() : 0);
   return(rc);
} // TeXpk::active


////////////////////////////////////////////////////////////////////////
US8  TeXpk::id() const
{
   US8  rc = (_impl ? _impl->id() : 0);
   return(rc);
} // TeXpk::id


////////////////////////////////////////////////////////////////////////
US32  TeXpk::designSize() const
{
   US32  rc = (_impl ? _impl->ds() : 0);
   return(rc);
} // TeXpk::designSize


////////////////////////////////////////////////////////////////////////
const string&  TeXpk::comment() const
{
   static const string  empty("");
   const string& s = (_impl ? _impl->comment() : empty);
   return(s);
} // TeXpk::comment


////////////////////////////////////////////////////////////////////////
US32  TeXpk::checkSum() const
{
   US32  rc = (_impl ? _impl->cs() : 0);
   return(rc);
} // TeXpk::checkSum


////////////////////////////////////////////////////////////////////////
US32  TeXpk::hppp() const
{
   US32  rc = (_impl ? _impl->hppp() : 0);
   return(rc);
} // TeXpk::hppp


////////////////////////////////////////////////////////////////////////
US32  TeXpk::vppp() const
{
   US32  rc = (_impl ? _impl->vppp() : 0);
   return(rc);
} // TeXpk::vppp


////////////////////////////////////////////////////////////////////////
US16  TeXpk::nChars() const
{
   US16  rc = (_impl ? _impl->nChars() : 0);
   return(rc);
} // TeXpk::nChars


////////////////////////////////////////////////////////////////////////
bool  TeXpk::hasChar(US8 c) const
{
   bool  rc = (_impl != 0) && (c < 256) && (_impl->charData()[c] != 0);
   return(rc);
} // TeXpk::hasChar


////////////////////////////////////////////////////////////////////////
US32  TeXpk::tfmWidth(US8 c) const
{
   const _TeXpkImplement::Char*  pc;
   US32  rc = (_impl && (c < 256) && ((pc = _impl->charData()[c]))
               ? pc->tfmWidth
               : 0);
   return(rc);
} // TeXpk::tfmWidth


////////////////////////////////////////////////////////////////////////
Int32  TeXpk::dx(US8 c) const
{
   const _TeXpkImplement::Char*  pc;
   Int32  rc = (_impl && (c < 256) && ((pc = _impl->charData()[c])) 
                ? pc->dx 
                : 0);
   return(rc);
} // TeXpk::dx


////////////////////////////////////////////////////////////////////////
Int32  TeXpk::dy(US8 c) const
{
   const _TeXpkImplement::Char*  pc;
   Int32  rc = (_impl && (c < 256) && ((pc = _impl->charData()[c])) 
                ? pc->dy 
                : 0);
   return(rc);
} // TeXpk::dy


////////////////////////////////////////////////////////////////////////
US32  TeXpk::bitmapWidth(US8 c) const
{
   const _TeXpkImplement::Char*  pc;
   US32  rc = (_impl && (c < 256) && ((pc = _impl->charData()[c]))
               ? pc->bitmapWidth
               : 0);
   return(rc);
} // TeXpk::bitmapWidth


////////////////////////////////////////////////////////////////////////
US32  TeXpk::bitmapHeight(US8 c)  const
{
   const _TeXpkImplement::Char*  pc;
   US32  rc = (_impl && (c < 256) && ((pc = _impl->charData()[c]))
               ? pc->bitmapHeight
               : 0);
   return(rc);
} // TeXpk::bitmapHeight


////////////////////////////////////////////////////////////////////////
Int32  TeXpk::hoff(US8 c) const
{
   const _TeXpkImplement::Char*  pc;
   Int32  rc = (_impl && (c < 256) && ((pc = _impl->charData()[c])) 
                ? pc->hoff 
                : 0);
   return(rc);
} // TeXpk::hoff


////////////////////////////////////////////////////////////////////////
Int32  TeXpk::voff(US8 c) const
{
   const _TeXpkImplement::Char*  pc;
   Int32  rc = (_impl && (c < 256) && ((pc = _impl->charData()[c])) 
                ? pc->voff 
                : 0);
   return(rc);
} // TeXpk::voff


////////////////////////////////////////////////////////////////////////
US32  TeXpk::nRects(US8 c) const
{
   DASSERT(_impl && c < 256 && _impl->charData()[c]);
   const _TeXpkImplement::Char*  pc = _impl->charData()[c];
   US32  rc = pc->rects.size();
   return(rc);
} // TeXpk::nRects


////////////////////////////////////////////////////////////////////////
void  TeXpk::rect(Rect& rect, US8 c, US32 rectInd) const
{
   DASSERT(_impl && c < 256 && _impl->charData()[c]);
   const _TeXpkImplement::Char*  pc = _impl->charData()[c];
   rect = pc->rects[rectInd];
} // TeXpk::rect


////////////////////////////////////////////////////////////////////////
US32  TeXpk::maxOffset(Direction di) const
{
   return(_impl != 0 ? _impl ->maxOffset(di) : 0);
} // TeXpk::maxOffset


////////////////////////////////////////////////////////////////////////
US8  TeXpk::charFirst() const
{
   return(_impl != 0 ? _impl->charFirst() : 0);
} // TeXpk::charFirst


////////////////////////////////////////////////////////////////////////
US8  TeXpk::charLast() const
{
   return(_impl != 0 ? _impl->charLast() : 0);
} // TeXpk::charLast


#if TEST
#include <cstdlib>
int  main(int argc, char** argv)
{
   if (argc != 3)
   {
      cerr << "Usage:" << endl << argv[0] << "<pkfile> <veboseFlags>" << endl;
      return 1;
   }
   US32  verbFlags = 0;
   verbFlags = atol(argv[2]);
   TeXpk  texpk;
   texpk  = TeXpk(argv[1], verbFlags);
   return 0;
} // main
#endif
