/*
** SearchDialog manages all the search information.  It is popped up, and 
** permits the user to conduct multiple searches before closing and actually
** working on the resulting index list.  It should be maintained as a 
** static object, so that it can be recalled to further refine an 
** existing search.
*/
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;

public class SearchDialog extends Dialog implements ItemListener, ActionListener, TextListener, WindowListener
{
    private Vector m_listVec, m_historyVec;
    private Button m_searchBut, m_resetBut, m_closeBut, m_cancelBut;
    private Choice m_levelChoice;
    private Checkbox m_andCheck, m_orCheck, m_exactCheck, m_caseCheck;
    private Checkbox m_keyCheck, m_crossCheck, m_allTypeCheck, m_allFieldCheck;
    private Checkbox[] m_typesCheck, m_fieldsCheck;
    private Panel m_typesPan, m_fieldsPan;
    private TextField m_phraseField;
    private TextArea m_historyArea;
    //    private RecList m_list;
    private Hashtable m_Index; // reference to file index
    private String[] m_Results; // storage for results of searches
    private RandomAccessFile m_RAFile; // link to file to work on

    private static boolean m_Debug;
    private String m_EOLN;

    /*
    ** Constructor, build everything as if empty
    */
    SearchDialog(Frame f, String types, String fields)
    {
        super(f, "Search Dialog", true);
        setDebug(false);
        setBackground(Color.lightGray);
        m_EOLN = System.getProperty("line.separator");
        addWindowListener(this);
        Panel inputPan = new Panel(new BorderLayout());
        Label lab = new Label("    Phrase: ");
        inputPan.add("West", lab);
        m_phraseField = new TextField(40);
        m_phraseField.addTextListener(this);
        m_phraseField.setBackground(Color.white);
        inputPan.add("Center", m_phraseField);
/*
**    private CheckBox m_addCheck, m_orCheck, m_exactCheck;
*/
        Panel checkPan = new Panel(new GridLayout(0, 5));
        CheckboxGroup cbg = new CheckboxGroup();
        lab = new Label("     Logic:");
        checkPan.add(lab);
        m_andCheck = new Checkbox("AND Words", cbg, true);
        checkPan.add(m_andCheck);
        m_orCheck = new Checkbox("OR Words", cbg, false);
        checkPan.add(m_orCheck);
        m_exactCheck = new Checkbox("Exact Phrase", cbg, false);
        checkPan.add(m_exactCheck);
        m_caseCheck = new Checkbox("Match Case");
        checkPan.add(m_caseCheck);
/*        
**    private Checkbox m_keyCheck, m_crossCheck, m_allTypeCheck, m_allFieldCheck;
*/
        lab = new Label("    Include:");
        checkPan.add(lab);
        m_keyCheck = new Checkbox("Citation Key", null, true);
        checkPan.add(m_keyCheck);
        checkPan.add(new Label(" "));
        m_allTypeCheck = new Checkbox("All Types", null, true);
        m_allTypeCheck.addItemListener(this);
        checkPan.add(m_allTypeCheck);
        m_allFieldCheck = new Checkbox("All Fields", null, true);
        m_allFieldCheck.addItemListener(this);
        checkPan.add(m_allFieldCheck);
        // make array of types
        StringTokenizer st = new StringTokenizer(types);
        int numTokens = st.countTokens();
        m_typesCheck = new Checkbox[numTokens];
        for(int i=0; i<numTokens; i++) {
            m_typesCheck[i] = new Checkbox(st.nextToken(), null, true);
            m_typesCheck[i].addItemListener(this);}
        // make an array of fields
        st = new StringTokenizer(fields);
        numTokens = st.countTokens();
        m_fieldsCheck = new Checkbox[numTokens];
        for(int i=0; i<numTokens; i++) {
            m_fieldsCheck[i] = new Checkbox(st.nextToken(), null, true);
            m_fieldsCheck[i].addItemListener(this);}
        // add the types
        for(int i=0; i<m_typesCheck.length; i++) {
            checkPan.add(m_typesCheck[i]);}
        // fill in to back with labels
        int blanks = 5-(m_typesCheck.length-5*((int)(m_typesCheck.length/5.0)));
        if(blanks == 5) blanks = 0;
        for(int i=0; i<blanks; i++) {
            checkPan.add(new Label(" "));}
        // add the fields
        for(int i=0; i<m_fieldsCheck.length; i++) {
            checkPan.add(m_fieldsCheck[i]);}

/*
** add together into top panel
*/
        Panel topPan = new Panel(new BorderLayout());
        topPan.add("North", inputPan);
        topPan.add("Center",checkPan);
/*
** set up the text area
*/
        m_historyArea = new TextArea(18,60);
        m_historyArea.setBackground(Color.lightGray);
/*
** create button panel
**    private Button m_searchBut, m_resetBut, m_closeBut;
**    private Choice m_levelChoice;
*/
        
        GridBagLayout gb = new GridBagLayout();
        GridBagConstraints gc = new GridBagConstraints();
        Panel butPan = new Panel(gb);
        gc.gridx = 0;
        gc.gridy = 0;
        gc.gridwidth = 1;
        gc.gridheight = 1;
        gc.fill = GridBagConstraints.BOTH;
        gc.ipadx = 2;
        gc.ipady = 2;
        gc.anchor = GridBagConstraints.CENTER;
        gc.weightx = 1;
        gc.weighty = 1;
        m_searchBut = new Button("Search");
        m_searchBut.setEnabled(false);
        m_searchBut.addActionListener(this);
        gb.setConstraints(m_searchBut, gc);
        butPan.add(m_searchBut);
        gc.gridx = 1;
        m_resetBut = new Button("Reset");
//        m_resetBut.setEnabled(false);
        m_resetBut.addActionListener(this);
        gb.setConstraints(m_resetBut, gc);
        butPan.add(m_resetBut);
        gc.gridx = 2;
        m_closeBut = new Button("Close");
        m_closeBut.setEnabled(true);
        m_closeBut.addActionListener(this);
        gb.setConstraints(m_closeBut, gc);
        butPan.add(m_closeBut);
        gc.gridx = 3;
        m_cancelBut = new Button("Cancel");
        m_cancelBut.setEnabled(true);
        m_cancelBut.addActionListener(this);
        gb.setConstraints(m_cancelBut, gc);
        butPan.add(m_cancelBut);
        gc.gridx = 4;
        gc.fill = GridBagConstraints.NONE;
        gc.anchor = GridBagConstraints.EAST;
        lab = new Label("Level");
        gb.setConstraints(lab, gc);
        butPan.add(lab);
        gc.gridx = 5;
        gc.fill = GridBagConstraints.BOTH;
        gc.anchor = GridBagConstraints.CENTER;
        m_levelChoice = new Choice();
        m_levelChoice.addItemListener(this);
        gb.setConstraints(m_levelChoice, gc);
        butPan.add(m_levelChoice);
/*
** bottom panel
**    private Panel m_typesPan, m_fieldsPan;
*/
        Panel botPan = new Panel(new BorderLayout());
        botPan.add("North", butPan);
        m_typesPan = new Panel(new GridLayout(0, 5));
        botPan.add("Center", m_typesPan);
        m_fieldsPan = new Panel(new GridLayout(0, 5));
        botPan.add("South", m_fieldsPan);
/*
** put it together
*/
        add("North", topPan);
        add("Center", m_historyArea);
        add("South", botPan);
        pack();
        center(f);
        m_listVec = new Vector();
        m_historyVec = new Vector();
    }

    /*
     ** center(Container over) centers the dialog over the passed
     ** container
     */
    public void center(Container over)
    {
        if(over.isVisible())
        {
            int width = getSize().width;
            int height = getSize().height;
            setSize(width, height);
            int parentScreenX = over.getLocationOnScreen().x;
            int parentScreenY = over.getLocationOnScreen().y;
            int parentWidth = over.getBounds().width;
            int parentHeight = over.getBounds().height;
            int screenX = Math.max(0, parentScreenX + parentWidth / 2 - width / 2);
            int screenY = Math.max(0, parentScreenY + parentHeight / 2 - height / 2);
            setLocation(screenX, screenY);
        }
    }

    /**
     Set a link to the random access file
     */
    public void setFile(RandomAccessFile ra) {
        m_RAFile = ra;}

    /**
     Set a link to the index for this file
     */
    public void setIndex(Hashtable in) {
        // reset things if m_Index has changed
        m_Index = in;
        // use this index as a source for creating the list of keys
        String[] keys = makeKeyList(m_Index);
        // set the vectors to blank
        m_listVec = new Vector();
        m_historyVec = new Vector();
        // push the list onto the vector
        m_listVec.addElement(keys);
        m_historyVec.addElement("Initial: "
                                + keys.length
                                + " total elements."
                                + m_EOLN);
        // display the history
        displayHistory();
    }

    /**
     Display the hisory elements, pushing them onto the
     history panel.
     */
    public void displayHistory() {
        // clear the level choice widgit
        m_levelChoice.removeAll();
        // clear the display text
        m_historyArea.setText("");
        // enumerate through the vector
        for(int i=0; i<m_historyVec.size(); i++) {
            String history = (String)m_historyVec.elementAt(i);
            // push it onto the display
            m_historyArea.append("***** Search level "
                                 + i + " *****" + m_EOLN
                                 + history + m_EOLN);
            // add entry to level pulldown
            m_levelChoice.addItem(String.valueOf(i));}
    }

    /**
     Get the results of a search, as an array of keys that can be
     set directly into the key list
     */
    public String[] getKeys() {
        return m_Results;}

    /**
     Make a list of keys in order from the one that is closest to the
     beginning of the file.
     */
    public String[] makeKeyList(Hashtable index) {
        // make some storage space
        String[] keys = new String[index.size()];
        long[] positions = new long[index.size()];
        for(int i=0; i<positions.length; i++) {positions[i] = -1;}
        int top = 0;
        // enumerate over the key elements
        Enumeration enum = index.keys();
        while(enum.hasMoreElements()) {
            String key = (String)enum.nextElement();
            String[] data = (String[])index.get(key);
            long pos = (Long.valueOf(data[ListFrame.FILE_POS])).longValue();
            // just for fun, output it
            for(int i=0; i<positions.length; i++) {
                debug(keys[i] + ", " + positions[i]);}
            // bubble the string in
            int point = 0;
            while((positions[point] != -1) && (pos > positions[point])) {
                point++;}
            // got the position, move everything else up
//            int point2 = positions.length - 1;
            int point2 = top;
            while(point2 > point) {
                point2--;
                positions[point2+1] = positions[point2];
                keys[point2+1] = keys[point2];}
            // now set the values
            keys[point] = key;
            positions[point] = pos;
            top++;}
        // just for fun, output it
        for(int i=0; i<positions.length; i++) {
            debug(keys[i] + ", " + positions[i]);}
        // return the string
        return keys;
    }

    /*
    ** Deal with item state changes
    */
    public void itemStateChanged(ItemEvent evt)
    {
        // toggle the all type check, if necessary
        if(evt.getSource().equals(m_allTypeCheck)) {
            for(int i=0; i<m_typesCheck.length; i++) {
                m_typesCheck[i].setState(m_allTypeCheck.getState());}}
        // toggle the all field check, if necessary
        if(evt.getSource().equals(m_allFieldCheck)) {
            for(int i=0; i<m_fieldsCheck.length; i++) {
                m_fieldsCheck[i].setState(m_allFieldCheck.getState());}}
        // switch all field and all type as necessary
        if((!evt.getSource().equals(m_allTypeCheck))
           && (!evt.getSource().equals(m_allFieldCheck))
           && (!evt.getSource().equals(m_levelChoice))) {
            // not one of masters, so check children
            boolean state = true;
            for(int i=0; i<m_typesCheck.length; i++) {
                state = state && m_typesCheck[i].getState();}
            m_allTypeCheck.setState(state);
            state = true;
            for(int i=0; i<m_fieldsCheck.length; i++) {
                state = state && m_fieldsCheck[i].getState();}
            m_allFieldCheck.setState(state);}
        // check to see if choosing a different level
        if(evt.getSource().equals(m_levelChoice)) {
            // chosen to change the level
            int level = m_levelChoice.getSelectedIndex();
            resetSearch(level);}
    }

    /*
    ** text event listener
    */
    public void textValueChanged(TextEvent evt) {
        if(evt.getSource().equals(m_phraseField)) {
            if((m_phraseField.getText().length() == 0)
               && (m_searchBut.isEnabled())) {
                m_searchBut.setEnabled(false);}
            else
                if((m_phraseField.getText().length() != 0)
                   && (!m_searchBut.isEnabled())) {
                    m_searchBut.setEnabled(true);}}
    }

    /*
    ** process action event
    */
    public void actionPerformed(ActionEvent evt)
    {
        if(evt.getSource().equals(m_searchBut)) {
            // put together the bibtexrecord that contains the search pattern
            BibTeXRecord rec = makeSearchFilter();
            debug(rec.toString());
            // do search with the filter
            m_Results = doSearch(rec);
            // add the results to the vectors
            if(m_Results.length > 0) {
                updateHistory(m_Results);
                displayHistory();}
            else OKDialog.createOKDialog("The result of the latest search is empty", this);
            // output the results
            debug("number in results = " + m_Results.length);
        }
        if(evt.getSource().equals(m_cancelBut)) {
            // delete search history and null results
            setVisible(false);
            m_Results = null;
            m_historyVec.removeAllElements();}
        if(evt.getSource().equals(m_closeBut)) {
            // just close window, keep results
            setVisible(false);}
        if(evt.getSource().equals(m_resetBut)) {
            // reset button, set everything to first level
            resetSearch(0);}
    }

    /**
     Reset the search level to that passed.
     */
    private void resetSearch(int level) {
        // return if highest level chosen
        if(level == (m_listVec.size() - 1)) return;
        // save stuff that needs saving
        m_Results = (String[])m_listVec.firstElement();
        String temp = (String)m_historyVec.firstElement();
        // now pull the stuff back
        if(level > 0) {
            // set to passed level
            for(int i=m_listVec.size(); i>(level+1); i--) {
                // remove the elements
                m_listVec.removeElementAt(i-1);
                m_historyVec.removeElementAt(i-1);}
            // set the level
            m_Results = (String[])m_listVec.lastElement();}
        else {
            m_listVec.removeAllElements();
            m_listVec.addElement(m_Results);
            m_Results = null;
            m_historyVec.removeAllElements();
            m_historyVec.addElement(temp);}
        // update the display
        displayHistory();
    }

    /**
     Update the vectors with the search results.
     */
    private void updateHistory(String[] results) {
        // push the vector onto the stack of searches
        m_listVec.addElement(results);
        // create a string to add to the display
        StringBuffer sb = new StringBuffer("Query string: " + m_phraseField.getText() + m_EOLN);
        // set logic
        sb.append("Search logic: ");
        if(m_andCheck.getState()) sb.append("AND");
        else if(m_orCheck.getState()) sb.append("OR");
        else sb.append("EXACT");
        sb.append(m_EOLN);
        // case match
        sb.append("Match case: ");
        if(m_caseCheck.getState()) sb.append("TRUE");
        else sb.append("FALSE");
        sb.append(m_EOLN);
        // types included
        sb.append("Record types:");
        for(int i=0; i<m_typesCheck.length; i++) {
            if(m_typesCheck[i].getState())
                sb.append(" " + m_typesCheck[i].getLabel());}
        sb.append(m_EOLN);
        // fields to include
        sb.append("Fields included:");
        if(m_keyCheck.getState()) sb.append(" Citation Key");
        for(int i=0; i<m_fieldsCheck.length; i++) {
            if(m_fieldsCheck[i].getState())
                sb.append(" " + m_fieldsCheck[i].getLabel());}
        sb.append(m_EOLN);
        // results
        sb.append("Number of records returned by search: " + results.length + m_EOLN);
        // update the vector
        m_historyVec.addElement(sb.toString());
        // update level choice
//        m_levelChoice.addItem(String.valueOf(m_listVec.size() - 1));
    }

    /**
     Go through each element in the most current keys array, building
     a new array that satisfies the search criterion.
     */
    public String[] doSearch(BibTeXRecord filter) {
        // get array of keys
        String[] keys = (String[])m_listVec.lastElement();
        Vector newKeys = new Vector();
        // go through all the elements in the array
        for(int i=0; i<keys.length; i++) {
            // get the file position
            String[] data = (String[])m_Index.get(keys[i]);
            long pos = Long.valueOf(data[ListFrame.FILE_POS]).longValue();
            // advance the file
            try{
                m_RAFile.seek(pos);
                // read the record
                BibTeXRecord rec = new BibTeXRecord();
                rec.read(m_RAFile);
                // test the record
                boolean test = false;
                if(m_caseCheck.getState()) {
                    test = rec.contains(filter);}
                else {
                    test = rec.containsIgnoreCase(filter);}
                // include it if appropriate
                debug("test result = " + test + " for " + keys[i]);
                if(test) newKeys.addElement(keys[i]);}
            catch(IOException e) {
                debug(e.toString());}}
        // all done, create a string array for return
        String[] ret = new String[newKeys.size()];
        for(int i=0; i<ret.length; i++) {
            ret[i] = (String)newKeys.elementAt(i);}
        // return it
        return ret;
    }

    /**
     construct a bibtex record from the settings that reflects the
     desired search pattern.
     */
    public BibTeXRecord makeSearchFilter() {
        BibTeXRecord filter = new BibTeXRecord();
        StringTokenizer st = new StringTokenizer(m_phraseField.getText());
        StringBuffer query = new StringBuffer(st.nextToken());
        while(st.hasMoreTokens()) {
            // set separator according to type
            if(m_andCheck.getState()) {
                query.append("&" + st.nextToken());}
            if(m_orCheck.getState()) {
                query.append("|" + st.nextToken());}
            if(m_exactCheck.getState()) {
                query.append(" " + st.nextToken());}}
        String text = query.toString();
        // check for the key
        if(m_keyCheck.getState()) {
            filter.setKey(text);}
        // set the types
        StringBuffer types = new StringBuffer(" ");
        for(int i=0; i<m_typesCheck.length; i++) {
            if(m_typesCheck[i].getState()) {
                types.append(m_typesCheck[i].getLabel() + " ");}}
        filter.setType(types.toString());
        // set the fields
        for(int i=0; i<m_fieldsCheck.length; i++) {
            if(m_fieldsCheck[i].getState()) {
                BibTeXField bf = new BibTeXField(m_fieldsCheck[i].getLabel(),
                                                 text);
                filter.put(m_fieldsCheck[i].getLabel(), bf);}}
        return filter;
    }

    /*
     ** track the window events so we know who is in the forground
     */
    public void windowOpened(WindowEvent evt)
    {
    }

    public void windowClosing(WindowEvent evt)
    {
        if(m_Results != null) {
            if(!AskDialog.createAskDialog("Retain Search Results?",
                                         "Keep", "Loose", this)) {
                // loose results, set results to null
                setVisible(false);
                m_Results = null;
                m_historyVec.removeAllElements();}}
        // keep results, just hide window
        setVisible(false);
    }

    public void windowClosed(WindowEvent evt)
    {
    }

    public void windowIconified(WindowEvent evt)
    {
    }

    public void windowDeiconified(WindowEvent evt)
    {
    }

    public void windowActivated(WindowEvent evt)
    {
    }

    public void windowDeactivated(WindowEvent evt)
    {
    }

    /**
     Method to output a string when a debug flag is set
     */
    public static void debug(String message)
    {
        if(m_Debug) System.out.println("FindDialog:"+message);}

    /**
     Method to set the debug flag
     */
    public static void setDebug(boolean flag) {m_Debug = flag;}

}


        
