/*
 *   Copyright (C) 1997, 1998, 1999 Loic Dachary
 *
 *   This program is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#ifdef HAVE_SOCKS_H
#include <socks.h>
#endif /* HAVE_SOCKS_H */
#ifdef HAVE_DMALLOC_H
#include <dmalloc.h>
#endif /* HAVE_DMALLOC_H */

#include <hash.h>
#include <getopttools.h>
#include <uri.h>
#include <salloc.h>
#include <strshift.h>

#include <webtools.h>

#define file_state_save \
  file->buffer = buffer; \
  file->buffer_size = buffer_size; \
  file->buffer_length = code < 0 ? 0 : buffer_length;

#define file_state_restore \
  char* buffer = file->buffer; \
  int buffer_size = file->buffer_size; \
  int buffer_length = file->buffer_length;

static int verbose = 0;

static int webtools_reader_http_header(webtools_params_t* params, int sd, struct webtools_file* file);
static int webtools_reader_http_body(webtools_params_t* params, int sd, struct webtools_file* file);
static int webtools_open_1(webtools_params_t* params, struct in_addr server, short port);
static webtools_params_t* params_alloc();
static struct webtools_file* file(webtools_params_t* params, int sd);

static struct option long_options[] =
{
  /* These options set a flag. */
  {"verbose_webtools", 0, &verbose, 1},
  {"http_level", 1, 0, 0},
  {0, 0, 0, WEBTOOLS_OPTIONS}
};

struct option* webtools_options(struct option options[])
{
  return long_options;
}

webtools_params_t* webtools_alloc(int argc, char** argv, struct option options[])
{
  webtools_params_t* params = params_alloc();

  optind = 0;
  while(1) {
    /* `getopt_long' stores the option index here. */
    int option_index = 0;
    int c;
    int found = 1;

    c = getopt_long_only(argc, argv, "", options, &option_index);

    /* Detect the end of the options. */
    if (c == -1)
      break;
     
    switch (c)
      {
      case 0:
	/* If this option set a flag, do nothing else now. */
	
	if (options[option_index].flag != 0)
	  break;
	if(!strcmp(options[option_index].name, "http_level")) {
	  params->http_level = strdup(optarg);
	  break;
	} else if(!strcmp(options[option_index].name, "port")) {
	  break;
	}
	found = 0;
	break;
      default:
	fprintf(stderr, "option parse error %c, 0x%x\n", c & 0xff, c);
	exit(1);
      }
    if(found) {
      hash_alloc_insert(params->options, (char*)options[option_index].name, strdup(optarg ? optarg : " "));
    }
  }

  return params;
}

void webtools_free(webtools_params_t* params)
{
  hash_free(params->options);
  free(params);
}

int webtools_open(webtools_params_t* params, char* host, char* port)
{
  struct hostent* hostent;
  
  if((hostent = gethostbyname(host)) == 0) {
    if(verbose) fprintf(stderr, "webtools_open: could not get hostname IP for %s\n", host);
    errno = EADDRNOTAVAIL;
    return -1;
  }
  
  return webtools_open_1(params, *(struct in_addr*)hostent->h_addr_list[0], atoi(port));
}

int webtools_write(webtools_params_t* params, int sd, char* buffer, int size)
{
  return write(sd, buffer, size);
}

int webtools_reader(webtools_params_t* params, int sd)
{
  int code;
  struct webtools_file* f = file(params, sd);
  if(verbose) fprintf(stderr, "webtools_reader: now using 0x%x for fd = %d\n", (int)f, sd);
  switch(params->mode) {
  case WEBTOOLS_READER_HTTP_HEADER:
    code = webtools_reader_http_header(params, sd, f);
    break;
  case WEBTOOLS_READER_HTTP_BODY:
    code = webtools_reader_http_body(params, sd, f);
    break;
  default:
    fprintf(stderr, "webtools_reader: 0x%x reader unknown\n", params->mode);
    code = WEBTOOLS_READER_UNKNOWN;
    break;
  }
  return code;
}

void webtools_close(webtools_params_t* params, int sd)
{
  struct webtools_file* f = file(params, sd);
  if(f) {
    if(f->buffer) free(f->buffer);
    f->buffer_length = 0;
    f->buffer_size = 0;
  }
  close(sd);
}

#define BUFFER_SIZE 10240 

int read_timeout(int fd, char* buffer, int buffer_size, int timeout)
{
  fd_set readfds;
  struct timeval t;

  FD_ZERO(&readfds);
  FD_SET(fd, &readfds);

  t.tv_sec = timeout;
  t.tv_usec = 0;

  if(verbose) fprintf(stderr, "reading with %d timeout\n", timeout);

  {
    int ret = select(fd + 1, &readfds, 0, 0, &t);
    if(ret == 1) {
      return read(fd, buffer, buffer_size);
    } else if(ret == 0) {
      errno = ETIME;
      return -1;
    } else {
      return -1;
    }
  }
}

static int webtools_reader_http_header(webtools_params_t* params, int sd, struct webtools_file* file)
{
  file_state_restore
  char* p;
  int code = WEBTOOLS_READER_OK;
  int done = 0;

  params->callback_http_header(params->callback_arg, sd, buffer, 0, WEBTOOLS_READER_START);

  while(!done) {
    int bytes_read;
    static_alloc(&buffer, &buffer_size, buffer_length + BUFFER_SIZE + 1);
    p = buffer + buffer_length;
    if((bytes_read = read_timeout(sd, p, BUFFER_SIZE, params->timeout)) <= 0) {
      if(bytes_read == 0) {
	code = WEBTOOLS_READER_HTTP_HEADER_SHORT;
      } else {
	fprintf(stderr, "reading socket (header) : %d\n", errno);
	code = -1;
      }
      done = 1;
    } else {
      int i;
      if(verbose) fprintf(stderr, "webtools_reader_http_header: got %.*s\n", bytes_read, p);
      /*
       * Find header end if present.
       */
      for(i = 0; i < buffer_length + bytes_read - 1; i++) {
	if((buffer[i] == '\n' && buffer[i + 1] == '\n') ||
	   (i < buffer_length + bytes_read - 3 &&
	    (buffer[i] == '\r' && buffer[i + 1] == '\n' &&
	     buffer[i + 2] == '\r' && buffer[i + 3] == '\n'))) {
	  done = 1;
	  break;
	}
      }

      if(!done) {
	p += bytes_read;
	buffer_length += bytes_read;
      } else {
	int total_length = buffer_length + bytes_read;
	int remain;
	i += (buffer[i] == '\n' ? 2 : 4);
	remain = total_length - i;
	/*
	 * - 2 so that the null will override the last \n or \r instead
	 * of the first character of the following text.
	 */
	buffer_length = i - 2;
	buffer[buffer_length] = '\0';
	if(verbose) fprintf(stderr, "webtools_reader_http_header: %.*s\n", buffer_length, buffer);
	params->callback_http_header(params->callback_arg, sd, buffer, buffer_length, WEBTOOLS_READER_END);
	strshift(buffer, total_length, total_length - remain);
	buffer_length = remain;
      }
    }
  }

  file_state_save
  return code;
}

static int webtools_reader_http_body(webtools_params_t* params, int sd, struct webtools_file* file)
{
  file_state_restore
  char* p;
  int code = WEBTOOLS_READER_OK;
  int done = 0;

  if(verbose) fprintf(stderr, "buffer_length = %d\n", buffer_length);
  static_alloc(&buffer, &buffer_size, buffer_length + BUFFER_SIZE + 1);
  p = buffer + buffer_length;

  params->callback_http_body(params->callback_arg, sd, buffer, 0, WEBTOOLS_READER_START);

  while(!done) {
    int bytes_read;
    if((bytes_read = read_timeout(sd, p, BUFFER_SIZE, params->timeout)) <= 0) {
      params->callback_http_body(params->callback_arg, sd, buffer, buffer_length, WEBTOOLS_READER_END);
      if(bytes_read < 0) {
	fprintf(stderr, "reading socket (body) : %d\n", errno);
	code = -1;
      }
      done = 1;
    } else {
      buffer_length += bytes_read;
      if(verbose) fprintf(stderr, "buffer_length = %d\n", buffer_length);
      params->callback_http_body(params->callback_arg, sd, buffer, buffer_length, WEBTOOLS_READER_CONTINUE);
      buffer_length = 0;
      p = buffer;
    }
  }

  file_state_save
  return code;
}

static int webtools_open_1(webtools_params_t* params, struct in_addr server, short port)
{
  struct sockaddr_in sin;
  int sd;

  if ((sd = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP)) == -1) {
    fprintf(stderr, "could not open socket\n");
    perror("");
    exit(1);
  }

  memset(&sin, '\0', sizeof(sin));
  sin.sin_family = AF_INET ;
  memcpy((char*)&sin.sin_addr.s_addr, &server, sizeof(struct in_addr));
  sin.sin_port = htons(port);
  if(connect(sd,(struct sockaddr *) &sin, sizeof(sin)) == -1) {
    switch(errno) {
    case ECONNREFUSED:
    case ETIMEDOUT:
    case EHOSTDOWN:
    case EHOSTUNREACH:
      close(sd);
      return -1;
      break;
    }
    fprintf(stderr, "webtools_open_1: could not connect\n");
    perror("");
    exit(1);
  }

  if(verbose) fprintf(stderr, "connected\n");

  return sd;
}

static void hnode_free(hnode_t *node, void *context)
{
  free(node->data);
  free(node);
}

static webtools_params_t* params_alloc()
{
  webtools_params_t* params = (webtools_params_t*)smalloc(sizeof(webtools_params_t));
  memset((char*)params, '\0', sizeof(webtools_params_t));
  params->options = hash_create(33, 0, 0);
  hash_set_allocator(params->options, 0, hnode_free, 0);

  return params;
}

static struct webtools_file* file(webtools_params_t* params, int sd)
{
  if(sd < WEBTOOLS_MAX_FD) {
    return &(params->files[sd]);
  } else {
    fprintf(stderr, "webtools_file: descriptor too high %d (max %d)\n", sd, WEBTOOLS_MAX_FD);
    return 0;
  }
}

