/* Source Installer, Copyright (c) 2005 Claudio Fontana

   package_info.c - package information structure and functions

   This program is free software; you can redistribute it and/or modify
       it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
       (at your option) any later version.

   This program is distributed in the hope that it will be useful,
       but WITHOUT ANY WARRANTY; without even the implied warranty of
       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
       GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
       along with this program (look for the file called COPYING);
       if not, write to the Free Software Foundation, Inc.,
           51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

       You can contact the author (Claudio Fontana) by sending a mail
       to claudio@gnu.org
*/

#include "src_stdinc.h"
#include "package_info.h"

struct srcinst_parsed_data _keywords_package_info[SRCINST_PACKAGE_INFO_NKEYS] = {
  { "BUILD_SUBDIR", 0, "%s" },
  { "CONFIGURED", 0, "%s" },
  { "D", 0, "%s" },
  { "DESCRIPTION", 0, "%s" },
  { "I", 0, "%s" },
  { "INSTALLED", 0, "%d" },
  { "INSTALLED_FILES", 0, "%%" },
  { "INSTALLED_SIZE", 0, OFF_FMT },
  { "LONG_DESCRIPTION", 0, "%%" },
  { "NAME", 0, "%%" },
  { "PREFIX", 0, "%s" },
  { "SOURCE_AVAILABLE", 0, "%%" },
  { "SOURCE_LOCATION", 0, "%s" },
  { "SOURCE_SIZE", 0, OFF_FMT }
};

static void _init_keywords_package_info(struct _package_info* info) {
  struct srcinst_parsed_data* data = _keywords_package_info;
  data[SRCINST_IKEY_BUILD_SUBDIR].variable = &(info->build_subdir);
  data[SRCINST_IKEY_CONFIGURED].variable = &(info->configured);
  data[SRCINST_IKEY_D].variable = &(info->long_description);
  data[SRCINST_IKEY_DESCR].variable = &(info->description);
  data[SRCINST_IKEY_I].variable = &(info->installed_files);
  data[SRCINST_IKEY_INSTALLED].variable = &(info->installed);
  data[SRCINST_IKEY_INSTALLED_FILES].variable = 0;
  data[SRCINST_IKEY_INSTALLED_SIZE].variable = &(info->installed_size);
  data[SRCINST_IKEY_LONG_DESCR].variable = 0;
  data[SRCINST_IKEY_NAME].variable = 0;
  data[SRCINST_IKEY_PREFIX].variable = &(info->prefix);
  data[SRCINST_IKEY_SOURCE_AVAILABLE].variable = 0;
  data[SRCINST_IKEY_SOURCE_LOCATION].variable = &(info->source_location);
  data[SRCINST_IKEY_SOURCE_SIZE].variable = &(info->source_size);
}

void _init_package_info(struct _package_info* info, char* n) {
  memset(info, 0, sizeof(struct _package_info));
  info->source_size = info->installed_size = (off_t)-1;
  info->name = n ? srcinst_strdup(n) : 0;
}

void _free_package_info(struct _package_info* i) {
  if (i->name) { free(i->name); }
  if (i->packdir) { free(i->packdir); }
  if (i->description) { free(i->description); }
  if (i->long_description) { free(i->long_description); }
  if (i->configured) { free(i->configured); }
  if (i->prefix) { free(i->prefix); }
  if (i->source_location) { free(i->source_location); }
  if (i->build_subdir) { free(i->build_subdir); }
  srcinst_free_string_list(&(i->installed_files));
  memset(i, 0, sizeof(struct _package_info));
}

int _parse_package_info(struct _package_info* info, char* k, char* s) {
  int i;

  /* find matching keyword entry */
    
  if ((i = srcinst_bsearch(_keywords_package_info, SRCINST_PACKAGE_INFO_NKEYS, k, &srcinst_get_parsed_data_key)) == -1)
    return 0;

  switch (i) {
  case SRCINST_IKEY_D:
      
    if (info->long_description) {
      info->long_description = srcinst_append(info->long_description, "\n");
    }
    info->long_description = srcinst_append(info->long_description, s);
    break;
    
  case SRCINST_IKEY_I:
    srcinst_add_string_list(&(info->installed_files), s);
    break;

  case SRCINST_IKEY_INSTALLED:
  
    *((char*)_keywords_package_info[i].variable) =
      strcmp(s, "yes") == 0 || strcmp(s, "YES") == 0 || strcmp(s, "1") == 0;
    break;
  
  default:
    /* call generic parsing method for the other cases */
    return srcinst_parse(_keywords_package_info + i, s);
  }

  return 1;
}

int _load_package_info(struct _package_info* info) {
  int rv; char* filename;

  _init_keywords_package_info(info);

  filename = srcinst_fnjoin(srcinst_get_dir(SRCINST_DIR_PACK), info->name);
  rv = srcinst_load_parse_file(info, filename, &_parse_package_info);
  free(filename);
  
  if (!rv)
    return 0;

  /* now validate derived package information */

  if (info->source_location) {
    if (srcinst_file_type(info->source_location) != SRCINST_TYPE_FILE) {
      srcinst_warning(SRCINST_ERR_PARSER, "SOURCE_LOCATION points to an unreachable file in package", info->name);
      free(info->source_location); info->source_location = 0;
    }
  }

  info->loaded = 1;
  return 1;
}

int _save_package_info(struct _package_info* info) {
  char* filename; FILE* f; int i;
  _init_keywords_package_info(info);

  filename = srcinst_fnjoin(srcinst_get_dir(SRCINST_DIR_PACK), info->name);
  
  f = fopen(filename, "w");
  free(filename);

  if (!f)
    return 0;

  for (i = 0; i < SRCINST_PACKAGE_INFO_NKEYS; i++) {
    struct srcinst_parsed_data* data = _keywords_package_info + i;

    switch (i) {
    case SRCINST_IKEY_I:
      if (info->installed && info->installed_files.count > 0) {
	int j; struct srcinst_string_list* l; l = &(info->installed_files);
	fprintf(f, "%s: \n", "INSTALLED_FILES");
      
	for (j = 0; j < l->count; j++) {
	  fprintf(f, "i: %s\n", l->strings[j]);
	}
      }
      break;

    case SRCINST_IKEY_D:
      if (info->long_description) {
	char* end, *start;
	start = info->long_description;
	
	fprintf(f, "%s: \n", "LONG_DESCRIPTION");
	
	while ((end = strchr(start, '\n'))) {
	  *end = '\0'; fprintf(f, "d: %s\n", start); *end = '\n';
	  start = end + 1;
	}
	
	fprintf(f, "d: %s\n", start);
      }
      break;

    default:

      switch (data->format[1]) {

      case '%':
	/* nop */
	break;

      case 's':
	if (*((char**)data->variable))
	  fprintf(f, "%s: %s\n", data->key, *((char**)(data->variable)));
	break;

      case 'd':
	fprintf(f, "%s: %d\n", data->key, *((char*)(data->variable)));
	break;

      default:
	/* OFF_FMT */
	{
	  off_t value; value = *((off_t*)data->variable);
	  if (value != (off_t)-1)
	    fprintf(f, "%s: " OFF_FMT "\n", data->key, value);
	}
      }
    }
  }
  
  i = ferror(f);
  fclose(f);
  return !i;
}

int _unlink_package_info(struct _package_info* info) {
  int rv; char* fn;

  rv = 0;
  fn = srcinst_fnjoin(_srcinst_state.dirs[SRCINST_DIR_PACK], info->name);
  
  if (srcinst_file_type(fn) != SRCINST_TYPE_FILE) {
    /* there is no file to remove. Package is not persistant. */
    rv = 1;
    goto end_proc;
  }
  
  if (!srcinst_spawn_wait(PRG_rm, "-f", fn, 0)) {
    srcinst_warning(SRCINST_ERR_CORE, "could not delete package information scheduled for removal.", fn);
    goto end_proc;
  }
  
  rv = 1;
 end_proc:
  free(fn);
  return rv;
}
    
