########### smtp.tcl
#
# Main SMTP protocol implementation
#
# This file is part of SAUCE, a very picky anti-spam receiver-SMTP.
# SAUCE is Copyright (C) 1997-1999 Ian Jackson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
#
# $Id: smtp.tcl,v 1.1 1999/10/09 15:46:15 ian Exp $


########## connection threads
# thread_start $chan $desc $remoteaddr $remoteport
#
# errors/results ignored
#
# state variables:
# chan              incoming SMTP connection
# ra                calling IP address
# rp                calling port
# rh                calling hostname (or IP-literal), available only after HELO
# cmdomain          calling host's mail domain to report to
# la                called (local) address
# lh                called (local) canonical hostname for interface
# lp                called (local) port
# ident             result of ident lookup (whether informative or error)
# dnsid             thread for forward DNS lookups (unset => none)
# dnsptrid          thread for reverse DNS lookup (unset => none)
# ichan             reverse ident lookup connection (unset => none)
# itoid             timeout for ident lookup or incoming command (unset => none)
# mtachan           channel to local MTA
# avfid             ID of current address verification thread (unset => none)
# quitting          set => we have been asked to shut down
# smtperrors        count of SMTP error responses
#
# state variables to do with incoming SMTP state machine:
# helostring        set => we have had helo/ehlo, and this was what they said
#                   this implies mtachan being open and connected
# helocmd           the command helo or ehlo that they used
# smtpstyle         `smtp' or `esmtp'.
# mf_lp             set => we have had MAIL FROM, and this is the local part
# mf_dm             domain from MAIL FROM, may be garbage or unset if mf_lp not set
# no                set => we are in one of the `NO' states from the overall
#                   state machine, and want to reject the incoming mail
#                   (this means states RN, MN, and associated incoming data areas
#                   of the flowchart)
#                   If add_bl is set then we hide this reason from the caller;
#                   otherwise we show it.
# defer             set => we want to defer this message due to lack of trust
#                   (no takes precedence).  This string is the reason string.
#
# whyreject         reason we are rejecting (if not in message)
# add_bl            `add-bl': we want to add something to the blacklist
# a_admin           `a-a': we have had an admin recipient and said 250
# a_normal          `a-n': we have had a normal recipient and said 250
# a_bypass          `a-b': we have had a bypass-check recipient and said 250
# t_w               `t-w': recipient addresses that we want to add to the
#                    whitelist if we accept the message (checked originators)
# lastchal          last challenge sent on this conn for SAUCEADMIN
# smtpcmd           SMTP command we are processing, or empty string
# att_rcpts         list of recipients they've asked for (perhaps not got)
#
# Recipient deferral:
# We can defer if we get mail from unknown senders or unknown hosts.
# The deferral period can be increased if the sending host is RBLd.
# So: both sender and host on whitelist => no deferral, no RBL lookup.
# Otherwise, do RBL lookup.  Sending host RBLd for reject => reject.
# Otherwise, defer until BOTH
#  first contact from this site was at least minimum site deferral ago
#  first contact from this sender was at least minimum sender deferral ago
# (minimum is calculated across all applicable RBLs, and new
# site or sender deferral).
# 
# mf_parms          parameters to MAIL FROM, if any
# mf_message        proposed success message for MAIL FROM
# rblids            RBL lookups in progress (or empty list)
# minsiteage        minimum time (in seconds) since first contact with site, or we defer
# minaddrage        minimum time (in seconds) since first contact with addr, or we defer
# mf_lp, mf_dm      set

thread_typedefine ic {chan lalhlp ra rp} {
    global ident_port ident_timeout chan_desc
    set state(chan) $chan
    set state(ra) $ra
    set state(rp) $rp
    set state(smtpcmd) {}
    set state(smtpreaderr) {}
    set state(whyreject) {}
    set state(smtperrors) 0
    set state(att_rcpts) {}
    set state(rblids) {}
    manyset $lalhlp state(la) state(lh) state(lp)
    set state(dnsptrid) [thread_start dnsptr "$state(desc) / reverse lookup" $ra]
    if {[catch {
	set state(ichan) [socket -myaddr $state(la) -async $state(ra) $ident_port]
	chanset_desc $state(ichan) "$state(desc) / ident"
    } emsg]} {
	log notice "ident error connecting to $ra: $emsg"
	ic_ident_done {}
    } else {
	fconfigure $state(ichan) -translation {binary crlf} -blocking false
	set state(itoid) [thread_after ic $id $ident_timeout ident_timeout]
	thread_fileevent ic $id $state(ichan) writable ident_connected
    }
} {
    global canonical_hostname
    set state(quitting) 1
    if {![info exists state(header)] && ![info exists state(sofar)]} {
	ic_commandfinalresponse immed "421 $canonical_hostname shutting down"
    }
} {
    foreach thr $state(rblids) { catch { thread_cancel dns $thr } }
    set state(rblids) {}
    catch { fileevent $state(chan) readable {} }
    catch { fileevent $state(chan) writable {} }
    catch_close_cleardesc state(ichan)
    catch { thread_cancel dnsptr $state(dnsptrid) }
    catch { thread_cancel avf $state(avfid) }
    catch { after cancel $state(itoid) }
    catch { unset state(itoid) }
    catch { after cancel $state(ptoid) }
    catch { unset state(ptoid) }
    catch_close_cleardesc state(mtachan)
}

thread_chainproc ic ident_timeout {} {
    catch_close_cleardesc state(ichan)
    unset state(itoid)
    log notice "ident timeout on $state(ra)"
    ic_ident_done {}
}

thread_chainproc ic ident_connected {} {
    threadio_putsgets ic $id $state(ichan) "$state(rp) , $state(lp)\n" \
	    ident_rx_ok ident_rx_err
}

thread_chainproc ic ident_rx_ok {data} {
    after cancel $state(itoid)
    unset state(itoid)
    set eof [eof $state(ichan)]
    catch_close_cleardesc state(ichan)
    regexp {^.*} $data data
    if {$eof} {
	log notice "ident eof on $state(ra)"
	ic_ident_done {}
    } elseif {[regexp -nocase {^[ \t]*[0-9]+[ \t]*,[ \t]*[0-9]+[ \t]*:[ \t]*userid[ \t]*:[^:]*:([^:]*)$} $data all userid]} {
	ic_ident_done [string trim $userid]
    } elseif {[regexp -nocase {^[ \t]*[0-9]+[ \t]*,[ \t]*[0-9]+[ \t]*:[ \t]*error[ \t]*:(.*)$} $data all error]} {
	log notice "ident remote error on $state(ra): [string trim $error]"
	ic_ident_done {}
    }
}

thread_chainproc ic ident_rx_err {emsg} {
    log debug "ident failed on $state(ra): $emsg"
    ic_ident_done {}
}

thread_subproc ic ident_done {ident} {
    catch { after cancel $state(itoid) }
    catch { unset state(itoid) }
    set state(ident) $ident
    thread_join ic $id dnsptr $state(dnsptrid) remotedns_ok remotedns_err
}

thread_chainproc ic remotedns_ok {answers} {
    unset state(dnsptrid)
    if {[llength $answers]} {
	ic_remotedns_done [join $answers]
    } else {
	ic_remotedns_done "\[$state(ra)\]"
    }
}

thread_chainproc ic remotedns_err {emsg} {
    global require_reverse_dns canonical_hostname
    if {$require_reverse_dns} {
	ic_commandfinalresponse major \
		"421 $canonical_hostname $state(ra): reverse DNS: $emsg"
    } else {
	ic_remotedns_done "\[$state(ra)\]"
    }
}

thread_subproc ic remotedns_done {dnsresult} {
    global canonical_hostname
    set state(rh) $dnsresult
    ic_commandresponse greeting "220 $canonical_hostname sauce-smtpd ESMTP ready"
}

thread_subproc ic commandresponse {evtype response} {
    ic_commandresponse_maybefinal $evtype 0 $response
}

proc intern_getsiteannoy {ra change} {
    global annoy_halflife annoy_grudge_max annoy_love_max
    global annoy_grumpy annoy_actout_max local_interface

    if {[info exists local_interface($ra)]} { return {0 Submissive} }

    set ca [ds_get site-annoy $ra]
    set now [clock seconds]
    if {"$ca" == "unknown"} {
	set cv 0
    } else {
	manyset [string map {a { } m -} $ca] ct cv
	set newcv [expr {
	    round( floor(
	    $cv * pow( 0.5, double($now-$ct)/$annoy_halflife )
	    ))
	}]
	debug 2 cv=$cv now=$now ct=$ct hl=$annoy_halflife newcv=$newcv
	set cv $newcv
    }
    incr cv $change
    if {$cv > $annoy_grudge_max} { set cv $annoy_grudge_max }
    if {$cv < -$annoy_love_max} { set cv -$annoy_love_max }
    ds_set site-annoy $ra \
	    [string map {{ } a - m} [list $now $cv]] \
	    [expr {$now + 3*$annoy_halflife}]
    if {$cv <= -$annoy_love_max/2 && $cv <= -($annoy_grumpy+$annoy_actout_max)} {
	set irritamt Ecstatic
    } elseif {$cv <= 0} {
	set irritamt Pleased
    } elseif {$cv <= $annoy_grumpy} {
	set irritamt Irritated
    } elseif {$cv <= $annoy_grumpy+$annoy_actout_max} {
	set irritamt Angry
    } else {
	set irritamt Furious
    }
    return [list $cv $irritamt]
}

thread_subproc ic getsiteannoy {change} {
    return [intern_getsiteannoy $state(ra) $change]
}

thread_subproc ic commandresponse_maybefinal {evtype final response} {
    global max_smtp_errors canonical_hostname
    global annoy_actout_max annoy_grumpy annoy_partrespevery annoy_actout_nopartresp
    global pleasure_command pleasure_delivery annoyance_major annoyance_minor

    if {"$evtype" != "immed" && [ic_check_quitting]} { return }

    switch -exact $evtype {
	command - delivery { set annoychange -[set pleasure_$evtype] }
	nopartresp { set annoychange -$pleasure_command }
	major - minor { set annoychange [set annoyance_$evtype] }
	rcpt-defer { set annoychange $annoyance_minor }
	immed - greeting { set annoychange 0 }
	default { error "$evtype ?" }
    }
    manyset [ic_getsiteannoy $annoychange] cv irritamt
    
    switch -exact $evtype {
	major - minor - rcpt-defer {
	    set delay $cv
	    incr state(smtperrors)

	    if {!$final && $state(smtperrors) > $max_smtp_errors} {
		log notice "too many errors from $state(rh), closing channel (annoy=$cv)"
		ic_commandfinalresponse major \
			"421 $canonical_hostname $response \[too many errors\]"
		return
	    }
	}
	command { set delay [expr {$cv - $annoy_grumpy}] }
	nopartresp - greeting {
	    set delay [expr {$cv - $annoy_grumpy}]
	    if {$delay > $annoy_actout_nopartresp} {
		set delay $annoy_actout_nopartresp
	    }
	}
	delivery - immed { set delay 0 }
	default { error "$evtype ?" }
    }
    if {$delay > $annoy_actout_max} { set delay $annoy_actout_max }
    switch -exact $evtype {
	major {
	    logreject reject state command $response [string tolower $irritamt] ${cv}ms
	}
	minor {
	    if {$delay > 0} {
		logreject notice state delay $response [string tolower $irritamt] ${cv}ms
	    }
	}
	rcpt-defer {
	    logreject notice state rcpt-defer {} [string tolower $irritamt] ${cv}ms
	}
	command - delivery - nopartresp - greeting - immed { }
	default { error "$evtype ?" }
    }

    if {[string length $response] && ($delay > 0 || "$evtype" == "greeting")} {
	regsub {(?m)$} $response " \[$irritamt\]" response
    }
    ic_commandresponsedelay $delay [expr {"$evtype" == "nopartresp"}] $final $response
}

thread_subproc ic commandresponsedelay {delay nopartresp final response} {
    global command_timeout annoy_partrespevery

    if {!$final && [ic_check_quitting]} { return }

    if {!$nopartresp && $delay > $annoy_partrespevery && \
	    [regsub {(?m)^([0-9][0-9][0-9])\s} $response {\1-} partresponse]} {
        incr delay -$annoy_partrespevery
	set state(ptoid) [thread_after ic $id $annoy_partrespevery \
		commandresponsedelay_part $delay $final $response $partresponse]
	return
    } elseif {$delay > 0} {
	set state(ptoid) [thread_after ic $id $delay \
		commandresponsedelay_after 0 1 $final $response]
	return
    }
	
    chanset_hide $state(chan) 1 1
    if {[string length $response]} {
	append response "\n"
    }
    set state(smtpcmd) {}
    set state(smtpreaderr) {}
    if {$final} {
	chanset_hide $state(chan) 1 1
	threadio_puts ic $id $state(chan) $response tellquit_done tellquit_done
    } else {
	set state(itoid) [thread_after ic $id $command_timeout timedout]
	threadio_putsgets ic $id $state(chan) $response command_ok command_err
    }
}

thread_chainproc ic commandresponsedelay_part {delay final response partresponse} {
    unset state(ptoid)
    threadio_puts ic $id $state(chan) "$partresponse\n" \
	    commandresponsedelay_ok commandresponsedelay_err $delay 0 $final $response
}

thread_chainproc ic commandresponsedelay_err {delay nopartresp final response emsg} {
    ic_command_err $emsg
}

thread_chainproc ic commandresponsedelay_ok {delay nopartresp final response} {
    ic_commandresponsedelay $delay $nopartresp $final $response
}

thread_chainproc ic commandresponsedelay_after {delay nopartresp final response} {
    unset state(ptoid)
    ic_commandresponsedelay $delay $nopartresp $final $response
}

thread_chainproc ic tellquit_done {args} {
    thread_finish ic $id
}

thread_subproc ic commandfinalresponse {evtype message} {
    if {[info exists state(mtachan)]} {
	threadio_commandresponse ic $id $state(mtachan) quit \
		{} mtaquit_done mtaquit_done $evtype $message
    } else {
	ic_commandresponse_maybefinal $evtype 1 $message
    }
}

thread_chainproc ic mtaquit_done {evtype message args} {
    catch_close_cleardesc state(mtachan)
    ic_commandresponse_maybefinal $evtype 1 $message
}

thread_chainproc ic timedout {} {
    global canonical_hostname

    fileevent $state(chan) readable {}
    ic_commandfinalresponse minor \
	    "421 $canonical_hostname Timed out waiting for command"
}

thread_chainproc ic command_err {emsg} {
    ic_command_err $emsg
}

thread_subproc ic command_err {emsg} {
    global annoyance_minor
    
    manyset [ic_getsiteannoy $annoyance_minor] cv irritamt
    set state(smtpreaderr) $emsg
    logreject notice state dropped {} [string tolower $irritamt] ${cv}ms
    thread_finish ic $id
}

thread_subproc ic commandnorhs {rhs} {
    if {[string length $rhs]} {
	ic_commandresponse major "501 No parameters allowed"
	return -code return
    }
}

thread_subproc ic check_quitting {} {
    global canonical_hostname
    if {![info exists state(quitting)]} { return 0 }
    ic_commandfinalresponse immed "421 $canonical_hostname Shutting down"
    return 1
}

thread_chainproc ic command_ok {cmd} {
    global canonical_hostname blacklist_message bland_message admin_chal_timeout
    global adminsecret blacksite_message allow_saucestate
    after cancel $state(itoid)
    unset state(itoid)
    regexp {^.*} $cmd cmd
    set state(smtpcmd) $cmd
    set state(smtpreaderr) {}
    if {![string length $cmd]} { set state(smtpreaderr) {Empty command} }
    set state(whyreject) {}
    if {[ic_check_quitting]} {
	return
    } elseif {[eof $state(chan)]} {
	set state(smtpreaderr) EOF
	ic_commandfinalresponse major ""
	return
    } elseif {![regexp -nocase -- {^([a-z0-9]+)[ \t]*(.*)$} $cmd all verb rhs]} {
	ic_commandresponse major "500 Syntax error"
	return
    } else {
	set verb [string tolower $verb]
	switch -exact -- $verb {
	    quit {
		ic_commandnorhs $rhs
		ic_commandfinalresponse command "221 $canonical_hostname goodbye"
	    }
	    helo {
		ic_helo helo smtp $rhs
	    }
	    ehlo {
		ic_helo ehlo esmtp $rhs
	    }
	    mail {
		if {![info exists state(helostring)]} {
		    ic_commandresponse major "503 need HELO or EHLO before MAIL"
		} elseif {[info exists state(mf_lp)]} {
		    ic_commandresponse major "503 MAIL already issued"
		} elseif {[regexp -nocase \
			{^from:[ \t]*<(.+)@([^@]+)>[ \t]*(.*)$} \
			$rhs all lp dm parms]} {
		    ic_msg_resetvars
		    set state(mf_lp) $lp
		    set state(mf_dm) $dm
		    set state(mf_parms) $parms
		    if {[regexp {^\[.*\]$} $state(mf_dm)]} {
			ic_mailfrom_fail "550 Domain-literal senders not allowed"
		    } elseif {[catch { address_dequote state(mf_lp) state(mf_dm) } \
			    emsg]} {
			ic_mailfrom_fail "501 Syntax error in sender ($emsg)"
		    } else {
			set str "$state(mf_lp)@$state(mf_dm)"
			set as [ds_get addr-list $str]
			set ss [ds_get site-list $state(ra)]

			if {"$as" == "white" && "$ss" == "white"} {
			    set state(mf_message) "You are on the whitelist"
			    ic_mailfrom_ok
			} elseif {"$as" == "black"} {
			    set state(mf_message) "You are on the blacklist"
			    set state(no) "Blacklisted sender `$str'"
			    ic_mailfrom_ok
			} elseif {"$as" == "unknown"} {
			    set state(avfid) [thread_start avf \
				    "$state(desc) / verify $str" \
				    $state(mf_lp) $state(mf_dm)]
			    thread_join ic $id avf $state(avfid) \
				    mailfrom_avf_ok mailfrom_avf_err
			} elseif {"$as" == "verified"} {
			    set state(mf_message) "You were verified previously"
			    ic_rbl
			} else {
			    set state(mf_message) "You are on the greylist"
			    ic_rbl
			}
		    }
		} elseif {[regexp -nocase \
			{^from:[ \t]*<>[ \t]*(.*)$} \
			$rhs all parms]} {
		    ic_msg_resetvars
		    set state(mf_lp) {}
		    set state(mf_dm) {}
		    set state(mf_parms) $parms
		    set ss [ds_get site-list $state(ra)]
		    if {"$ss" == "white"} {
			set state(mf_message) "Bounce is from whitelisted site"
			ic_mailfrom_ok
		    } else {
			set state(mf_message) "Ready to receive a bounce"
			ic_rbl
		    }
		} else {
		    ic_commandresponse major "501 Syntax error in parameter to MAIL"
		}
	    }
	    vrfy {
		ic_commandresponse command "252 VRFY not supported by SAUCE."
	    }
	    rcpt {
		if {![info exists state(mf_lp)]} {
		    ic_commandresponse minor "503 need MAIL before RCPT"
		} elseif {[regexp -nocase -- \
			{^to:[ \t]*<(.+)@([^@]+)>[ \t]*$} \
			$rhs all lp dm]} {
		    if {[catch { address_dequote lp dm } emsg]} {
			ic_commandresponse major "501 Syntax error in recipient ($emsg)"
		    } else {
			set str "$lp@$dm"
			set rtcmd "rcpt to:<[lp_quote $lp]@$dm>"
			set atype [addr_classify $lp $dm]
			lappend state(att_rcpts) [list $atype $lp@$dm]
			if {"$atype" == "admin"} {
			    set state(a_admin) 1
			    threadio_commandresponse ic $id $state(mtachan) \
				    $rtcmd {^250} mta_rcptadmin_ok {}
			} elseif {"$atype" == "bait"} {
                            set str "Sent mail to bait address $lp@$dm"
			    set state(add_bl) $str
			    set state(no) $str
			    set state(a_normal) 1
			    ic_commandresponse command "250 $bland_message"
			} elseif {[info exists state(no)]} {
			    if {[info exists state(add_bl)]} {
				set state(a_normal) 1
				ic_commandresponse command "250 $bland_message"
			    } else {
				ic_reject_rcpt $str
			    }
			} elseif {[info exists state(defer)]} {
			    ic_commandresponse rcpt-defer "450 $state(defer)"
			} elseif {"[ds_get site-list $state(ra)]" == "black"} {
			    set state(no) "Blacklisted site \[$state(ra)\]"
			    ic_reject_rcpt $str
			} else {
			    threadio_commandresponse ic $id $state(mtachan) \
				    $rtcmd {} mta_rcpt_ok {} $atype
			}
		    }
		} else {
		    ic_commandresponse major "501 Syntax error in parameter to RCPT"
		}
	    }
	    data {
		ic_commandnorhs $rhs
		if {![info exists state(a_admin)] &&
		    ![info exists state(a_normal)] &&
		    ![info exists state(a_bypass)]} {
		    ic_commandresponse minor "503 No recipients specified"
		} else {
		    threadio_puts ic $id $state(chan) \
			    "354 Send text\n" askfordata_done command_err
		}
	    }
	    sauceadmin {
		if {![string length $rhs]} {
		    set chal [exec -keepnewline \
			    dd if=/dev/urandom bs=1 count=8 2>/dev/null]
		    binary scan $chal H* chal
		    if {[string length $chal] != 16} { error "urandom failed `$chal'" }
		    append chal [format %08lx [clock seconds]]
		    set state(lastchal) $chal
		    ic_commandresponse immed "393 $chal"
		} elseif {![info exists state(lastchal)]} {
		    ic_commandresponse major "503 Need SAUCEADMIN on its own first"
		} else {
		    set waschal $state(lastchal)
		    log notice "$state(desc): ATTEMPTING SWITCH TO ADMIN MODE"
		    if {![regexp {^([0-9a-f]{16})([0-9a-f]{8})[ \t]+([0-9a-f]{32})$} \
			    $rhs all chal wasdate resp]} {
			ic_commandresponse major "501 \\x{24} \\x{32} please"
		    } elseif {"$chal$wasdate" != "$waschal"} {
			ic_commandresponse immed "490 challenge overwritten"
		    } elseif "[clock seconds] - 0x$wasdate > $admin_chal_timeout" {
			ic_commandresponse immed "491 challenge timed out"
		    } elseif {![string length $adminsecret]} {
			ic_commandresponse immed "495 admin secret missing"
		    } elseif {"$resp" != \
 "[exec <<"[binary format H* $waschal]$adminsecret" md5sum]"} {
                        ic_commandresponse immed "492 incorrect response"
                        unset state(lastchal)
                    } else {
			log notice "$state(desc): switch to admin mode ok"
			threadio_puts ic $id $state(chan) "294 yes master\n" \
				yesmaster_outdone command_err
		    }
		}
	    }
	    saucestate {
		if {$allow_saucestate} {
		    set op "100-\n"
		    foreach x [lsort [array names state]] {
			append op "100-[list $x $state($x)]\n"
		    }
		    append op "100"
		    ic_commandresponse immed $op
		} else {
		    ic_commandresponse immed "504 SAUCESTATE not available."
		}
	    }
	    help {
		ic_commandnorhs $rhs
		ic_commandresponse command \
{214-
214 QUIT HELP NOOP HELO EHLO MAIL RCPT DATA QUIT RSET VRFY}
	    }
	    noop {
		ic_commandnorhs $rhs
		ic_commandresponse command "250 NOOP OK"
	    }
	    rset {
		ic_commandnorhs $rhs
		if {[info exists state(mtachan)]} {
		    threadio_commandresponse ic $id $state(mtachan) rset \
			    {^2[0-9][0-9]} mta_rset_ok {}
		} else {
		    catch { unset state(mf_lp) }
		    ic_commandresponse command "250 OK"
		}
	    }
	    default {
		ic_commandresponse major "502 Command unrecognised"
	    }
	}
    }
}

thread_subproc ic helo {helocmd smtpstyle rhs} {
    global forbid_helo_ipliteral require_reverse_dns canonical_hostname
    set state(helocmd) $helocmd
    set state(smtpstyle) $smtpstyle
    if {[info exists state(helostring)]} {
	ic_commandresponse major "503 HELO or EHLO already specified"
    } elseif {[regexp {^\[(\d+\.\d+\.\d+\.\d+)\]$} $rhs all ipliteral]} {
	if {$forbid_helo_ipliteral} {
	    ic_commandresponse major \
		    "504 IP literal ($rhs) in HELO forbidden by adminstrator"
	} else {
	    ic_find_maildomain $state(rh) $rhs
	}
    } elseif {![domain_ok $rhs]} {
	ic_commandresponse major "501 Syntax error in HELO domain"
    } else {
	if {"[string tolower $rhs]" == "[string tolower $state(rh)]"} {
	    ic_find_maildomain $rhs $rhs
	} elseif {"$state(ra)" == "127.0.0.1"} {
	    ic_set_maildomain $canonical_hostname $rhs
	} elseif {"[ds_get site-list $state(ra)]" == "white"} {
	    ic_set_maildomain "\[$state(ra)\]" $rhs
	} else {
	    set state(dnsid) [thread_start dns "$state(desc) / HELO lookup" $rhs A]
	    thread_join ic $id dns $state(dnsid) helodns_ok helodns_err $rhs
	}
    }
}

thread_chainproc ic helodns_ok {hs answers errors how} {
    global check_helo_name
    unset state(dnsid)
    if {[llength $answers]} {
	if {[lsearch -exact $answers $state(ra)] != -1} {
	    ic_find_maildomain $hs $hs
	} elseif {$check_helo_name} {
	    ic_commandresponse major \
		    "504 HELO name $hs has no address matching $state(ra)"
	} else {
	    ic_find_maildomain $state(rh) $hs
	}
    } elseif {$check_helo_name} {
	ic_commandresponse major \
		"504 HELO name $hs does not map to any addresses: $errors"
    } else {
	ic_find_maildomain $state(rh) $hs
    }
}

thread_chainproc ic helodns_err {hs emsg} {
    unset state(dnsid)
    ic_commandresponse major "450 HELO name $hs lookup failed: $emsg"
}

thread_subproc ic find_maildomain {chstart hs} {
    global require_callingmaildomain_name
    if {![string match {\[*\]} $chstart]} {
	ic_findmore_maildomain $chstart $chstart $hs
    } elseif {$require_callingmaildomain_name} {
	ic_commandresponse major \
		"504 Cannot find $state(ra) host name via reverse DNS or HELO"
    } else {
	ic_set_maildomain $chstart $hs
    }
}

thread_subproc ic findmore_maildomain {chstart chnow hs} {
    if {[llength [split $chnow .]] == 1} {
	ic_set_maildomain $chstart $hs
    } else {
	set state(dnsid) [thread_start dns "$state(desc) / maildomain lookup" $chnow MX]
	thread_join ic $id dns $state(dnsid) fch_ok fch_err $chstart $chnow $hs
    }
}

thread_chainproc ic fch_ok {chstart chnow hs answers errors how} {
    unset state(dnsid)
    if {[llength $answers]} {
	ic_set_maildomain $chnow $hs
    } else {
	regsub {^[^.]+\.} $chnow {} chnow
	ic_findmore_maildomain $chstart $chnow $hs
    }
}

thread_chainproc ic fch_err {chstart chnow hs emsg} {
    global require_callingmaildomain_dnsok
    unset state(dnsid)
    if {$require_callingmaildomain_dnsok} {
	ic_commandresponse major "450 Cannot find mail domain (MX for $chnow): $emsg"
    } else {
	ic_set_maildomain $chstart $hs
    }
}

thread_subproc ic set_maildomain {ch hs} {
    set state(cmdomain) $ch
    set state(helostring) $hs
    ic_mtachan_open
    threadio_commandresponse ic $id $state(mtachan) {} {} mta_greeting_ok {}
}

thread_subproc ic mtachan_open {} {
    set lcmd [list open |[list sendmail -bs -oem \
	    -oMa $state(ra) -oMr $state(smtpstyle)-sauce \
	    -oMs $state(rh) -oMt $state(ident)] r+]
    debug 2 "running sendmail: $lcmd"
    set state(mtachan) [eval $lcmd]
    fconfigure $state(mtachan) -blocking false -translation {binary crlf}
    chanset_desc $state(mtachan) "$state(desc) / MTA"
}

thread_chainproc ic mta_greeting_ok {data} {
    if {![regexp {^220} $data]} {
	ic_mta_greethelo_err $data
	return
    }	
    threadio_commandresponse ic $id $state(mtachan) \
	    "$state(helocmd) $state(helostring)" {} mta_helo_ok {}
}

thread_subproc ic mta_greethelo_err {emsg} {
    global canonical_hostname
    
    regsub -nocase {^[0-9]* ?[-+.:0-9a-z]* *} $emsg {} emsg
    ic_commandfinalresponse major "421 $canonical_hostname $emsg"
}

thread_chainproc ic mta_helo_ok {data} {
    global canonical_hostname
    if {![regexp {^2[0-9][0-9]} $data]} {
	ic_mta_greethelo_err $data
	return
    }
    set str "$canonical_hostname hello $state(ident)@$state(rh)"
    if {![string match {\[*\]} $state(cmdomain)]} {
	append str " (postmaster@$state(cmdomain)?)"
    }
    if {"$state(helocmd)" == "helo"} {
	set op "250 $str"
    } elseif {"$state(helocmd)" == "ehlo"} {
	set op "250-$str\n"
	foreach l [lrange [split $data "\n"] 1 end] {
	    if {[regexp -nocase {^250[- ]([-a-z0-9]+)(.*)$} $l all keyword params]} {
		set params [string trim $params]
		switch -exact -- [string tolower $keyword] {
		    8bitmime - size {
			append op "250-[string toupper $keyword] $params\n"
		    }
		}
	    }
	}
	append op "250 PIPELINING"
#	append op "250 X-SAUCE"
    } else {
	error "internal error - ugh? $helocmd"
    }
    ic_commandresponse nopartresp $op
}

thread_subproc ic mailfrom_fail {message} {
    unset state(mf_lp)
    unset state(mf_dm)
    unset state(mf_parms)
    catch {
	unset state(mf_message)
    }
    catch {
	unset state(minsiteage)
	unset state(minaddrage)
    }
    ic_commandresponse major $message
}

thread_chainproc ic mailfrom_avf_ok {ok message} {
    unset state(avfid)
    if {$ok} {
	set state(mf_message) $message
	ic_rbl
    } else {
	smtp_prefix_response $message 550 message
	ic_mailfrom_fail $message
    }
}

thread_chainproc ic mailfrom_avf_err {emsg} {
    unset state(avfid)
    ic_mailfrom_fail "450 Unable to verify: [proto_quote $emsg]"
}
	
thread_subproc ic rbl {} {
    global rbls new_addr_defer new_site_defer new_addr_message new_site_message

    set state(minaddrage) 0
    set state(minsiteage) 0
    ic_rbl_minage addr $state(mf_lp)@$state(mf_dm) $new_addr_defer $new_addr_message
    ic_rbl_minage site $state(ra) $new_site_defer $new_site_message
	
    set tolookup {}
    foreach dq [split $state(ra) .] { set tolookup $dq.$tolookup }

    foreach rbl $rbls {
	manyset $rbl dm maa msa rblmsg
	regsub -all {%d} $rblmsg $dm rblmsg
	set thread [thread_start dns "$state(desc) / rbl $dm" $tolookup$dm TXT]
	lappend state(rblids) $thread
	thread_join ic $id dns $thread rbl_done rbl_err $thread $maa $msa $rblmsg
    }
    ic_rbl_checkdone
}

thread_chainproc ic rbl_done {thread maa msa rblmsg answers errors etype} {
    ic_rbl_rmthread $thread
    if {[llength $answers]} {
	set l {}
	foreach a $answers {
	    regsub {^\"} $a {} a
	    regsub {\"$} $a {} a
	    set a [proto_quote $a]
	    lappend l $a
	}
	regsub -all {%m} $rblmsg [join $l ", "] rblmsg
	regsub -all {%p} $rblmsg {%} rblmsg
	if {![string length $maa]} {
	    set state(no) $rblmsg
	} else {
	    ic_rbl_minage addr $state(mf_lp)@$state(mf_dm) $maa $rblmsg
	    ic_rbl_minage site $state(ra) $msa $rblmsg
	}
    }
    ic_rbl_checkdone
}

thread_chainproc ic rbl_err {thread maa msa rblmsg emsg} {
    ic_rbl_rmthread $thread
    ic_rbl_checkdone
}

thread_subproc ic rbl_rmthread {thread} {
    set ntl {}
    foreach t $state(rblids) {
	if {"$t" != "$thread"} { lappend ntl $t }
    }
    set state(rblids) $ntl
}

thread_subproc ic rbl_minage {what key newminage msg} {
    upvar #0 remember_${what}_defer remember_defer

    debug 2 rbl_minage $what $key $newminage $msg
    if {!$newminage} return
    if {"$key" == "@"} return
    set whatstate [ds_get $what-list $key]
    if {"$whatstate" == "white" || "$whatstate" == "whitesoon"} return
    if {$state(min${what}age) >= $newminage} return
    set state(min${what}age $newminage

    set now [clock seconds]
    set firstcontact [ds_get $what-seen $key]
    if {"$firstcontact" == "unknown"} {
	set firstcontact $now
	ds_set $what-seen $key $now $remember_defer
	debug 2 rbl_minage ... firstcontact $now
    }
    if {$now < $firstcontact+$newminage} {
	set state(defer) $msg
	debug 2 rbl_minage ... defer $now $firstcontact+$newminage
    }
}

thread_subproc ic rbl_checkdone {} {
    if {[llength $state(rblids)]} return
    ic_mailfrom_ok
}

thread_subproc ic mailfrom_ok {} {
    global max_smtpparms_size
    if {[string length $state(mf_parms)] > $max_smtpparms_size} {
	ic_mailfrom_fail "503 MAIL FROM parameter string too long"
    } else {
	set addr "[lp_quote $state(mf_lp)]@$state(mf_dm)"
	if {"$addr" == "@"} { set addr {} }
	threadio_commandresponse ic $id $state(mtachan) \
		"mail from:<$addr> $state(mf_parms)" {} mta_mailfrom_ok {}
    }
}

thread_chainproc ic mta_mailfrom_ok {data} {
    if {[regexp {^2[0-9][0-9]} $data]} {
	smtp_prefix_response $state(mf_message) 250 message
	ic_commandresponse command $message
	unset state(mf_message)
	unset state(mf_parms)
	catch {
	    unset state(minsiteage)
	    unset state(minaddrage)
	}
    } else {
	ic_mailfrom_fail $data
    }
}

thread_chainproc ic mta_rcptadmin_ok {data} {
    global bland_message
    set state(a_admin) 1
    ic_commandresponse command "250 $bland_message"
}

thread_chainproc ic mta_rcpt_ok {atype data} {
    if {[regexp {^2[0-9][0-9]} $data]} {
	set state(a_$atype) 1
	ic_commandresponse command $data
    } else {
	ic_commandresponse major $data
    }
}

thread_chainproc ic mta_rset_ok {data} {
    catch { unset state(mf_lp) }
    ic_commandresponse command "250 OK"
}

thread_subproc ic reject_rcpt {str} {
    global blacklist_message
    
    if {[info exists state(add_bl)]} {
	set msg $blacklist_message
	set state(whyreject) $state(no)
    } else {
	set msg $state(no)
	set state(whyreject) {}
    }
    ic_commandresponse major "550 $msg"
}
