/*
   XGFont.m

   NSFont for GNUstep GUI X/GPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Scott Christley
   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May, October 1998
   Author:  Michael Hanni <mhanni@sprintmail.com>
   Date: August 1998

   This file is part of the GNUstep GUI X/GPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <gnustep/xgps/XGContext.h>
#include <Foundation/NSDebug.h>

@interface XGFont : NSFont
{
}

+ (NSFont*) fontWithName: (NSString*)name matrix: (const float*)fontMatrix;
- (XFontStruct *) xFontStruct;

@end

extern void XRSetCurrentFont(NSFont *aFont);                // set drawing
							    // engine's font
//
// Backend structure for XGFont
//
typedef struct                                              // Font info struct
{
  XFontStruct *font_info;                                 // ptr to xfont
  float tabWidth;

} XGFont_struct;

#define XFONT (((XGFont_struct *)be_font_reserved)->font_info)
#define TABWIDTH (((XGFont_struct *)be_font_reserved)->tabWidth)

@implementation XGFont

// class global dictionary of existing fonts

static NSMutableDictionary	*xFontDictionary = nil;
static NSMutableDictionary	*fontDictionary = nil;

- (XFontStruct *) xFontStruct
{
  return XFONT;
}

// private XGPS init method not part of OS spec

- (void) _initWithXFont: (XFontStruct *)xFontStruct
{
  XFONT = xFontStruct;
  TABWIDTH = 0;
  TABWIDTH = [self widthOfString: @"\t"];
}

+ (void) initialize
{
  static BOOL initialized = NO;

  if (!initialized)
  {
    initialized = YES;
    fontDictionary = [NSMutableDictionary new];
    xFontDictionary = [NSMutableDictionary new];
  }
}

+ (void)makeKnownFonts:(NSMutableDictionary *)knownFontsDict
{
  [xFontDictionary addEntriesFromDictionary:knownFontsDict];
}

+ (NSFont*) fontWithName: (NSString*)name matrix: (const float*)fontMatrix
{
  Display *xDisplay = [XGContext currentXDisplay];
  XGFont* font;
  NSString* xfontname;
  XFontStruct *xFont;

  // Retrieve the XLFD matching the given fontName. DPS->X.

  xfontname = [NSString stringWithFormat:[xFontDictionary
objectForKey:name], (int)fontMatrix[0]];

  // if the desired XLFD does not exist we fudge it with 9x15.

  if (!xfontname)
    xfontname = @"9x15";

  // Do we have an already contructed XGFont for this request? If yes we
  // return it.

  if ((font = [fontDictionary objectForKey:xfontname]))
    return font;

  // Now we create an XGFont to add to the original fontDict.

  font = AUTORELEASE([self new]);
  [fontDictionary setObject: font forKey: xfontname];

  font->fontName = [name copy];                           // set font attr's
  memcpy (font->matrix, fontMatrix, sizeof (font->matrix));
						    // alloc back-end structure
  font->be_font_reserved = calloc(1, sizeof(XGFont_struct));

  // Load X font and get font info structure.

  if (xDisplay)
    {
      if ((xFont = XLoadQueryFont(xDisplay, [xfontname cString])) == NULL)
	{
	  NSLog(@"Selected font: %@ is not available.\n"
	    @"Using system fixed font instead", xfontname);

	  // Shouldn't get here, but still.
	  // try default font
	  if ((xFont = XLoadQueryFont(xDisplay, "9x15")) == NULL)
	    NSLog(@"Unable to open fixed font");
	}
      else
	NSDebugLog(@"Loaded font: %@", xfontname);

      [font _initWithXFont: xFont];
    }

  return font;
}

- (void) dealloc
{
  XUnloadFont([XGContext currentXDisplay], XFONT->fid);
  free(be_font_reserved);
  [super dealloc];
}

- (void) set
{
  XRSetCurrentFont(self);
}

- (float) widthOfString: (NSString*)string
{
  int strLength;
  const char *cStr = [string cString];

  if ((strLength = [string length]) == 1)                  // optimize calc of
    {                                                   // tab width
      if (*cStr == '\t')
	return TABWIDTH;
    }

  return XTextWidth(XFONT, cStr, strLength);
}

- (NSFont*) printerFont
{
  return self;
}

- (NSFont*) screenFont
{
  return self;
}

- (NSString*) familyName
{
  return fontName;
}

- (NSRect) boundingRectForFont
{
  return NSMakeRect(
		    (float)(0 + XFONT->min_bounds.lbearing),
		    (float)(0 - XFONT->max_bounds.ascent),
		    (float)(XFONT->max_bounds.rbearing - XFONT->min_bounds.lbearing),
		    (float)(XFONT->max_bounds.ascent + XFONT->max_bounds.descent));
}	
 
- (NSString *) encodingScheme
{
  return nil;
}

- (BOOL) isFixedPitch
{
  BOOL fixedFont = NO;		/* Is this font fixed pitch? default, NO */
  
  Bool found;			/* Did we find the property? */
  unsigned long resultAtom;	/* We get an atom value returned */
  char* result;			/* but we need the string version of it */

  /* Need to get the spacing atom... */
  Display *xDisplay = [XGContext currentXDisplay];
  Atom spacingAtom = XInternAtom(xDisplay, "SPACING", False);
  
  /* Try to read the SPACING property of the font */
  found = XGetFontProperty(XFONT, spacingAtom, &resultAtom);

  if (found)
    {
      /* Turn the atom we got back into a string. */
      result = XGetAtomName(xDisplay, resultAtom);

      /* Only proportional fonts are not fixed pitch... */
      if (result[0] != 'P')
	fixedFont = YES;

      /* Release the memory - safely :) */
      if (result)
	XFree(result);
    }

  /* We could calculate the pitch from the XLFD but that does not seem to be
     saved. If we can't get the property, say no and cope. */
  return fixedFont;
}

- (BOOL) isBaseFont
{
  return NO;
}

- (float) ascender
{
  return XFONT->max_bounds.ascent;
}

- (float) descender
{
  return -(XFONT->max_bounds.descent);
}

- (float) capHeight
{
  return 0;
}

- (float) italicAngle
{
  return 0;
}

- (NSSize) maximumAdvancement
{
  return NSMakeSize(XFONT->max_bounds.width,
		      (XFONT->max_bounds.ascent +XFONT->max_bounds.descent));
}

- (NSSize) minimumAdvancement
{
  return NSMakeSize(0,0);
}

- (float) underlinePosition
{
  return 0;
}

- (float) underlineThickness
{
  return 0;
}

- (float*) widths
{
  return 0;
}

// glyph is an ascii char value
- (NSSize) advancementForGlyph: (NSGlyph)glyph
{
  NSSize	advancement = NSMakeSize(0, 0);
  unsigned	min = XFONT->min_char_or_byte2;
  unsigned	max = XFONT->max_char_or_byte2;

            // xfont helv seems to have info at -32?
  if (XFONT->per_char && glyph >= min && glyph < max)
    advancement.width = (float)XFONT->per_char[glyph - min].width;
  else                            // if per_char is NULL assume max bounds
    advancement.width = (float)XFONT->max_bounds.width;

  return advancement;
}

- (NSRect) boundingRectForGlyph: (NSGlyph)glyph
{
  return NSZeroRect;
}

- (BOOL) glyphIsEncoded: (NSGlyph)glyph
{
  return NO;
}

- (NSGlyph) glyphWithName: (NSString*)glyphName
{
  return 0;
}

- (NSPoint) positionOfGlyph: (NSGlyph)curGlyph
	    precededByGlyph: (NSGlyph)prevGlyph
		  isNominal: (BOOL *)nominal
{
  return NSZeroPoint;
}

- (float) pointSize
{
  return XFONT->max_bounds.ascent;
}

- (Class) classForCoder: (NSCoder*)aCoder
{
  if ([self class] == [XGFont class])
    return [super class];
  return [self class];
}

- (id) copyWithZone: (NSZone*)zone
{
  return self;
}
@end
