/* XGContextWindows - DPS like methods for window/screen handling

   Copyright (C) 1999 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@gnu.org>
   Date: Nov 1999
   
   This file is part of GNUstep

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

#include <Foundation/NSString.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSValue.h>
#include <AppKit/GSWraps.h>

#include "SharedX/wraster.h"
#ifdef XDPS_BACKEND_LIBRARY
#define XGContext       NSDPSContext
#include "gnustep/xdps/NSDPSContextWindow.h"
#include "PXKDrawingEngine.h"
#else
#include "gnustep/xgps/XGContextWindow.h"
#include "gnustep/xgps/XGContextPrivate.h"
#include "XGDrawingEngine.h"
#endif
#include <X11/cursorfont.h>

#define	OVERRIDE	0

#define MAX_SCREENS	100
#define XDPY (((RContext *)context)->dpy)
#define XSCR (((RContext *)context)->screen_number)

/* This works around problems with non-"identity" matrices */
#ifdef XDPS_BACKEND_LIBRARY
#define DPSNormalMatrix \
  do { \
    if ([[NSUserDefaults standardUserDefaults] boolForKey: @"DPSDefaultMatrix"] == NO) \
      { \
        float nctm[6]; \
        NSDebugLog(@"Resetting default matrix\n"); \
        nctm[0]=1; nctm[1]=0; nctm[2]=0; nctm[3]=-1; nctm[4]=0; nctm[5]=0; \
        GSWSetMatrix(self, nctm); \
      } \
    } while (0)
#else
#define DPSNormalMatrix
#endif

/* Window decoration Atom */
static	Atom win_decor_atom = None;
/* Keep track of the windows we control */
static NSMapTable* windowmaps = NULL;
static NSMapTable* windowtags = NULL;
/* Current mouse grab window */
static gswindow_device_t *grab_window = NULL;

@interface NSEvent (WindowHack)
- (void) _patchLocation: (NSPoint)loc;
@end

@implementation NSEvent (WindowHack)
- (void) _patchLocation: (NSPoint)loc
{
  location_point = loc;
}
@end

@implementation XGContext (DPSWindow)

+ (gswindow_device_t *) _windowForXWindow: (Window)xWindow
{
  return NSMapGet(windowmaps, (void *)xWindow);
}

+ (gswindow_device_t *) _windowWithTag: (int)windowNumber
{
  return NSMapGet(windowtags, (void *)windowNumber);
}

- (gswindow_device_t *)_rootWindowForScreen: (int)screen
{
  int x, y, width, height;
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)screen);
  if (window)
    return window;

  window = objc_malloc(sizeof(gswindow_device_t));

  window->screen = screen;
  window->ident  = RootWindow(XDPY, screen);
  window->root   = window->ident;
  window->type   = NSBackingStoreNonretained;
  window->number = screen;
  window->map_state = IsViewable;
  if (window->ident)
    XGetGeometry(XDPY, window->ident, &window->root, 
		 &x, &y, &width, &height,
		 &window->border, &window->depth);

  window->xframe = NSMakeRect(x, y, width, height);
  NSMapInsert (windowtags, (void*)window->number, window);
  NSMapInsert (windowmaps, (void*)window->ident,  window);
  return window;
}

/* Create the window and screen list if necessary, add the root window to
   the window list as window 0 */
- (void) _checkWindowlist
{
  if (windowmaps)
    return;

  windowmaps = NSCreateMapTable(NSNonOwnedPointerMapKeyCallBacks,
				 NSNonOwnedPointerMapValueCallBacks, 20);
  windowtags = NSCreateMapTable(NSIntMapKeyCallBacks,
				 NSNonOwnedPointerMapValueCallBacks, 20);
}

/* Sets up a backing pixmap when a window is created or resized.  This is
   only done if the Window is buffered or retained. */
- (void) _createBuffer: (gswindow_device_t *)window resize: (BOOL)resize
{
  if (window->type == NSBackingStoreNonretained)
    return;
  if (!resize && window->buffer)
    return;

  /* If this was a resize, then create a new pixmap and free the old one.
     I assume a resize event will trigger a redisplay of the data, and we
     don't need to keep any information from the old pixmap. */
  if (resize && window->buffer)
    XFreePixmap(XDPY, window->buffer);

  if (window->depth == 0)
    window->depth = DefaultDepth(XDPY, XSCR);
  if (NSWidth(window->xframe) == 0 && NSHeight(window->xframe) == 0)
    {
      NSDebugLLog(@"NSWindow", @"Cannot create buffer for ZeroRect frame");
      window->buffer = 0;
      return;
    }

  window->buffer = XCreatePixmap(XDPY, window->root,
				 NSWidth(window->xframe),
				 NSHeight(window->xframe),
				 window->depth);
  if (!window->buffer)
    {
      NSLog(@"DPS Windows: Unable to create backing store\n");
      return;
    }

  XFillRectangle(XDPY,
		 window->buffer,
		 window->gc,
		 0, 0, 
		 NSWidth(window->xframe),
		 NSHeight(window->xframe));
}

/* Fill in various window parameters. */
- (void) _updateWindowParameters: (gswindow_device_t *)window
{
  int      x, y;
  unsigned width, height;
  unsigned old_width = NSWidth(window->xframe);
  unsigned old_height = NSHeight(window->xframe);
  XWindowAttributes winattr;
  Window   qRoot;
  Window   qParent;
  Window   *qChildren;
  unsigned qNum;
  
  if (!window->ident)
    return;

  XFlush (XDPY);
  XGetGeometry(XDPY, window->ident, &window->root, 
	       &x, &y, &width, &height,
	       &window->border, &window->depth);
  window->xframe = NSMakeRect(x, y, width, height);
  if (XQueryTree (XDPY, window->ident, &qRoot, &qParent, &qChildren, 
		  &qNum))
    XFree(qChildren);
  
  if (qParent != qRoot)
    {
      int nx, ny;
      Window child;
      // window has a title bar so have X server
      // translate to display coordinates first
      XTranslateCoordinates(XDPY, window->ident, window->root,
			    x, y, &nx, &ny, &child);
      window->xframe = NSMakeRect(nx, y+ny, width, height);
      x = nx;
      y = ny;
    }

  XGetWindowAttributes(XDPY, window->ident, &winattr);
  window->map_state = winattr.map_state;
  
  NSDebugLLog (@"NSWindow", @"window geom device ((%f, %f), (%f, %f))",
	        window->xframe.origin.x,  window->xframe.origin.y, 
	        window->xframe.size.width,  window->xframe.size.height);
  
  if (window->buffer && (old_width != width || old_height != height))
    {
      XFreePixmap(XDPY, window->buffer);
      window->buffer = 0;
    }
}

- (void) DPSwindow: (float)x : (float)y : (float)w : (float)h : (int)type : (int *)num
{
  static int		last_win_num = 0;
  gswindow_device_t	*window;
  gswindow_device_t	*root;
  XGCValues		values;
  unsigned long		valuemask;

  if (protocols_atom == None)
    {
      protocols_atom = XInternAtom(XDPY, "WM_PROTOCOLS", False);
    }
  if (take_focus_atom == None)
    {
      take_focus_atom = XInternAtom(XDPY, "WM_TAKE_FOCUS", False);
    }
  if (delete_win_atom == None)
    {
      delete_win_atom = XInternAtom(XDPY, "WM_DELETE_WINDOW", False);
    }

  [self _checkWindowlist];
  root = [self _rootWindowForScreen: XSCR];

  /* We're not allowed to create a zero rect window */
  if (w == 0 && h == 0)
    {
      w = 200;
      h = 200;
    }
  /* Translate to X coordinates */
  y = DisplayHeight(XDPY, XSCR) - (y + h);

  window = objc_malloc(sizeof(gswindow_device_t));
  memset(window, '\0', sizeof(gswindow_device_t));
  window->screen = XSCR;
  window->xframe = NSMakeRect(x, y, w, h);
  window->type = type;
  window->root = root->ident;
  window->parent = root->ident;
  window->depth = ((RContext *)context)->depth;
  window->xwn_attrs.border_pixel = ((RContext *)context)->black;
  window->xwn_attrs.background_pixel = ((RContext *)context)->white;
  window->xwn_attrs.colormap = ((RContext *)context)->cmap;

  window->ident = XCreateWindow(XDPY, window->root,
				x, y, w, h,
				0, 
				((RContext *)context)->depth,
				CopyFromParent,
				((RContext *)context)->visual,
				(CWColormap | CWBackPixel|CWBorderPixel),
				&window->xwn_attrs);

  window->xwn_attrs.save_under = False;
  window->xwn_attrs.override_redirect = False;
  window->map_state = IsUnmapped;


  // Create an X GC for the content view set it's colors
  values.foreground = window->xwn_attrs.background_pixel;
  values.background = window->xwn_attrs.background_pixel;
  values.function = GXcopy;
  valuemask = (GCForeground | GCBackground | GCFunction);
  window->gc = XCreateGC(XDPY, window->ident, valuemask, &values);

  // Set the X event mask
  XSelectInput(XDPY, window->ident, ExposureMask | KeyPressMask |
				KeyReleaseMask | ButtonPressMask |
			     ButtonReleaseMask | ButtonMotionMask |
			   StructureNotifyMask | PointerMotionMask |
			       EnterWindowMask | LeaveWindowMask |
			       FocusChangeMask | PropertyChangeMask |
			    ColormapChangeMask | KeymapStateMask |
			    VisibilityChangeMask);

  window->gen_hints.window_group = appRootWindow;
  window->gen_hints.initial_state = NormalState;
  window->gen_hints.input = True;
  window->gen_hints.flags = StateHint | InputHint | WindowGroupHint;
  XSetWMHints(XDPY, window->ident, &window->gen_hints);

  window->protocols[0] = take_focus_atom;
  window->protocols[1] = delete_win_atom;
  XSetWMProtocols(XDPY, window->ident, window->protocols, 2);

  window->exposedRects = [NSMutableArray new];
  window->region = XCreateRegion();
  window->buffer = 0;
  window->alpha_buffer = 0;

  /*
   * FIXME - should this be protected by a lock for thread safety?
   * generate a unique tag for this new window.
   */
  do
    {
      last_win_num++;
    }
  while (last_win_num == 0 || NSMapGet(windowtags, (void*)last_win_num) != 0);
  window->number = last_win_num;

  if (num)
    *num = window->number;

  // Insert window into the mapping
  NSMapInsert(windowmaps, (void*)window->ident, window);
  NSMapInsert(windowtags, (void*)window->number, window);
}

- (void) DPStermwindow: (int)num
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)num);
  if (!window)
    return;

  if (window->ident)
    {
      XDestroyWindow(XDPY, window->ident);
      XFreeGC (XDPY, window->gc);
      NSMapRemove(windowmaps, (void*)window->ident);
    }

  if (window->buffer)
    XFreePixmap (XDPY, window->buffer);
  XDestroyRegion (window->region);
  RELEASE(window->exposedRects);
  NSMapRemove(windowtags, (void*)num);
  objc_free(window);
}

/*
 * Return the offsets between the window content-view and it's frame
 * depending on the window style.
 */
- (void) DPSstyleoffsets: (float *) l : (float *) r : (float *) t : (float *) b : (int) style ;
{
  /*
   * FIXME
   * This should get window-manager specific decoration information somehow
   * at the moment, hardcode ideal values for when Window Maker gets it right.
   */
  if (style == NSBorderlessWindowMask)
    {
      *l = *r = *t = *b = 0.0;
    }
  else
    {
      *l = *r = *t = *b = 1;
      if (NSResizableWindowMask & style)
	{
	  *b = 11;
	}
      if ((NSTitledWindowMask|NSClosableWindowMask|NSMiniaturizableWindowMask)
	& style)
	{
	  *t = 22;
	}
    }
}

- (void) DPSstylewindow: (int)style : (int) num
{
  gswindow_device_t	*window;

  window = NSMapGet(windowtags, (void *)num);
  if (!window)
    return;

  if (window->win_attrs.window_style != style
    || (window->win_attrs.flags & GSWindowStyleAttr) == 0)
    {
      if (OVERRIDE && window->win_attrs.window_level >= NSDockWindowLevel
	&& style == NSBorderlessWindowMask)
	{
	  if (window->xwn_attrs.save_under == False
	    || window->xwn_attrs.override_redirect == False)
	    {
	      window->xwn_attrs.save_under = True;
	      window->xwn_attrs.override_redirect = True;
	      XChangeWindowAttributes(XDPY, window->ident,
		(CWSaveUnder|CWOverrideRedirect), &window->xwn_attrs);
	    }
	}
      else
	{
	  if (window->xwn_attrs.save_under == True
	    || window->xwn_attrs.override_redirect == True)
	    {
	      window->xwn_attrs.save_under = False;
	      window->xwn_attrs.override_redirect = False;
	      XChangeWindowAttributes(XDPY, window->ident,
		(CWSaveUnder|CWOverrideRedirect), &window->xwn_attrs);
	    }
	}

      window->win_attrs.flags |= GSWindowStyleAttr;
      window->win_attrs.window_style = style;
      if ((win_decor_atom == (Atom)None))
	win_decor_atom = XInternAtom(XDPY,"_GNUSTEP_WM_ATTR", False);

      // send WindowMaker WM window style hints
      if ((win_decor_atom != (Atom)None))
	XChangeProperty(XDPY, window->ident, win_decor_atom, win_decor_atom,
	  32, PropModeReplace, (unsigned char *)&window->win_attrs,
	  sizeof(GNUstepWMAttributes)/sizeof(CARD32));
    }
}

- (void) DPStitlewindow: (const char *)window_title : (int) num
{
  XTextProperty windowName;
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)num);
  if (!window)
    return;

  if (window_title && window->ident)
    {
      XStringListToTextProperty((char**)&window_title, 1, &windowName);
      XSetWMName(XDPY, window->ident, &windowName);
      XSetWMIconName(XDPY, window->ident, &windowName);
    }
}

- (void) DPSdocedited: (int)edited : (int) num
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)num);
  if (!window)
    return;

  window->win_attrs.flags |= GSExtraFlagsAttr;
  if (edited)
    {
      window->win_attrs.extra_flags |= GSDocumentEditedFlag;
    }
  else
    {
      window->win_attrs.extra_flags &= ~GSDocumentEditedFlag;
    }
  if ((win_decor_atom == (Atom)None))
    win_decor_atom = XInternAtom(XDPY,"_GNUSTEP_WM_ATTR", False);

  // send WindowMaker WM window style hints
  if ((win_decor_atom != (Atom)None))
    {
      XChangeProperty(XDPY, window->ident, win_decor_atom, win_decor_atom,
	32, PropModeReplace, (unsigned char *)&window->win_attrs,
	sizeof(GNUstepWMAttributes)/sizeof(CARD32));
    }
}

- (void) DPSminiwindow: (int) num
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)num);
  if (!window)
    return;

  XIconifyWindow(XDPY, window->ident, XSCR);
}

- (gswindow_device_t *) _windowdevice: (int)num
{
  gswindow_device_t *window;

  [self _checkWindowlist];
  window = NSMapGet(windowtags, (void *)num);
  if (!window)
    [NSException raise: @"DPSinvalidparam" 
		 format: @"Invalid window number %d", num];

  [self _updateWindowParameters: window];
  if (window->buffer == 0)
    [self _createBuffer: window resize: NO];

  GSWinitcontext (self, num, window->gc, 
  		  (window->buffer) ? window->buffer : window->ident, 
		  0, NSHeight(window->xframe));
  DPSNormalMatrix;

  return window;
}

- (void) DPSwindowdevice: (int)num
{
  [self _windowdevice: num];
}

- (void) DPSwindowdeviceround: (int)num
{
  [self _windowdevice: num];
}

- (void) DPScurrentwindow: (int *)num
{
}

- (void) DPSorderwindow: (int)op : (int)otherWin : (int)winNum
{
  gswindow_device_t	*window;
  gswindow_device_t	*other;
  NSWindow      *nswin;
  NSRect	frame;
  int		level;

  window = [XGContext _windowWithTag: winNum];
  if (window == NULL)
    return;
  if (otherWin)
    other = [XGContext _windowWithTag: otherWin];
  else
    other = NULL;

  nswin = [NSWindow _windowWithNumber: winNum];
  frame = [nswin frame];
  level = [nswin level];

  DPSsetwindowlevel(self, level, winNum);

  if (op != NSWindowOut && window->map_state != IsViewable)
    {
      /*
       * If this is an undecorated window, we must make sure
       * that we have sensible offsets form positioning it so
       * that when it becomes decorated, it's in the right place.
       */
      if (window->parent == window->root)
	{
	  if (window->win_attrs.window_style
	    == NSBorderlessWindowMask)
	    {
	      /*
	       * Borderless window - won't be decorated.
	       */
	      window->xoff = 0;
	      window->yoff = 0;
	    }
	}

      window->siz_hints.flags |= PPosition | USPosition;
      window->siz_hints.x = frame.origin.x + window->xoff;
      window->siz_hints.y = DisplayHeight(XDPY, window->screen)
	- (frame.origin.y + frame.size.height + window->yoff);

      NSDebugLLog(@"Moving", @"Want-X %d - x:%d y:%d w:%d h:%d\n",
	window->ident, window->siz_hints.x, window->siz_hints.y,
	window->siz_hints.width, window->siz_hints.height);
      XMoveWindow(XDPY, window->ident, window->siz_hints.x,
	window->siz_hints.y);
      XSetNormalHints(XDPY, window->ident, &window->siz_hints);
    }

  switch (op)
    {
      case NSWindowBelow:
        if (other)
	  {
	    XWindowChanges chg;
	    chg.sibling = other->ident;
	    chg.stack_mode = Below;
	    XConfigureWindow(XDPY, window->ident, CWSibling|CWStackMode, &chg);
	  }
	else
	  {
	    XWindowChanges chg;
	    chg.stack_mode = Below;
	    XMapWindow(XDPY, window->ident);
	    XConfigureWindow(XDPY, window->ident, CWStackMode, &chg);
	    //XLowerWindow(XDPY, window->ident);
	  }
	break;

      case NSWindowAbove:
	XMapRaised(XDPY, window->ident);
	break;

      case NSWindowOut:
	XUnmapWindow(XDPY, window->ident);
	break;
    }
  XFlush(XDPY);

  /*
   * Now do general window management stuff.
   */
  if ([windowList indexOfObjectIdenticalTo: nswin] == NSNotFound)
    {
      [windowList addObject: nswin];
    }
  if (op != NSWindowOut)
    {
      [[nswin contentView] setNeedsDisplay: YES];
      [nswin update];
    }
}

- (void) DPSmovewindow: (float)x : (float)y : (int)num
{
}

/* Sync window information with possible change in X window */
- (void) DPSupdatewindow: (int)win
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  [self _updateWindowParameters: window];
  if (window->buffer == 0)
    {
      [self _createBuffer: window resize: YES];
    }

  GSWinitcontext (self, win, window->gc, 
  		  (window->buffer) ? window->buffer : window->ident, 
		  0, NSHeight(window->xframe));
  DPSNormalMatrix;
}

- (void) DPSplacewindow: (float)x : (float)y : (float)w : (float)h : (int)win
{
  NSAutoreleasePool	*arp;
  NSRect		rect = NSMakeRect(x, y, w, h);
  NSRect		xVal;
  NSRect		last;
  NSEvent		*event;
  NSDate		*limit;
  NSMutableArray	*tmpQueue;
  unsigned		pos;
  float			xdiff;
  float			ydiff;
  gswindow_device_t	*window;
  NSWindow              *nswin;
  NSRect		frame;
  BOOL			resize = NO;
  BOOL			move = NO;

  window = [XGContext _windowWithTag: win];
  if (window == 0)
    return;
  nswin  = [NSWindow _windowWithNumber: win];
  frame = [nswin frame];
  if (NSEqualRects(rect, frame) == YES)
    return;
  if (NSEqualSizes(rect.size, frame.size) == NO)
    {
      resize = YES;
      move = YES;
    }
  if (NSEqualPoints(rect.origin, frame.origin) == NO)
    {
      move = YES;
    }
  NSDebugLLog(@"Moving", @"Want-OS %d - x:%f y:%f w:%f h:%f\n",
    window->ident, rect.origin.x, rect.origin.y,
    rect.size.width, rect.size.height);
  xdiff = rect.origin.x - frame.origin.x;
  ydiff = rect.origin.y - frame.origin.y;

  /*
   * If this is an undecorated window, we must make sure
   * that we have sensible offsets form positioning it so
   * that when it becomes decorated, it's in the right place.
   */
  if (window->parent == window->root)
    {
      if (window->win_attrs.window_style
	== NSBorderlessWindowMask)
	{
	  /*
	   * Borderless window - won't be decorated.
	   */
	  window->xoff = 0;
	  window->yoff = 0;
	}
    }

  window->siz_hints.width = (int)rect.size.width;
  window->siz_hints.height = (int)rect.size.height;
  window->siz_hints.x = rect.origin.x + window->xoff;
  window->siz_hints.y = DisplayHeight(XDPY, window->screen)
    - (rect.origin.y + rect.size.height + window->yoff);

  last = window->xframe;

  NSDebugLLog(@"Moving", @"Want-X %d - x:%d y:%d w:%d h:%d\n", window->ident,
    window->siz_hints.x, window->siz_hints.y,
    window->siz_hints.width, window->siz_hints.height);
  
  XMoveResizeWindow (XDPY, window->ident,
    window->siz_hints.x, window->siz_hints.y,
    window->siz_hints.width, window->siz_hints.height);
  XSetNormalHints(XDPY, window->ident, &window->siz_hints);

  /*
   * Now massage all the events currently in the queue to make sure
   * mouse locations in our window are adjusted as necessary.
   */
  arp = [NSAutoreleasePool new];
  limit = [NSDate distantPast];	/* Don't wait for new events.	*/
  tmpQueue = [NSMutableArray arrayWithCapacity: 8];
  for (;;)
    {
      NSEventType	type;

      event = DPSGetEvent(self, NSAnyEventMask, limit,
	NSEventTrackingRunLoopMode);
      if (event == nil)
	break;
      type = [event type];
      if (type == NSAppKitDefined && [event windowNumber] == win)
	{
	  GSAppKitSubtype	sub = [event subtype];

	  /*
	   *	Window movement or resize events for the window we are
	   *	watching are posted immediately, so they can take effect.
	   */
	  if (sub == GSAppKitWindowMoved || sub == GSAppKitWindowResized)
	    {
	      [nswin sendEvent: event];
              if (sub == GSAppKitWindowResized)
                [nswin update];
	    }
	  else
	    {
	      [tmpQueue addObject: event];
	    }
	}
      else if (type != NSPeriodic && type != NSLeftMouseDragged
	&& type != NSRightMouseDragged && type != NSMouseMoved)
	{
	  /*
	   * Save any events that arrive before our window is moved - excepting
	   * periodic events (which we can assume will be outdated) and mouse
	   * movement events (which might flood us).
	   */
	  [tmpQueue addObject: event];
	}
      if (NSEqualRects(rect, [nswin frame]) == YES)
	{
	  break;
	}
      if (NSEqualRects(last, window->xframe) == NO)
	{
	  NSDebugLog(@"From: %@\nWant %@\nGot %@",
	    NSStringFromRect(last),
	    NSStringFromRect(xVal),
	    NSStringFromRect(window->xframe));
	  last = window->xframe;
	}
    }
  /*
   * If we got any events while waiting for the window movement, we
   * may need to adjust their locations to match the new window position.
   */
  pos = [tmpQueue count];
  while (pos-- > 0) 
    {
      event = [tmpQueue objectAtIndex: pos];
      if ([event windowNumber] == win)
	{
	  NSPoint	loc = [event locationInWindow];

	  loc.x -= xdiff;
	  loc.y -= ydiff;
	  [event _patchLocation: loc];
	}
      DPSPostEvent(self, event, YES);
    }
  [arp release];
  /*
   * Failsafe - if X hasn't told us it has moved/resized the window, we
   * fake the notification.
   */
  if (NSEqualRects([nswin frame], rect) == NO)
    {
      NSEvent	*e;

      if (resize == YES)
	{
	  e = [NSEvent otherEventWithType: NSAppKitDefined
				 location: NSZeroPoint
			    modifierFlags: 0
				timestamp: 0
			     windowNumber: win
				  context: self
				  subtype: GSAppKitWindowResized
				    data1: rect.size.width
				    data2: rect.size.height];
	  [nswin sendEvent: e];
	}
      if (move == YES)
	{
	  e = [NSEvent otherEventWithType: NSAppKitDefined
				 location: NSZeroPoint
			    modifierFlags: 0
				timestamp: 0
			     windowNumber: win
				  context: self
				  subtype: GSAppKitWindowMoved
				    data1: rect.origin.x
				    data2: rect.origin.y];
	  [nswin sendEvent: e];
	}
    }
}

- (void) DPSfrontwindow: (int *)num
{
}

- (void) DPSfindwindow: (float)x : (float)y : (int)op : (int)otherWin : (float *)lx : (float *)ly : (int *)winFound : (int *)didFind
{
}

- (void) DPScurrentwindowbounds: (int)num : (float *)x : (float *)y : (float *)w : (float *)h
{
  gswindow_device_t *window;
  int screenHeight;

  [self _checkWindowlist];
  window = NSMapGet(windowtags, (void *)num);
  if (!window && num < MAX_SCREENS)
    window = [self _rootWindowForScreen: num];
  if (!window)
    return;

  if (window->buffer == 0 && window->map_state != IsViewable)
    {
      /* Sort of a hack. Prevent routines from thinking they can
         get pixel information from a window that is unmapped */
     if (x) *x = 0;
     if (y) *y = 0;
     if (w) *w = 0;
     if (h) *h = 0;
     return;
    }

  screenHeight = DisplayHeight(XDPY, window->screen);
  if (x)
    *x = NSMinX(window->xframe);
  if (y)
    *y = screenHeight - NSMaxY(window->xframe);
  if (w)
    *w = NSWidth(window->xframe);
  if (h)
    *h = NSHeight(window->xframe);
}

- (void) DPSsetexposurecolor
{
}

- (void) DPSsetsendexposed: (int)truth : (int)num
{
}

- (void) DPSsetautofill: (int)truth : (int)num
{
}

- (void) DPScurrentwindowalpha: (int)win : (int *)alpha
{
}

- (void) DPScountscreenlist: (int)ctxt : (int *)count
{
  int c = ScreenCount(XDPY);
  if (count)
    *count = c;
}

- (void) DPSscreenlist: (int)ctxt : (int)count : (int *)windows
{
  /* I guess screen numbers are in order starting from zero, but we
     put the main screen first */
 int i, j;
 if (count > 0)
   windows[0] = XSCR;
 j = 1;
 for (i = 0; i < count; i++)
   {
     if (i != XSCR)
       windows[j++] = i;
   }
}

- (void) DPSsetowner: (int)owner : (int)win
{
}

- (void) DPScurrentowner: (int)win : (int *)owner
{
}

- (void) DPSsetwindowtype: (int)type : (int)win
{
}

- (void) DPSsetwindowlevel: (int)level : (int)win
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  if (window->win_attrs.window_level != level
    || (window->win_attrs.flags & GSWindowLevelAttr) == 0)
    {
      if (OVERRIDE && level >= NSDockWindowLevel
	&& window->win_attrs.window_style == NSBorderlessWindowMask)
	{
	  if (window->xwn_attrs.save_under == False
	    || window->xwn_attrs.override_redirect == False)
	    {
	      window->xwn_attrs.save_under = True;
	      window->xwn_attrs.override_redirect = True;
	      XChangeWindowAttributes(XDPY, window->ident,
		(CWSaveUnder|CWOverrideRedirect), &window->xwn_attrs);
	    }
	}
      else
	{
	  if (window->xwn_attrs.save_under == True
	    || window->xwn_attrs.override_redirect == True)
	    {
	      window->xwn_attrs.save_under = False;
	      window->xwn_attrs.override_redirect = False;
	      XChangeWindowAttributes(XDPY, window->ident,
		(CWSaveUnder|CWOverrideRedirect), &window->xwn_attrs);
	    }
	}

      window->win_attrs.flags |= GSWindowLevelAttr;
      window->win_attrs.window_level = level;
      if ((win_decor_atom == (Atom)None))
	win_decor_atom = XInternAtom(XDPY,"_GNUSTEP_WM_ATTR", False);

      // send WindowMaker WM window style hints
      if ((win_decor_atom != (Atom)None))
	XChangeProperty(XDPY, window->ident, win_decor_atom, win_decor_atom,
	  32, PropModeReplace, (unsigned char *)&window->win_attrs,
	  sizeof(GNUstepWMAttributes)/sizeof(CARD32));
    }
}

- (void) DPScurrentwindowlevel: (int)win : (int *)level
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  /*
   * If we have previously set a level for this window - return the value set.
   */
  if (window != 0 && (window->win_attrs.flags & GSWindowLevelAttr))
    *level = window->win_attrs.window_level;
}

- (void) DPScountwindowlist: (int)context : (int *)count
{
}

- (void) DPSwindowlist: (int)context : (int)count : (int *)windows
{
}

- (void) DPSsetwindowdepthlimit: (int)limit : (int)win
{
}

- (void) DPScurrentwindowdepthlimit: (int)win : (int *)limit
{
}

- (void) DPScurrentwindowdepth: (int)win : (int *)depth
{
  gswindow_device_t *window;

  [self _checkWindowlist];
  window = NSMapGet(windowtags, (void *)win);
  if (!window && win < MAX_SCREENS)
    window = [self _rootWindowForScreen: win];
  if (!window)
    return;

  if (depth)
    *depth = window->depth;
}

- (void) DPSsetdefaultdepthlimit: (int)limit
{
}

- (void) DPScurrentdefaultdepthlimit: (int *)limit
{
}

- (void) DPSsetmaxsize: (float)width : (float)height : (int)win
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);

  if (!window)
    return;

  window->siz_hints.flags |= PMaxSize;
  window->siz_hints.max_height = height;
  window->siz_hints.max_width = width;
  XSetNormalHints(XDPY, window->ident, &window->siz_hints);
}

- (void) DPSsetminsize: (float)width : (float)height : (int)win
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  window->siz_hints.flags |= PMinSize;
  window->siz_hints.min_height = height;
  window->siz_hints.min_width = width;
  XSetNormalHints(XDPY, window->ident, &window->siz_hints);
}

- (void) DPSsetresizeincrements: (float)width : (float)height : (int)win
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  window->siz_hints.flags |= PResizeInc;
  window->siz_hints.height_inc = height;
  window->siz_hints.width_inc = width;
  XSetNormalHints(XDPY, window->ident, &window->siz_hints);
}

// process expose event
- (void)_addExposedRectangle: (XRectangle)rectangle : (int)win
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  if (window->type != NSBackingStoreNonretained)
    {
      XGCValues values;
      unsigned long valuemask;

      // window has a backing store so just copy the exposed rect from the
      // pixmap to the X window

      NSDebugLLog (@"NSWindow", @"copy exposed area ((%d, %d), (%d, %d))",
		  rectangle.x, rectangle.y, rectangle.width, rectangle.height);

      values.function = GXcopy;
      values.plane_mask = AllPlanes;
      values.clip_mask = None;
      values.foreground = ((RContext *)context)->white;
      valuemask = (GCFunction | GCPlaneMask | GCClipMask | GCForeground);
      XChangeGC(XDPY, window->gc, valuemask, &values);

      XCopyArea (XDPY, window->buffer, window->ident, window->gc,
		 rectangle.x, rectangle.y, rectangle.width, rectangle.height,
		 rectangle.x, rectangle.y);
    }
  else
    {
      NSRect	rect;

      // no backing store, so keep a list of exposed rects to be
      // processed in the _processExposedRectangles method
      // Add the rectangle to the region used in -_processExposedRectangles
      // to set the clipping path.
      XUnionRectWithRegion (&rectangle, window->region, window->region);

      // Transform the rectangle's coordinates to PS coordinates and add
      // this new rectangle to the list of exposed rectangles.
      rect.origin = NSMakePoint((float)rectangle.x, rectangle.y);
      rect.size = NSMakeSize(rectangle.width, rectangle.height);
      [window->exposedRects addObject: [NSValue valueWithRect: rect]];
    }
}

- (void) DPSflushwindowrect: (float)x : (float)y : (float)w : (float)h : (int)win
{
  int xi, yi, width, height;
  XGCValues values;
  unsigned long valuemask;
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  if (window->type == NSBackingStoreNonretained)
    {
      XFlush(XDPY);
      return;
    }

  /* FIXME: Doesn't take into account any offset added to the window
     (from PSsetXgcdrawable) or possible scaling (unlikely in X-windows,
     but what about other devices?) */
  y = NSHeight(window->xframe) - (y+h);

  values.function = GXcopy;
  values.plane_mask = AllPlanes;
  values.clip_mask = None;
  valuemask = (GCFunction | GCPlaneMask | GCClipMask);
  XChangeGC(XDPY, window->gc, valuemask, &values);

  xi = x;		// width/height seems
  yi = y;		// to require +1 pixel
  width = w + 1;	// to copy out
  height = h + 1;

  NSDebugLLog (@"NSWindow", 
	       @"copy X rect ((%d, %d), (%d, %d))", xi, yi, width, height);

  if (width > 0 || height > 0)
    XCopyArea (XDPY, window->buffer, window->ident, window->gc, 
	       xi, yi, width, height, xi, yi);

  XFlush(XDPY);
}

// handle X expose events
- (void) _processExposedRectangles: (int)win
{
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  if (window->type != NSBackingStoreNonretained)
    return;

  // Set the clipping path to the exposed rectangles
  // so that further drawing will not affect the non-exposed region
  XSetRegion (XDPY, window->gc, window->region);

  // We should determine the views that need to be redisplayed. Until we
  // fully support scalation and rotation of views redisplay everything.
  // FIXME: It seems wierd to trigger a front-end method from here...
  [[NSWindow _windowWithNumber: win] display];

  // Restore the exposed rectangles and the region
  [window->exposedRects removeAllObjects];
  XDestroyRegion (window->region);
  window->region = XCreateRegion();
  XSetClipMask (XDPY, window->gc, None);
}

- (void) DPScapturemouse: (int)win
{
  int ret;
  gswindow_device_t *window;

  window = NSMapGet(windowtags, (void *)win);
  if (!window)
    return;

  ret = XGrabPointer(XDPY, window->ident, False,
		     PointerMotionMask | ButtonReleaseMask | ButtonPressMask,
		     GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

  if (ret != GrabSuccess)
    NSLog(@"Failed to grab pointer\n");
  else
    {
      grab_window = window;
      NSDebugLLog(@"NSWindow", @"Grabbed pointer\n");
    }
}

- (void) DPSreleasemouse
{
  XUngrabPointer(XDPY, CurrentTime);
  grab_window = NULL;
}

- (void) DPSsetinputfocus: (int)win
{
  gswindow_device_t *window = NSMapGet(windowtags, (void*)win);
extern long currentFocusWindow;
extern long desiredFocusWindow;
extern long focusRequestNumber;

  if (window == 0)
    {
      NSDebugLLog(@"Focus", @"Setting focus to unknown win %d", win);
      return;
    }
  /*
   * If we have an outstanding request to set focus to this window,
   * we don't want to do it again.
   */
  if (win == desiredFocusWindow && focusRequestNumber != 0)
    {
      NSDebugLLog(@"Focus", @"Resetting focus to %d", window->ident);
    }
  else
    {
      NSDebugLLog(@"Focus", @"Setting focus to %d", window->ident);
    }
  desiredFocusWindow = win;
  focusRequestNumber = XNextRequest(XDPY);
  XSetInputFocus(XDPY, window->ident, RevertToParent, lastTime);
}

/* Cursor Ops */
typedef struct _xgps_cursor_id_t {
  Cursor c;
  BOOL   defined;
} xgps_cursor_id_t;

static char xgps_blank_cursor_bits [] = {
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static Cursor xgps_blank_cursor = None;
static BOOL   cursor_hidden = NO;

static void
xgps_set_cursor(Display *xdpy, Window appRootWindow, Cursor c, BOOL set)
{
  NSApplication *theApp = [NSApplication sharedApplication];
  NSWindow *w = [theApp mainWindow];
  Window window;

  if (w)
    window = ([XGContext _windowWithTag: [w windowNumber]])->ident;
  else
    window = appRootWindow;
  if (set)
    XDefineCursor(xdpy, window, c);
  else
    XUndefineCursor(xdpy, window);
}

- (void) DPShidecursor
{
  Pixmap shape, mask;
  XColor black, white;

  if (cursor_hidden)
    return;

  if (xgps_blank_cursor == None)
    {
      shape = XCreatePixmapFromBitmapData(XDPY, appRootWindow,
					  xgps_blank_cursor_bits, 
					  16, 16, 1, 0, 1);
      mask = XCreatePixmapFromBitmapData(XDPY, appRootWindow,
					 xgps_blank_cursor_bits, 
					 16, 16, 1, 0, 1);
      black.red = black.green = black.blue = 0;
      black = [self xColorFromColor: black];
      white.red = white.green = white.blue = 65535;
      white = [self xColorFromColor: white];
      
      xgps_blank_cursor = XCreatePixmapCursor(XDPY, shape, mask, 
					      &white, &black,  0, 0);
      XFreePixmap(XDPY, shape);
      XFreePixmap(XDPY, mask);
    }
  xgps_set_cursor(XDPY, appRootWindow, xgps_blank_cursor, YES);
  cursor_hidden = YES;
}

- (void) DPSshowcursor
{
  if (cursor_hidden)
    xgps_set_cursor(XDPY, appRootWindow, None, NO);
  cursor_hidden = NO;
}

- (void) DPSstandardcursor: (int)style : (void **)cid
{
  xgps_cursor_id_t *cursor;
  cursor = NSZoneMalloc([self zone], sizeof(xgps_cursor_id_t));
  switch (style)
    {
    case GSArrowCursor:
      cursor->c = XCreateFontCursor(XDPY, XC_left_ptr);     
      break;
    case GSIBeamCursor:
      cursor->c = XCreateFontCursor(XDPY, XC_xterm);
      break;
    default:
      cursor->c = XCreateFontCursor(XDPY, XC_left_ptr);     
      break;
    }
  cursor->defined = NO;
  if (cid)
    *cid = (void *)cursor;
}

- (void) DPSimagecursor: (float) hotx : (float) hoty : (float) w :  (float) h : (int)colors : (const char *)image : (void **)cid
{
}

- (void) DPSsetcursorcolor: (float)fr : (float)fg : (float)fb : (float)br : (float)bg : (float)bb : (void *)cid
{
  XColor xf, xb;
  xgps_cursor_id_t *cursor;

  cursor = (xgps_cursor_id_t *)cid;
  if (cursor->defined == NO)
    xgps_set_cursor(XDPY, appRootWindow, cursor->c, YES);

  xf.red   = 65535 * fr;
  xf.green = 65535 * fg;
  xf.blue  = 65535 * fb;
  xb.red   = 65535 * br;
  xb.green = 65535 * bg;
  xb.blue  = 65535 * bb;
  xf = [self xColorFromColor: xf];
  xb = [self xColorFromColor: xb];
  XRecolorCursor(XDPY, cursor->c, &xf, &xb);
}


@end

