/* xrtools - Color conversion routines and other low-level X support

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@gnu.org>
   Date: Oct 1998
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
   */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include "xrtools.h"

#undef SET_GLOBAL_COLORMAP

static Atom	_XA_GNUSTEP_RGB_MAP = 0;
static Atom	_XA_GNUSTEP_GRAY_RAMP = 0;

static Display *tmp_dpy;

/* Maintaining and using colormaps */
static int
xrAllocGrayScale(RContext* context, XStandardColormap *cmap,
		 unsigned long *colors, int ncolors)
{
  long i;
    
  if (!colors)
    return -1;
  for (i=0; i < ncolors; i++) 
    {
      XColor color;
      color.red = color.green = color.blue = i * 65535 / (ncolors-1);
      color.flags = DoRed | DoGreen | DoBlue;
      color.pixel = colors[i];
      XStoreColor(context->dpy, cmap->colormap, &color);
    }

  cmap->red_max = ncolors - 1;
  cmap->red_mult = 1;
  cmap->green_max = 0;
  cmap->green_mult = 0;
  cmap->blue_max = 0;
  cmap->blue_mult = 0;
  cmap->base_pixel = colors[0];
  return 0;
}

int
xrAllocPseudoColor(RContext* context, XStandardColormap *cmap,
		   unsigned long *colors, int ncolors)
{
  long i, cpc;
  if (!colors)
    return -1;

  cpc = context->attribs->colors_per_channel;
  if ((context->attribs->flags & RC_GammaCorrection) 
      && context->attribs->rgamma > 0
      && context->attribs->ggamma > 0 && context->attribs->bgamma > 0) 
    {
      int r, g, b;
      double rg, gg, bg;
      double tmp;
      XColor color;

      /* do gamma correction */
      i = 0;
      rg = 1.0/context->attribs->rgamma;
      gg = 1.0/context->attribs->ggamma;
      bg = 1.0/context->attribs->bgamma;
      for (r=0; r<cpc; r++) {
	for (g=0; g<cpc; g++) {
	  for (b=0; b<cpc; b++) {

	    color.red=(r*0xffff) / (cpc-1);
	    color.green=(g*0xffff) / (cpc-1);
	    color.blue=(b*0xffff) / (cpc-1);
	    color.flags = DoRed|DoGreen|DoBlue;

	    tmp = (double)color.red / 65536.0;
	    color.red = (unsigned short)(65536.0*pow(tmp, rg));

	    tmp = (double)color.green / 65536.0;
	    color.green = (unsigned short)(65536.0*pow(tmp, gg));

	    tmp = (double)color.blue / 65536.0;
	    color.blue = (unsigned short)(65536.0*pow(tmp, bg));
	    color.pixel = colors[i];
	    XStoreColor(context->dpy, cmap->colormap, &color);

	    i++;
	  }
	}
      }

    } 
  else 
    {
      int r, g, b;
      XColor color;
      i = 0;
      for (r=0; r<cpc; r++) {
	for (g=0; g<cpc; g++) {
	  for (b=0; b<cpc; b++) {
	    color.red=(r*0xffff) / (cpc-1);
	    color.green=(g*0xffff) / (cpc-1);
	    color.blue=(b*0xffff) / (cpc-1);
	    color.flags = DoRed|DoGreen|DoBlue;
	    color.pixel = colors[i];
	    XStoreColor(context->dpy, cmap->colormap, &color);
	    i++;
	  }
	}
      }
    }

  cmap->red_max = cpc-1;
  cmap->green_max = cpc-1;
  cmap->blue_max = cpc-1;
  cmap->red_mult = cpc*cpc;
  cmap->green_mult = cpc;
  cmap->blue_mult = 1;
  cmap->base_pixel = colors[0];
  return 0;
}

/* Find the offset for a given mask */
unsigned long
mask_offset(unsigned long mask)
{
  unsigned long number = 1;
  if (mask == 0)
    return 0;
  while ((number & mask) == 0)
    number = (number << 1L);
  return number;
}

static XStandardColormap *
xrGetStandardColormap(RContext* context, Atom map_type)
{
  int nitems_ret;
  XStandardColormap *cmap, *retmap;
  
  /* Do we already have a colormap ready? */
  retmap = NULL;
  if (XGetRGBColormaps(context->dpy,
                       RootWindow(context->dpy,context->screen_number),
                       &cmap,
                       &nitems_ret,
                       map_type))
    {
      int i;

      for (i = 0; i < nitems_ret; i++)
        if (cmap[i].visualid == context->visual->visualid)
          {
            retmap = XAllocStandardColormap();
            memcpy(retmap, &cmap[i], sizeof(XStandardColormap));
            break;
          }
      XFree(cmap);
    }
  return retmap;
}

static int
xrGetDefaultGrayMap(RContext* context)
{
  unsigned long *cells;
  XStandardColormap *cmap;
  
  /* Do we already have colormaps ready? */
  if (_XA_GNUSTEP_GRAY_RAMP == 0)
    _XA_GNUSTEP_GRAY_RAMP = XInternAtom(context->dpy, "RGB_GRAY_RAMP",
					False);
  if ((context->std_gray_map = xrGetStandardColormap(context, _XA_GNUSTEP_GRAY_RAMP))
    != NULL)
    {
      context->ngrays = context->std_gray_map->red_max + 1;
#ifdef DEBUG
      fprintf(stderr, "Found default graymap (%d colors)\n", context->ngrays);
#endif
      return 0;
    }
  
  /* No colormap found, create one */
  if (context->vclass == DirectColor || context->vclass == TrueColor)
    {
      /* Use the RGB map */
      context->ngrays = 0;
      return 0;
    }

  context->std_gray_map = XAllocStandardColormap();
  cmap = context->std_gray_map;

  /* See how many colors we can allocate */
  /*
  if (context->attribs->flags & RC_NumberOfGrays)
    context->ngrays = context->attribs->number_of_grays;
  else
  */
    context->ngrays = 17;
#ifdef DEBUG
  fprintf(stderr, "Trying graymap of %d colors\n", context->ngrays);
#endif
  if (context->vclass == GrayScale || context->vclass == PseudoColor)
    {
      int i, ngrays;

      ngrays = context->ngrays;
      /*
      if ((context->attribs->flags & RC_ColormapStyle) 
          && context->attribs->colormap_style == four_colormap)
	ngrays = 4;
      */
      cmap->colormap = DefaultColormap(context->dpy, 
                                       context->screen_number);
      i = 0;
      while (i == 0 && ngrays >= 4) 
        {
          cells = (unsigned long *)malloc(sizeof(unsigned long) * ngrays);
          i = XAllocColorCells(context->dpy, 
                               cmap->colormap,
                               TRUE,
                               NULL,
                               0,
                               cells,
                               ngrays);
          if (i == 0) 
            {
              free(cells);
              cells = NULL;
	      if (ngrays > 8)
		ngrays -= 4;
	      else
		ngrays -= 1;
            }
        }

      /*
      if (i == 0 && (context->attribs->flags & RC_PrivateColormap))
        {
          free(cells);
          cells = NULL;
          fprintf(stderr, "Warning (xraw): Creating private colormap\n");
          cmap->colormap = XCreateColormap(context->dpy, 
                       RootWindow(context->dpy, context->screen_number),
                       context->vinfo.visual,
                       AllocAll);
	  ngrays = context->ngrays;
        }
      else 
      */
      if (i == 0)
        {
          fprintf(stderr, "Warning (xraw): No colors. Using black/white\n");
	  /*
          context->attribs->flags |= RC_ColormapStyle;
          context->attribs->colormap_style = no_colormap;
	  */
          context->white = WhitePixel(context->dpy, context->screen_number);
          context->black = BlackPixel(context->dpy, context->screen_number);
          context->ngrays = 0;
	  XFree(context->std_gray_map);
	  context->std_gray_map = NULL;
          return -1;
        }
      context->ngrays = ngrays;
    }
  else
    context->ngrays = 0; /* Just use the RGB map */
      
  context->std_gray_map = cmap;
#ifdef DEBUG
  fprintf(stderr, "Allocated graymap of %d colors\n", context->ngrays);
#endif
  xrAllocGrayScale(context, context->std_gray_map, cells, context->ngrays);
  XFree(cells);

  if (context->std_gray_map->colormap != DefaultColormap(context->dpy, 
                                       context->screen_number))
    XSetWindowColormap(context->dpy, context->drawable, 
		       context->std_gray_map->colormap);

  /* Set our new colormap so other apps can use it */
#ifdef SET_GLOBAL_COLORMAP
  XSetRGBColormaps(tmp_dpy, 
		   RootWindow(tmp_dpy,context->screen_number),
		   context->std_gray_map,
		   1,
		   _XA_GNUSTEP_GRAY_RAMP);
#endif
  return 0;
}

static int
xrGetDefaultRGBmap(RContext* context)
{
  int cpc;
  unsigned long *cells;
  XStandardColormap *cmap;
  
  /* Do we already have colormaps ready? */
  /*
  _XA_GNUSTEP_RGB_MAP = (context->attribs->colormap_style == rgb_colormap) ? 
    XA_RGB_DEFAULT_MAP : XA_RGB_GRAY_MAP;
  */
  _XA_GNUSTEP_RGB_MAP = XA_RGB_DEFAULT_MAP;
  if ((context->std_rgb_map = xrGetStandardColormap(context, _XA_GNUSTEP_RGB_MAP)) 
      != NULL)
    {
      context->ncolors = (context->std_rgb_map->red_max + 1)
	* (context->std_rgb_map->green_max + 1)
	* (context->std_rgb_map->blue_max + 1);
#ifdef DEBUG
      fprintf(stderr, "Found default rgbmap\n");
#endif
      return 0;
    }

  /* No colormap found, create one */
  /*
  if ((context->attribs->flags & RC_ColormapStyle) 
      && context->attribs->colormap_style == four_colormap)
    {
      return -1;
    }
  */

  context->std_rgb_map = XAllocStandardColormap();
  cmap = context->std_rgb_map;
  if (context->std_gray_map)
    cmap->colormap = context->std_gray_map->colormap;
  else
    cmap->colormap = DefaultColormap(context->dpy, 
				     context->screen_number);
  if (context->vclass == DirectColor || context->vclass == TrueColor)
    {
      cmap->red_mult = mask_offset(context->visual->red_mask);
      cmap->green_mult = mask_offset(context->visual->green_mask);
      cmap->blue_mult = mask_offset(context->visual->blue_mask);
      cmap->red_max = context->visual->red_mask / cmap->red_mult;
      cmap->green_max = context->visual->green_mask / cmap->green_mult;
      cmap->blue_max = context->visual->blue_mask / cmap->blue_mult;
      cmap->base_pixel = 0;
      context->ncolors = context->visual->map_entries;
      return 0;
    }

  /* See how many colors we can allocate */
  if (context->attribs->flags & RC_ColorsPerChannel)
    cpc = context->attribs->colors_per_channel;
  else
    cpc = 5;
#ifdef DEBUG
  fprintf(stderr, "Trying RGB map of %d color cube\n", cpc);
#endif
  if (context->vclass == GrayScale || context->vclass == PseudoColor)
    {
      int i, ncolors;
      if (context->vclass == PseudoColor)
	ncolors = cpc * cpc * cpc;
      else
	ncolors = context->visual->map_entries / 2;

      i = 0;
      while (i == 0 && ncolors >= 4) 
        {
          cells = (unsigned long *)malloc(sizeof(unsigned long) * ncolors);
          i = XAllocColorCells(context->dpy, 
                               cmap->colormap,
                               TRUE,
                               NULL,
                               0,
                               cells,
                               ncolors);
          if (i == 0) 
            {
              free(cells);
              cells = NULL;
	      if (context->vclass == PseudoColor)
		{
		  cpc -= 1;
		  ncolors = cpc * cpc * cpc;
		}
	      else
		ncolors -= 4;
            }
        }

      if (i == 0)
	return -1;

      context->ncolors = ncolors;
    }
  else
    context->ncolors = context->visual->map_entries;
      
  context->std_rgb_map = cmap;
  context->attribs->flags |= RC_ColorsPerChannel;
  context->attribs->colors_per_channel = pow(context->ncolors, 0.3334);

#ifdef DEBUG
  fprintf(stderr, "Allocated RGB map of %d colors\n", context->ncolors);
#endif
  if (/*context->attribs->colormap_style == gray_colormap || */
      context->vclass == StaticGray
      || context->vclass == GrayScale)
    xrAllocGrayScale(context, context->std_rgb_map, cells, context->ncolors);
  else if (context->vclass == StaticColor
           || context->vclass == PseudoColor)
    xrAllocPseudoColor(context, context->std_rgb_map, cells, context->ncolors);
  XFree(cells);

  /* Set our new colormap so other apps can use it */
#ifdef SET_GLOBAL_COLORMAP
  XSetRGBColormaps(tmp_dpy, 
		   RootWindow(tmp_dpy,context->screen_number),
		   context->std_rgb_map,
		   1,
		   _XA_GNUSTEP_RGB_MAP);
#endif
  return 0;
}


int
xrGetDefaultColormap(RContext* context)
{
  if (context->std_rgb_map)
    {
      XFree(context->std_rgb_map);
      context->std_rgb_map = NULL;
    }
  context->ncolors = 0;
  if (context->std_gray_map)
    {
      XFree(context->std_gray_map);
      context->std_gray_map = NULL;
    }
  context->ngrays = 0;
  context->nextracolors = 0;

  /*
  if ((context->attribs->flags & RC_ColormapStyle) 
           && context->attribs->colormap_style == no_colormap)
    {
      context->white = WhitePixel(context->dpy, context->screen_number);
      context->ltgray = WhitePixel(context->dpy, context->screen_number);
      context->black = BlackPixel(context->dpy, context->screen_number);
      context->dkgray = BlackPixel(context->dpy, context->screen_number);
      return 0;
    }
  */

  /* Grab the server because we might want to set a new colormap */
#ifdef SET_GLOBAL_COLORMAP
#ifdef DEBUG
  fprintf(stderr, "Grabbing server to set colormaps\n");
#endif
  tmp_dpy = XOpenDisplay(DisplayString(context->dpy));
  XGrabServer(tmp_dpy);
#endif

  if (xrGetDefaultGrayMap(context) == 0)
    xrGetDefaultRGBmap(context);

  if (context->std_gray_map || context->std_rgb_map)
    {
      context->white = xrGrayToPixel(context, 1.000);
      context->black = xrGrayToPixel(context, 0.000);
    }

  
#ifdef SET_GLOBAL_COLORMAP
  XSetCloseDownMode(tmp_dpy, RetainTemporary);
  XUngrabServer(tmp_dpy);
  XCloseDisplay(tmp_dpy);
#ifdef DEBUG
  fprintf(stderr, "Ungrabbed server\n");
#endif
#endif
  return 0;
}

#define MAX_EXTRA_COLORS 256

unsigned long 
xrAllocActualRGB(RContext* context, float red, float green, float blue)
{
  Colormap cmap;
  XColor color;

  if (context->extra_colors == NULL)
    context->extra_colors = (XColor *)malloc(sizeof(XColor)*MAX_EXTRA_COLORS);

  if (context->nextracolors > MAX_EXTRA_COLORS)
    return -1;

  if (context->std_gray_map)
    cmap = context->std_gray_map->colormap;
  else if (context->std_rgb_map)
    cmap = context->std_rgb_map->colormap;
  else
    return -1;

  if ((context->attribs->flags & RC_GammaCorrection) 
      && context->attribs->rgamma > 0
      && context->attribs->ggamma > 0 && context->attribs->bgamma > 0) 
    {
      double rg, gg, bg;
      double tmp;
      XColor color;

      rg = 1.0/context->attribs->rgamma;
      gg = 1.0/context->attribs->ggamma;
      bg = 1.0/context->attribs->bgamma;
      color.flags = DoRed|DoGreen|DoBlue;
      color.red = (unsigned short)(65536.0*pow(tmp, rg));
      color.green = (unsigned short)(65536.0*pow(tmp, gg));
      color.blue = (unsigned short)(65536.0*pow(tmp, bg));
      color.pixel = 0;
    } 
  else 
    {
      color.flags = DoRed|DoGreen|DoBlue;
      color.red   = red * 65536.0;
      color.green = green * 65536.0;
      color.blue = blue * 65536.0;
      color.pixel = 0;
    }

  if (XAllocColor(context->dpy, cmap, &color))
    {
      context->extra_colors[context->nextracolors] = color;
      return context->nextracolors++;
    }

  return -1;
}

u_long
xrExactToPixel(RContext* context, float red, float green, float blue)
{
  int i;
  for (i = 0; i < context->nextracolors; i++)
    {
      if ((red == context->extra_colors[i].red)
	  && (green == context->extra_colors[i].green)
	  && (blue == context->extra_colors[i].blue)
	  )
	return context->extra_colors[i].pixel;
    }
  return 0;
}

/* Internal conversion of colors to pixels values */
u_long   
xrGrayToPixel(RContext* context, float gray)
{
  XStandardColormap *map;
  u_long color;

  if (context->nextracolors 
      && (color == xrExactToPixel(context, gray, gray, gray)))
    return color;
  
  if (context->ngrays)
    map = context->std_gray_map;
  else
    map = context->std_rgb_map;
  if (context->colors)
    {
      XColor cc;
      RColor rcolor;
      rcolor.red = 255 * gray;
      rcolor.green = 255 * gray;
      rcolor.blue = 255 * gray;
      rcolor.alpha = 0;
      RGetClosestXColor(context, &rcolor, &cc);
      color = cc.pixel;
    }
  else if (map == NULL) 
    {
      if (gray < 0.5)
	color = context->black;
      else
	color = context->white;
    }
  else
    color = ((u_long)(0.5 + (gray * map->red_max)) * map->red_mult)
      + ((u_long)(0.5 + (gray*map->green_max)) * map->green_mult)
      + ((u_long)(0.5 + (gray*map->blue_max)) * map->blue_mult)
      + map->base_pixel;
  return color;
}

/* FIXME: Need to handle gamma correction */
u_long   
xrRGBToPixel(RContext* context, float red, float green, float blue)
{
  XStandardColormap *map;
  u_long color;
  
  if (context->nextracolors 
      && (color == xrExactToPixel(context, red, green, blue)))
    return color;
  
  if (context->ncolors)
    map = context->std_rgb_map;
  else
    map = context->std_gray_map;
  if (context->colors)
    {
      XColor cc;
      RColor rcolor;
      rcolor.red = 255 * red;
      rcolor.green = 255 * green;
      rcolor.blue = 255 * blue;
      rcolor.alpha = 0;
      RGetClosestXColor(context, &rcolor, &cc);
      color = cc.pixel;
    }
  else if (map == NULL) 
    {
      int gray = ((0.3*red) + (0.59*green) + (0.11*blue));
      if (gray < 0.5)
	color = context->black;
      else
	color = context->white;
    }
  else
    color = ((u_long)(0.5 + (red * map->red_max)) * map->red_mult)
      + ((u_long)(0.5 + (green * map->green_max)) * map->green_mult)
      + ((u_long)(0.5 + (blue * map->blue_max)) * map->blue_mult)
      + map->base_pixel;
  return color;
}

u_long   
xrHSBToPixel(RContext* context, float h, float s, float v)
{
  int i;
  float f, p, q, t;
  float red, green, blue;

  h = fmod(h, 360);

  if (s == 0) 
    return xrRGBToPixel(context, v, v, v);
  i = fmod(h, 60);
  f = fmod(h, 60);
  p = v * (1.0 - s);
  q = v * (1.0 - s * f / 60);
  t = v * (1.0 - s * (60 - f) / 60);
  
  switch (i) 
    {
    case 0:
      red = v;
      green = t;
      blue = p;
      break;
    case 1:
      red = q;
      green = v;
      blue = p;
      break;
    case 2:
      red = p;
      green = v;
      blue = t;
      break;
    case 3:
      red = p;
      green = q;
      blue = v;
      break;
    case 4:
      red = t;
      green = p;
      blue = v;
      break;
    case 5:
      red = v;
      green = p;
      blue = q;
      break;
    }
    return xrRGBToPixel(context, red, green, blue);
}

/* Not implemented. FIXME */
u_long   
xrCMYKToPixel(RContext* context, float c, float m, float y, float k) 
{
    return 0;
}

u_long   
xrColorToPixel(RContext* context, xr_device_color_t  color)
{
  u_long pix;
  switch(color.space)
    {
    case gray_colorspace:
      pix = xrGrayToPixel(context, color.field[0]);
      break;
    case rgb_colorspace:
      pix = xrRGBToPixel(context, color.field[0], 
			 color.field[1], color.field[2]);
      break;
    case hsb_colorspace:
      pix = xrHSBToPixel(context, color.field[0], 
			 color.field[1], color.field[2]);
      break;
    case cmyk_colorspace: 
      pix = xrCMYKToPixel(context, color.field[0], color.field[1],
			  color.field[2], color.field[3]);
      break;
    default:
      break;
    }
    return pix;
}

xr_device_color_t 
xrConvertToGray(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = gray_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      new = color;
      break;
    case hsb_colorspace:
    case cmyk_colorspace: 
      color = xrConvertToRGB(color);
      /* NO BREAK */
    case rgb_colorspace:
      new.field[0] = 
	((0.3*color.field[0]) + (0.59*color.field[1]) + (0.11*color.field[2]));
      break;
    default:
      break;
    }
  return new;
}

xr_device_color_t 
xrConvertToRGB(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = rgb_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      new.field[0] = color.field[0];
      new.field[1] = color.field[0];
      new.field[2] = color.field[0];
      break;
    case rgb_colorspace:
      new = color;
      break;
    case hsb_colorspace: 
    case cmyk_colorspace: 
      break;
    default:
      break;
    }
  return new;
}

xr_device_color_t 
xrConvertToHSB(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = hsb_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      break;
    case rgb_colorspace:
      break;
    case hsb_colorspace: 
      new = color;
      break;
    case cmyk_colorspace: 
      break;
    default:
      break;
    }
  return new;
}

xr_device_color_t 
xrConvertToCMYK(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = gray_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      break;
    case rgb_colorspace:
      break;
    case hsb_colorspace:
      break;
    case cmyk_colorspace: 
      new = color;
      break;
    default:
      break;
    }
  return new;
}


