/* 
   NSDPSContext.m

   Encapsulation of Display Postscript contexts

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Scott Christley <scottc@net-community.com>
   Date: 1996
   
   This file is part of the GNUstep GUI Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <math.h>
#include <Foundation/NSString.h>
#include <Foundation/NSThread.h>
#include <Foundation/NSLock.h>
#include <Foundation/NSData.h>
#include <Foundation/NSDictionary.h>
#include <gnustep/xdps/NSDPSContext.h>
#include <gnustep/xdps/DPSOperators.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>
#include <X11/extensions/shape.h>

#include <AppKit/NSAffineTransform.h>
#include <AppKit/NSView.h>
#include <AppKit/NSWindow.h>

#define BOOL XWINDOWSBOOL
#include <DPS/dpsXclient.h>
#include <DPS/dpsXshare.h>
#undef BOOL

#include "PXKDrawingEngine.h"
#include "general.h"
#include "xdnd.h"

//
// DPS exceptions
//
NSString *DPSPostscriptErrorException = @"DPSPostscriptErrorException";
NSString *DPSNameTooLongException = @"DPSNameTooLongException";
NSString *DPSResultTagCheckException = @"DPSResultTagCheckException";
NSString *DPSResultTypeCheckException = @"DPSResultTypeCheckException";
NSString *DPSInvalidContextException = @"DPSInvalidContextException";
NSString *DPSSelectException = @"DPSSelectException";
NSString *DPSConnectionClosedException = @"DPSConnectionClosedException";
NSString *DPSReadException = @"DPSReadException";
NSString *DPSWriteException = @"DPSWriteException";
NSString *DPSInvalidFDException = @"DPSInvalidFDException";
NSString *DPSInvalidTEException = @"DPSInvalidTEException";
NSString *DPSInvalidPortException = @"DPSInvalidPortException";
NSString *DPSOutOfMemoryException = @"DPSOutOfMemoryException";
NSString *DPSCantConnectException = @"DPSCantConnectException";

#define XDPY (context->dpy)
#define XSCR (context->screen_number)

//
// Class variables
//
static BOOL GNU_CONTEXT_TRACED = NO;
static BOOL GNU_CONTEXT_SYNCHRONIZED = NO;

static BOOL                     xDndInitialized = NO;
static DndClass                 dnd;

DndClass xdnd(void)
{
  return dnd;                 // FIX ME rename with private desig
}

enum {
  A_COEFF = 0,
  B_COEFF,
  C_COEFF,
  D_COEFF,
  TX_CONS,
  TY_CONS
};

static NSString* windowName = nil;

static Atom	WM_STATE;

static NSEvent*
process_key_event(XEvent* xEvent, NSDPSContext* ctxt, NSEventType eventType);

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym, unsigned eventModifierFlags);

static unsigned
process_modifier_flags(unsigned int state);

Atom
GSActionForDragOperation(NSDragOperation op)
{
  Atom xaction;
  switch (op)
    {
    case NSDragOperationNone: xaction = None; break;
    case NSDragOperationCopy: xaction = dnd.XdndActionCopy; break;
    case NSDragOperationLink: xaction = dnd.XdndActionLink; break;
    case NSDragOperationGeneric: xaction = dnd.XdndActionCopy; break;
    case NSDragOperationPrivate: xaction = dnd.XdndActionPrivate; break;
    case NSDragOperationAll: xaction = dnd.XdndActionPrivate; break;
    }
  return xaction;
}

NSDragOperation 
GSDragOperationForAction(Atom xaction)
{
  NSDragOperation action;
  if (xaction == dnd.XdndActionCopy)
    action = NSDragOperationCopy;
  else if (xaction == dnd.XdndActionMove)
    action = NSDragOperationCopy;
  else if (xaction == dnd.XdndActionLink) 
    action = NSDragOperationLink;
  else if (xaction == dnd.XdndActionAsk) 
    action = NSDragOperationGeneric;
  else if (xaction == dnd.XdndActionPrivate) 
    action = NSDragOperationPrivate;
  else
    action = NSDragOperationNone;
  return action;
}


@interface NSEvent (WindowHack)
- (void) _patchLocation: (NSPoint)loc;
@end

@implementation NSEvent (WindowHack)
- (void) _patchLocation: (NSPoint)loc
{
  location_point = loc;
}
@end


@interface NSDPSContext (Private)
- (void) receivedEvent: (void*)data
                  type: (RunLoopEventType)type
                 extra: (void*)extra
               forMode: (NSString*)mode;
- (void) setupRunLoopInputSourcesForMode: (NSString*)mode;
- (NSDate*) timedOutEvent: (void*)data
                     type: (RunLoopEventType)type
                  forMode: (NSString*)mode;
@end

@implementation NSDPSContext

+ (void)initialize
{
  if (self == [NSDPSContext class])
    {
      // Set initial version
      [self setVersion: 1];

     GNU_CONTEXT_TRACED = NO;
     GNU_CONTEXT_SYNCHRONIZED = NO;
    }
}

- _initXContext
{
  Display  *dpy;
  int      screen_number;
  NSString *display_name;
  
  display_name = [context_info objectForKey: @"DisplayName"];
  if (display_name)
    dpy = XOpenDisplay([display_name cString]);
  else
    dpy = XOpenDisplay(NULL);

  /* Use the fact that the screen number is specified like an extension
     e.g. hostname:0.1 */
  screen_number = [[display_name pathExtension] intValue];

  if (dpy == NULL)
    {
      char *dname = XDisplayName([display_name cString]);
      NSLog(@"Unable to connect to X Server %s", dname);
      exit(1);
    }

  OBJC_MALLOC(context, RContext, 1);
  XDPY = dpy;
  XSCR = screen_number;
  context->black = BlackPixel(XDPY, XSCR);
  context->white = WhitePixel(XDPY, XSCR);

  /* Create a window for initial drawing */
  /* This window is never mapped */
  xAppRootWindow = XCreateSimpleWindow(XDPY, 
				      RootWindow(XDPY, XSCR),
				      0, 0, 100, 100, 1, 1,
				      context->black);

  return self;
}

//
// Initializing a Context
//
- init
{
  NSMutableData *data = [NSMutableData data];

  return [self initWithMutableData: data
	       forDebugging: NO
	       languageEncoding: dps_ascii
	       nameEncoding: dps_strings
	       textProc: NULL
	       errorProc: NULL];
}

- (id) initWithContextInfo: (NSDictionary *)info
{
  [super initWithContextInfo: info];

  is_screen_context = YES;
  //error_proc = errorProc;
  //text_proc = tProc;
  chained_parent = nil;
  chained_child = nil;

  [self _initXContext];

#if HAVE_DPS_DPSNXARGS_H    
  /* Make it possible for this client to start a DPS NX agent */
  XDPSNXSetClientArg(XDPSNX_AUTO_LAUNCH, (void *)True);
#else
  if (!XDPSExtensionPresent(XDPY)) {
    NSLog (@"DPS extension not in server!");
    exit (1);
  }
#endif

  // create the DPS context
  [self createDPSContext];

  /*
   * Make self the current context 'cos some of our initialization needs it.
   */
  [isa setCurrentContext: self];

  [self setupRunLoopInputSourcesForMode: NSDefaultRunLoopMode];
  [self setupRunLoopInputSourcesForMode: NSConnectionReplyMode];
  [self setupRunLoopInputSourcesForMode: NSModalPanelRunLoopMode];
  [self setupRunLoopInputSourcesForMode: NSEventTrackingRunLoopMode];

  WM_STATE = XInternAtom(XDPY, "WM_STATE", False);

  return self;
}

- initWithMutableData:(NSMutableData *)data
	 forDebugging:(BOOL)debug
     languageEncoding:(DPSProgramEncoding)langEnc
	 nameEncoding:(DPSNameEncoding)nameEnc
	     textProc:(DPSTextProc)tProc
	    errorProc:(DPSErrorProc)errorProc
{
  ASSIGN(context_data, data);
  [self initWithContextInfo: nil];
  return self;
}

- (void)dealloc
{
  DPSDestroySpace(DPSSpaceFromContext(dps_context));
  XFreeGC (XDPY, context->copy_gc);
  XDestroyWindow (XDPY, xAppRootWindow);
  [windowList release];

  [chained_child release];
  XCloseDisplay(XDPY);
  [super dealloc];
}

- (void) setupRunLoopInputSourcesForMode: (NSString*)mode
{
  int		xEventQueueFd = XConnectionNumber(context->dpy);
  NSRunLoop	*currentRunLoop = [NSRunLoop currentRunLoop];

#if defined(LIB_FOUNDATION_LIBRARY)
  {
    id fileDescriptor = [[[NSPosixFileDescriptor alloc]
	initWithFileDescriptor: xEventQueueFd]
	autorelease];

    // Invoke limitDateForMode: to setup the current
    // mode of the run loop (the doc says that this
    // method and acceptInputForMode: beforeDate: are
    // the only ones that setup the current mode).

    [currentRunLoop limitDateForMode: mode];

    [fileDescriptor setDelegate: self];
    [fileDescriptor monitorFileActivity: NSPosixReadableActivity];
  }
#elif defined(NeXT_PDO)
  {
    id fileDescriptor = [[[NSFileHandle alloc]
	initWithFileDescriptor: xEventQueueFd]
	autorelease];

    [[NSNotificationCenter defaultCenter] addObserver: self
	selector: @selector(activityOnFileHandle: )
	name: NSFileHandleDataAvailableNotification
	object: fileDescriptor];
    [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
	[NSArray arrayWithObject: mode]];
  }
#else
  [currentRunLoop addEvent: (void*)(gsaddr)xEventQueueFd
		      type: ET_RDESC
		   watcher: (id<RunLoopEvents>)self
		   forMode: mode];
#endif
}

#if LIB_FOUNDATION_LIBRARY
- (void) activity: (NSPosixFileActivities)activity
		posixFileDescriptor: (NSPosixFileDescriptor*)fileDescriptor
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
}
#elif defined(NeXT_PDO)
- (void) activityOnFileHandle: (NSNotification*)notification
{
  id fileDescriptor = [notification object];
  id runLoopMode = [[NSRunLoop currentRunLoop] currentMode];

  [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
	[NSArray arrayWithObject: runLoopMode]];
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
}
#endif

- (void) receivedEvent: (void*)data
                  type: (RunLoopEventType)type
                 extra: (void*)extra
               forMode: (NSString*)mode
{
  static NSRect		xFrame;
  static Time		timeOfLastClick = 0;
  static unsigned long	timeOfLastMotion = 0;
  static BOOL		switchingFocusBetweenOurWindows = NO;
  static int		clickCount = 1;
  static unsigned int	eventFlags;
  static unsigned int	last_XEvent_state = -1;
  static PXKWindow	*window = nil;
  static Window		lastXWin = None;
  static int		windowNumber;
  NSEvent		*e = nil;
  XEvent		xEvent;
  NSRect		new_frame;
  NSPoint		eventLocation;
  Window		xWin;
  PXKWindow		*w;
  NSEventType		eventType;
  int			count;
  BOOL			inTrackingLoop = (mode == NSEventTrackingRunLoopMode);

  while ((count = XPending(context->dpy)) > 0)
    {
      // loop and grab all of the events from the X queue
      while (count-- > 0)
	{
	  NSDebugLog(@"Get next XWindows event\n");
	  XNextEvent(context->dpy, &xEvent);
	  if (XDPSDispatchEvent(&xEvent))
	    continue;
	  switch (xEvent.type)
	    {
	      // mouse button events
	      case ButtonPress:
		NSDebugLog(@"ButtonPress: \
			    xEvent.xbutton.time %u timeOfLastClick %u \n",
			    xEvent.xbutton.time, timeOfLastClick);
	      // hardwired test for a double click
	      // default of 300 should be user set;
	      // under NS the windowserver does this
	      if (xEvent.xbutton.time < (unsigned long)(timeOfLastClick +300))
		clickCount++;
	      else
		clickCount = 1;		// reset the click count
	      timeOfLastClick = xEvent.xbutton.time;

	      eventType = (xEvent.xbutton.button == Button1 ?
			    NSLeftMouseDown : NSRightMouseDown);
	      if (xEvent.xbutton.state != last_XEvent_state)
		{
		  eventFlags = process_modifier_flags(xEvent.xbutton.state);
		  last_XEvent_state = xEvent.xbutton.state;
		}
	      // if pointer is grabbed use grab window
	      xWin = (grabWindow == 0) ? xEvent.xbutton.window : grabWindow;
	      if (xWin != lastXWin)
		{
		  window = [PXKWindow _windowForXWindow: xWin];
		  windowNumber = [window windowNumber];
		  lastXWin = xWin;
		}
	      xFrame = [window xFrame];
	      eventLocation.x = xEvent.xbutton.x;
	      eventLocation.y = xFrame.size.height-xEvent.xbutton.y;

	      // create NSEvent
	      e = [NSEvent mouseEventWithType: eventType
				     location: eventLocation
				modifierFlags: eventFlags
				    timestamp: (NSTimeInterval)xEvent.xbutton.time
				 windowNumber: windowNumber
				      context: self
				  eventNumber: xEvent.xbutton.serial
				   clickCount: clickCount
				     pressure: 1.0];
	      break;

	    case ButtonRelease:
	      eventType = (xEvent.xbutton.button == Button1 ?
			    NSLeftMouseUp : NSRightMouseUp);
	      if (xEvent.xbutton.state != last_XEvent_state)
		{
		  eventFlags = process_modifier_flags(xEvent.xbutton.state);
		  last_XEvent_state = xEvent.xbutton.state;
		}
	      // if pointer is grabbed use grab window
	      xWin = (grabWindow == 0) ? xEvent.xbutton.window : grabWindow;
	      if (xWin != lastXWin)
		{
		  window = [PXKWindow _windowForXWindow: xWin];
		  windowNumber = [window windowNumber];
		  lastXWin = xWin;
		}
	      xFrame = [window xFrame];
	      eventLocation.x = xEvent.xbutton.x;
	      eventLocation.y = xFrame.size.height- xEvent.xbutton.y;

	      e = [NSEvent mouseEventWithType: eventType
				     location: eventLocation
				modifierFlags: eventFlags
				    timestamp: (NSTimeInterval)xEvent.xbutton.time
				 windowNumber: windowNumber
				      context: self
				  eventNumber: xEvent.xbutton.serial
				   clickCount: clickCount
				     pressure: 1.0];
	      break;

	    case CirculateNotify:
	      // change to the stacking order
	      NSDebugLog(@"CirculateNotify\n");
	      break;

	    case CirculateRequest:
	      NSDebugLog(@"CirculateRequest\n");
	      break;

	    case ClientMessage:
	      {
		NSTimeInterval time;
		Atom protocols_atom;

		NSDebugLog(@"ClientMessage\n");
		w =[PXKWindow _windowForXWindow: xEvent.xclient.window];

		protocols_atom = XInternAtom(context->dpy,
		  "WM_PROTOCOLS", False);
		if (xEvent.xclient.message_type == protocols_atom)
		  {
		    // check if WM is asking us to close a window
		    Atom delete_win_atom;
		    delete_win_atom = XInternAtom(context->dpy,
						  "WM_DELETE_WINDOW", False);
		    if (xEvent.xclient.data.l[0] == delete_win_atom)
		      {
			[w performClose: NSApp];
		      }
		  }
		else if (xEvent.xclient.message_type == dnd.XdndEnter)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndEnter message\n");
		    source = XDND_ENTER_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0,0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingEnter
			        data1: source
			        data2: 0];
		    /* If this is a non-local drag, set the dragInfo */
		    if ([PXKWindow _windowForXWindow: source] == nil)
		      [[PXKDragView sharedDragView] setDragInfoFromEvent: e];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndPosition)
		  {
		    Window source;
		    Atom action;
		    NSDebugLLog(@"NSDragging", @"  XdndPosition message\n");
		    source = XDND_POSITION_SOURCE_WIN(&xEvent);
		    xFrame = [window xFrame];
		    eventLocation.x = XDND_POSITION_ROOT_X(&xEvent)
		      - NSMinX(xFrame);
		    eventLocation.y = XDND_POSITION_ROOT_Y(&xEvent)
		      - NSMinY(xFrame);
		    eventLocation.y = xFrame.size.height - eventLocation.y;
		    time = XDND_POSITION_TIME(&xEvent);
		    action = XDND_POSITION_ACTION(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: time
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingUpdate
			        data1: source
			        data2: GSDragOperationForAction(action)];
		    /* If this is a non-local drag, set the dragInfo */
		    if ([PXKWindow _windowForXWindow: source] == nil)
		      [[PXKDragView sharedDragView] setDragInfoFromEvent: e];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndStatus)
		  {
		    Window target;
		    Atom action;
		    NSDebugLLog(@"NSDragging", @"  XdndStatus message\n");
		    target = XDND_STATUS_TARGET_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    action = XDND_STATUS_ACTION(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingStatus
			        data1: target
			        data2: GSDragOperationForAction(action)];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndLeave)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndLeave message\n");
		    source = XDND_LEAVE_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingExit
			        data1: 0
			        data2: 0];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndDrop)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndDrop message\n");
		    source = XDND_DROP_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    time = XDND_DROP_TIME(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: time
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingDrop
			        data1: source
			        data2: 0];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndFinished)
		  {
		    Window target;
		    NSDebugLLog(@"NSDragging", @"  XdndFinished message\n");
		    target = XDND_FINISHED_TARGET_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingFinished
			        data1: target
			        data2: 0];
		  }
	
	      }
	      break;

	      case ColormapNotify:
		// colormap attribute
		NSDebugLog(@"ColormapNotify\n");
		break;

	      // the window has been resized, change the width and height
	      // and update the window so the changes get displayed
	      case ConfigureNotify:
		NSDebugLog(@"ConfigureNotify\n");
		new_frame.origin.x = (float)xEvent.xconfigure.x;
		new_frame.origin.y = (float)xEvent.xconfigure.y;
		new_frame.size.width = (float)xEvent.xconfigure.width;
		new_frame.size.height = (float)xEvent.xconfigure.height;
		NSDebugLog(@"New frame %f %f %f %f\n",
			    new_frame.origin.x, new_frame.origin.y,
			    new_frame.size.width, new_frame.size.height);
		NSDebugLog(@"border_width %d\n",
			    xEvent.xconfigure.border_width);
		w = [PXKWindow _windowForXWindow: xEvent.xconfigure.window];
		if (w != nil)
		  {
		    NSTimeInterval	t = (NSTimeInterval)timeOfLastMotion;
		    NSRect		xFrame = [w xFrame];
		    BOOL		resized = NO;
		    int			width = xEvent.xconfigure.width;
		    int			height = xEvent.xconfigure.height;

		    [w xSetFrameFromXFrame: new_frame];

		    if (xFrame.size.width != new_frame.size.width
		      || xFrame.size.height != new_frame.size.height)
		      {
			e = [NSEvent otherEventWithType: NSAppKitDefined
					       location: eventLocation
					  modifierFlags: eventFlags
					      timestamp: t
					   windowNumber: [w windowNumber]
						context: self
						subtype: GSAppKitWindowResized
						  data1: width
						  data2: height];
			resized = YES;
			[w sendEvent: e];
			e = nil;
		      }
		    if (xFrame.origin.x != new_frame.origin.x
		      || xFrame.origin.y != new_frame.origin.y)
		      {
			int	xScreen;
			int	h;
			int	x = xEvent.xconfigure.x;
			int	y = xEvent.xconfigure.y;
			Window	r;
			Window	qParent;
			Window	*qChildren;
			unsigned	qNum;

			xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
			h = DisplayHeight(XDPY, xScreen);

			XQueryTree(XDPY, [w xWindow],
			  &r, &qParent, &qChildren, &qNum);
			XFree(qChildren);

			if (qParent != r)
			  {
			    int		X;
			    int		Y;
			    int		b;
			    int		d;

			    /*
			     * get info relative to parent window in case the
			     * window manager has messed things up.
			     */
			    XGetGeometry(XDPY, [w xWindow], &r,
			      &x, &y, &width, &height, &b, &d);
			    /*
			     * Window has a title bar so have X server
			     * translate to display coordinates first,
			     * then convert to local.
			     */
			    XTranslateCoordinates(XDPY, qParent, 
			      xAppRootWindow, x, y, &X, &Y, &r);
			    x = X;
			    y = Y;
			  }

			y = h - (y + height);

			if (e != nil)
			  [event_queue addObject: e];
			e = [NSEvent otherEventWithType: NSAppKitDefined
					       location: eventLocation
					  modifierFlags: eventFlags
					      timestamp: t
					   windowNumber: [w windowNumber]
						context: self
						subtype: GSAppKitWindowMoved
						  data1: x
						  data2: y];
			[w sendEvent: e];
			e = nil;
		      }
                    if (resized)
                      [w update];
		  }
		NSDebugLog(@"after ConfigureNotify\n");
		break;

	      // same as ConfigureNotify but we get this event
	      // before the change has actually occurred
	      case ConfigureRequest:
		NSDebugLog(@"ConfigureRequest\n");
		break;

	      // a window has been created
	      case CreateNotify:
		NSDebugLog(@"CreateNotify\n");
		break;

	      // a window has been destroyed
	      case DestroyNotify:
		NSDebugLog(@"DestroyNotify\n");
		break;

	      // when the pointer enters a window
	      case EnterNotify:
		NSDebugLog(@"EnterNotify\n");
		break;

	      // when the pointer leaves a window
	      case LeaveNotify:
		NSDebugLog(@"LeaveNotify\n");
		break;

	      // the visibility of a window has changed
	      case VisibilityNotify:
		NSDebugLog(@"VisibilityNotify\n");
		break;

	      // a portion of the window has become visible and
	      // we must redisplay it
	      case Expose:
		{
		  XRectangle rectangle;
		  static PXKWindow *exposeWin;
		  static Window lastExposeXWin = None;

		  NSDebugLog(@"Expose\n");

		  if (xEvent.xbutton.window != lastExposeXWin)
		    {
		      lastExposeXWin = xEvent.xbutton.window;
		      exposeWin = [PXKWindow _windowForXWindow: lastExposeXWin];
		    }
		  rectangle.x = xEvent.xexpose.x;
		  rectangle.y = xEvent.xexpose.y;
		  rectangle.width = xEvent.xexpose.width;
		  rectangle.height = xEvent.xexpose.height;
		  [exposeWin _addExposedRectangle: rectangle];

		  if (xEvent.xexpose.count == 0)
		    [exposeWin _processExposedRectangles];
		  break;
		}

	      // keyboard focus entered a window
	      case FocusIn:
		NSDebugLog(@"FocusIn\n");
		[NSApp activateIgnoringOtherApps: YES];
		w = [PXKWindow _windowForXWindow: xEvent.xfocus.window];
		[w makeKeyWindow];	
		break;

	      // keyboard focus left a window
	      case FocusOut:
		NSDebugLog(@"FocusOut\n");
		if (1)
		  {
		    Window	fw;
		    int		rev;

		    /*
		     * See where the focus has moved to -
		     * If it has gone to 'none' or 'PointerRoot' then 
		     * it's not one of ours.
		     * If it has gone to our root window - use the icon window.
		     * If it has gone to a window - we see if it is one of ours.
		     */
		    XGetInputFocus(xEvent.xfocus.display, &fw, &rev);
		    if (fw == None || fw == PointerRoot)
		      {
			w = nil;
		      }
		    else if (fw == xAppRootWindow)
		      {
			w = iconWindow;
			XSetInputFocus(context->dpy, [w xWindow],
			  RevertToParent, CurrentTime);
		      }
		    else
		      {
			w = [PXKWindow _windowForXWindow: fw];
		      }
#if 0
		    fprintf(stderr, "Focus win:%x,%x,%x mode:%x detail:%x\n",
			    xEvent.xfocus.window, fw, w,
			    xEvent.xfocus.mode,
			    xEvent.xfocus.detail);
#endif
		    if (w == nil)
		      [NSApp deactivate]; 
		  }
		break;

	      case GraphicsExpose:
		NSDebugLog(@"GraphicsExpose\n");
		break;

	      case NoExpose:
		NSDebugLog(@"NoExpose\n");
		break;

	      // window is moved because of a change in the size of its parent
	      case GravityNotify:
		NSDebugLog(@"GravityNotify\n");
		break;

	      // a key has been pressed
	      case KeyPress:
		e = process_key_event (&xEvent, self, NSKeyDown);
		break;

	      // a key has been released
	      case KeyRelease:
		if (!context)
		e = process_key_event (&xEvent, self, NSKeyUp);
		break;

	      // reports the state of the keyboard when pointer or
	      // focus enters a window
	      case KeymapNotify:
		NSDebugLog(@"KeymapNotify\n");
		break;

	      // when a window changes state from ummapped to
	      // mapped or vice versa
	      case MapNotify:
		NSDebugLog(@"MapNotify\n");
		w = [PXKWindow _windowForXWindow: xEvent.xunmap.window];
		[(PXKWindow *)w _setVisible: YES];
		break;

	      // Window is no longer visible.
	      case UnmapNotify:
		NSDebugLog(@"UnmapNotify\n");
		w = [PXKWindow _windowForXWindow: xEvent.xunmap.window];
		[(PXKWindow *)w _setVisible: NO];
		break;

	      // like MapNotify but occurs before the request is carried out
	      case MapRequest:
		NSDebugLog(@"MapRequest\n");
		break;

	      // keyboard or mouse mapping has been changed by another client
	      case MappingNotify:
		NSDebugLog(@"MappingNotify\n");
		break;

	      case MotionNotify:
		{
		  BOOL	shouldMakeEvent = NO;

		  /*
		   * the mouse has moved regulate the translation of X motion
		   * events in order to avoid saturating the AppKit with
		   * unneeded NSEvents - needs fixing!
		   */
		  if ((inTrackingLoop && count == 1)
		    || (xEvent.xmotion.time > (unsigned)(timeOfLastMotion+10)))
		    {
		      shouldMakeEvent = YES;
		    }

		  timeOfLastMotion = xEvent.xmotion.time;
		  if (shouldMakeEvent)
		    {
		      unsigned int	state = xEvent.xmotion.state;

		      if (state & Button1Mask)
			{
			  eventType = NSLeftMouseDragged;
			}
		      else if (state & Button2Mask)
			{
			  eventType = NSRightMouseDragged;
			}
		      else
			{
			  eventType = NSMouseMoved;
			}

		      if (state != last_XEvent_state)
			{
			  eventFlags = process_modifier_flags(state);
			  last_XEvent_state = state;
			}

		      // if pointer is grabbed use grab window instead
		      xWin = (grabWindow == 0)
			? xEvent.xmotion.window : grabWindow;
		      if (xWin != lastXWin)
			{
			  window = [PXKWindow _windowForXWindow: xWin];
			  windowNumber = [window windowNumber];
			  lastXWin = xWin;
			  xFrame = [window xFrame];
			}

		      eventLocation = NSMakePoint(xEvent.xmotion.x,
				    xFrame.size.height - xEvent.xmotion.y);

		      e = [NSEvent mouseEventWithType: eventType
					     location: eventLocation
					modifierFlags: eventFlags
					    timestamp: (NSTimeInterval)timeOfLastMotion
					 windowNumber: windowNumber
					      context: self
					  eventNumber: xEvent.xmotion.serial
					   clickCount: 1
					     pressure: 1.0];
		    }

		  NSDebugLog(@"MotionNotify\n");
		  break;
		}

	      // a window property has changed or been deleted
	      case PropertyNotify:
		NSDebugLog(@"PropertyNotify\n");
                if (WM_STATE == xEvent.xproperty.atom)
		  {
                    switchingFocusBetweenOurWindows = YES;
		  }
		break;

	      // a client successfully reparents a window
	      case ReparentNotify:
		NSDebugLog(@"ReparentNotify\n");
		NSDebugLog(@"parent offset %f %f\n", xEvent.xreparent.x,
					      xEvent.xreparent.y);
		break;

	      // another client attempts to change the size of a window
	      case ResizeRequest:
		NSDebugLog(@"ResizeRequest\n");
		break;

	      // events dealing with the selection
	      case SelectionClear:
	      case SelectionNotify:
	      case SelectionRequest:
		NSDebugLog(@"Selection*\n");
		break;

	      // We shouldn't get here unless we forgot to trap an event above
	      default:
		NSLog(@"Received an untrapped event\n");
		break;
	    }
	  if (e)
	    [event_queue addObject: e];
	  e = nil;
	}
    }
}

	// TODO: correctly map key events to
	// NSEvents. This can be made when 16-bit
	// character strings will be available.
static NSEvent*
process_key_event (XEvent* xEvent, NSDPSContext* context, NSEventType eventType)
{
  char buf[256];
  int count;
  XComposeStatus compose;
  NSString *keys, *ukeys;
  KeySym keysym;
  NSPoint eventLocation;
  unsigned short keyCode;
  unsigned int eventFlags;
  NSEvent *event;
  NSApplication *theApp = [NSApplication sharedApplication];
  PXKWindow *window;
  NSRect xFrame = NSZeroRect;

  NSDebugLog(@"Process key event");

  eventFlags = process_modifier_flags(xEvent->xkey.state);

  count = XLookupString((XKeyEvent *)xEvent, buf, 256, &keysym, &compose);

  // Make sure that the string is properly terminated
  if (count > 255)
    buf[255] = '\0';
  else
    {
      if (count < 1)
	buf[0] = '\0';
      else
	buf[count] = '\0';
    }

  window = (PXKWindow*)[theApp keyWindow];
  if (window)
    xFrame = [window xFrame];

  eventLocation.x = xEvent->xbutton.x;
  eventLocation.y = xFrame.size.height - xEvent->xbutton.y;
  NSDebugLog (@"xLocation = (%d, %d), userLocation = (%f, %f)",
	      xEvent->xbutton.x, (int)(xFrame.size.height -
	      xEvent->xbutton.y), eventLocation.x, eventLocation.y);

  keyCode = process_key_code(xEvent, keysym, eventFlags);

  ukeys = [NSString stringWithCString: buf];
  keys = ukeys;			/* Stupid implementation (to be replaced) */

  event = [NSEvent keyEventWithType: eventType
			   location: eventLocation
		      modifierFlags: eventFlags
			  timestamp: (NSTimeInterval)xEvent->xkey.time
		       windowNumber: [window windowNumber]
			    context: context
			 characters: keys
        charactersIgnoringModifiers: ukeys
			  isARepeat: NO
			    keyCode: keyCode];

  return event;
}

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym, unsigned int eventModifierFlags)
{
  unsigned short keyCode = 0;

  if ((keysym == XK_Return) || (keysym == XK_KP_Enter)
      || (keysym == XK_Linefeed))
    {				// do nothing for now
      keyCode = 0x0d;
    }
  else if ((keysym >= XK_Shift_L) && (keysym <= XK_Hyper_R))
    {
      // The shift button is depressed.  This should have
      // already been detected in the process_modifier_flags
      // function.  Therefore, nothing is done here...
    }
  if ((keysym >= XK_F1) && (keysym <= XK_F35))
    {
      // if a function key was pressed
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;

      switch (xEvent->xkey.keycode)
	{
	  case XK_F1: keyCode = NSF1FunctionKey; break;
	  case XK_F2: keyCode = NSF2FunctionKey; break;
	  case XK_F3: keyCode = NSF3FunctionKey; break;
	  case XK_F4: keyCode = NSF4FunctionKey; break;
	  case XK_F5: keyCode = NSF5FunctionKey; break;
	  case XK_F6: keyCode = NSF6FunctionKey; break;
	  case XK_F7: keyCode = NSF7FunctionKey; break;
	  case XK_F8: keyCode = NSF8FunctionKey; break;
	  case XK_F9: keyCode = NSF9FunctionKey; break;
	  case XK_F10: keyCode = NSF10FunctionKey; break;
	  case XK_F11: keyCode = NSF11FunctionKey; break;
	  case XK_F12: keyCode = NSF12FunctionKey; break;
	  case XK_F13: keyCode = NSF13FunctionKey; break;
	  case XK_F14: keyCode = NSF14FunctionKey; break;
	  case XK_F15: keyCode = NSF15FunctionKey; break;
	  case XK_F16: keyCode = NSF16FunctionKey; break;
	  case XK_F17: keyCode = NSF17FunctionKey; break;
	  case XK_F18: keyCode = NSF18FunctionKey; break;
	  case XK_F19: keyCode = NSF19FunctionKey; break;
	  case XK_F20: keyCode = NSF20FunctionKey; break;
	  case XK_F21: keyCode = NSF21FunctionKey; break;
	  case XK_F22: keyCode = NSF22FunctionKey; break;
	  case XK_F23: keyCode = NSF23FunctionKey; break;
	  case XK_F24: keyCode = NSF24FunctionKey; break;
	  case XK_F25: keyCode = NSF25FunctionKey; break;
	  case XK_F26: keyCode = NSF26FunctionKey; break;
	  case XK_F27: keyCode = NSF27FunctionKey; break;
	  case XK_F28: keyCode = NSF28FunctionKey; break;
	  case XK_F29: keyCode = NSF29FunctionKey; break;
	  case XK_F30: keyCode = NSF30FunctionKey; break;
	  case XK_F31: keyCode = NSF31FunctionKey; break;
	  case XK_F32: keyCode = NSF32FunctionKey; break;
	  case XK_F33: keyCode = NSF33FunctionKey; break;
	  case XK_F34: keyCode = NSF34FunctionKey; break;
	  case XK_F35: keyCode = NSF35FunctionKey; break;

	  default: 			// do nothing
	}
    }
  else
    {
      if (keysym == XK_BackSpace)
	{
	  keyCode = NSBackspaceKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Delete)
	{
	  keyCode = NSDeleteFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Home)
	{
	  keyCode = NSHomeFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Left)
	{
	  keyCode = NSLeftArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Up)
	{
	  keyCode = NSUpArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Right)
	{
	  keyCode = NSRightArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Down)
	{
	  keyCode = NSDownArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Prior)
	{
	  keyCode = NSPrevFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Page_Up)
	{
	  keyCode = NSPageUpFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_Next)
	{
	  keyCode = NSNextFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Page_Down)
	{
	  keyCode = NSPageDownFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_End)
	{
	  keyCode = NSEndFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Begin)
	{
	  keyCode = NSBeginFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Select)
	{
	  keyCode = NSSelectFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Print)
	{
	  keyCode = NSPrintFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Execute)
	{
	  keyCode = NSExecuteFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Insert)
	{
	  keyCode = NSInsertFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Undo)
	{
	  keyCode = NSUndoFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Redo)
	{
	  keyCode = NSRedoFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Menu)
	{
	  keyCode = NSMenuFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Find)
	{
	  keyCode = NSFindFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Help)
	{
	  keyCode = NSHelpFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Break)
	{
	  keyCode = NSBreakFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Mode_switch)
	{
	  keyCode = NSModeSwitchFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Sys_Req)
	{
	  keyCode = NSSysReqFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_Scroll_Lock)
	{
	  keyCode = NSScrollLockFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Pause)
	{
	  keyCode = NSPauseFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Clear)
	{
	  keyCode = NSClearDisplayFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if ((keysym == XK_Shift_L) || (keysym == XK_Shift_R))
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSShiftKeyMask;
	}
      else if ((keysym == XK_Control_L) || (keysym == XK_Control_R))
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSControlKeyMask;
	}
      else if (keysym == XK_Alt_R)
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSAlternateKeyMask;
	}
      else if (keysym == XK_Alt_L)
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSCommandKeyMask;
	}
      else if (keysym == XK_Tab)
	{
	  keyCode = 0x09;
	}
    }
  // If the key press originated from the key pad
  if ((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
    {
      eventModifierFlags = eventModifierFlags | NSNumericPadKeyMask;

      switch (keysym)
	{
	  case XK_KP_F1:        keyCode = NSF1FunctionKey;         break;
	  case XK_KP_F2:        keyCode = NSF2FunctionKey;         break;
	  case XK_KP_F3:        keyCode = NSF3FunctionKey;         break;
	  case XK_KP_F4:        keyCode = NSF4FunctionKey;         break;
#ifndef NeXT
	  case XK_KP_Home:      keyCode = NSHomeFunctionKey;       break;
	  case XK_KP_Left:      keyCode = NSLeftArrowFunctionKey;  break;
	  case XK_KP_Up:        keyCode = NSUpArrowFunctionKey;    break;
	  case XK_KP_Right:     keyCode = NSRightArrowFunctionKey; break;
	  case XK_KP_Down:      keyCode = NSDownArrowFunctionKey;  break;
	  case XK_KP_Prior:     keyCode = NSPrevFunctionKey;       break;
//		case XK_KP_Page_Up:   keyCode = NSPageUpFunctionKey;     break;
	  case XK_KP_Next:      keyCode = NSNextFunctionKey;       break;
//		case XK_KP_Page_Down: keyCode = NSPageDownFunctionKey;   break;
	  case XK_KP_End:       keyCode = NSEndFunctionKey;        break;
	  case XK_KP_Begin:     keyCode = NSBeginFunctionKey;      break;
	  case XK_KP_Insert:    keyCode = NSInsertFunctionKey;     break;
	  case XK_KP_Delete:    keyCode = NSDeleteFunctionKey;     break;
#endif
	  default:  break;  /* Nothing to do */
	}
    }

  if (((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
    || ((keysym >= XK_space) && (keysym <= XK_asciitilde)))
    {
    }					// Not processed

  return keyCode;
}

// process_modifier_flags() determines which modifier keys (Command, Control,
// Shift,  and so forth) were held down while the event occured.
static unsigned int
process_modifier_flags(unsigned int state)
{
  unsigned int eventModifierFlags = 0;

  if (state & ControlMask)
    eventModifierFlags = eventModifierFlags | NSControlKeyMask;

  if (state & ShiftMask)
    eventModifierFlags = eventModifierFlags | NSShiftKeyMask;

  if (state & Mod1Mask)
    {
      eventModifierFlags = eventModifierFlags | NSAlternateKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod1Mask\n");
    }

  if (state & Mod2Mask)
    {
      eventModifierFlags = eventModifierFlags | NSCommandKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod2Mask\n");
    }

  if (state & Mod3Mask)
    {
      eventModifierFlags = eventModifierFlags | NSAlphaShiftKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod3Mask\n");
    }

  if (state & Mod4Mask)
    {
      eventModifierFlags = eventModifierFlags | NSHelpKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod4Mask\n");
    }

  if (state & Mod5Mask)
    {
      eventModifierFlags = eventModifierFlags | NSAlternateKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod5Mask\n");
    }

  return eventModifierFlags;
}

- (NSDate*) timedOutEvent: (void*)data
                     type: (RunLoopEventType)type
                  forMode: (NSString*)mode
{
  return nil;
}

//
// Testing the Drawing Destination
//
- (BOOL)isDrawingToScreen
{
  return is_screen_context;
}

- (NSDPSContext *)DPSContext
{
  return self;
}

//
// Controlling the Context
//
- (void)interruptExecution
{}

//- (void)notifyObjectWhenFinishedExecuting:(id <NSDPSContextNotification>)obj
//{}

- (void)resetCommunication
{}

- (void)wait
{
  DPSWaitContext (dps_context);
}


//
// Managing Returned Text and Errors
//
+ (NSString *)stringForDPSError:(const DPSBinObjSeqRec *)error
{
  return nil;
}

- (DPSErrorProc)errorProc
{
  return error_proc;
}

- (void)setErrorProc:(DPSErrorProc)proc
{
  error_proc = proc;
}

- (void)setTextProc:(DPSTextProc)proc
{
  text_proc = proc;
}

- (DPSTextProc)textProc
{
  return text_proc;
}

//
// Sending Raw Data
//
- (void)printFormat:(NSString *)format,...
{}

- (void)printFormat:(NSString *)format arguments:(va_list)argList
{}

- (void)writeData:(NSData *)buf
{}

- (void)writePostScriptWithLanguageEncodingConversion:(NSData *)buf
{}

//
// Managing Binary Object Sequences
//
- (void)awaitReturnValues
{}

- (void)writeBOSArray:(const void *)data
		count:(unsigned int)items
	       ofType:(DPSDefinedType)type
{}

- (void)writeBOSNumString:(const void *)data
		   length:(unsigned int)count
		   ofType:(DPSDefinedType)type
		    scale:(int)scale
{}

- (void)writeBOSString:(const void *)data
		length:(unsigned int)bytes
{}

- (void)writeBinaryObjectSequence:(const void *)data
			   length:(unsigned int)bytes
{}

- (void)updateNameMap
{}

//
// Managing Chained Contexts
//
- (void)setParentContext:(NSDPSContext *)parent
{
  chained_parent = parent;
}

- (void)chainChildContext:(NSDPSContext *)child
{
  if (child)
    {
      chained_child = [child retain];
      [child setParentContext: self];
    }
}

- (NSDPSContext *)childContext
{
  return chained_child;
}

- (NSDPSContext *)parentContext
{
  return chained_parent;
}

- (void)unchainContext
{
  if (chained_child)
    {
      [chained_child setParentContext: nil];
      [chained_child release];
      chained_child = nil;
    }
}

//
// Debugging Aids
//
+ (BOOL)areAllContextsOutputTraced
{
  return GNU_CONTEXT_TRACED;
}

+ (BOOL)areAllContextsSynchronized
{
  return GNU_CONTEXT_SYNCHRONIZED;
}

+ (void)setAllContextsOutputTraced:(BOOL)flag
{
  GNU_CONTEXT_TRACED = flag;
}

+ (void)setAllContextsSynchronized:(BOOL)flag
{
  GNU_CONTEXT_SYNCHRONIZED = flag;
}

- (BOOL)isOutputTraced
{
  return is_output_traced;
}

- (BOOL)isSynchronized
{
  return is_synchronized;
}

- (void)setOutputTraced:(BOOL)flag
{
  is_output_traced = flag;
  XDPSChainTextContext(dps_context, flag);
}

- (void)setSynchronized:(BOOL)flag
{
  is_synchronized = flag;
}

@end

static void
myDPSErrorProc (DPSContext ctxt, DPSErrorCode errCode, unsigned arg1, unsigned args)
{
  DPSDefaultErrorProc (ctxt, errCode, arg1, arg1);
}

//
// Methods for XWindows implementation
//
@implementation NSDPSContext (GNUstepXDPS)

+ (Display*) currentXDisplay
{
  return [(NSDPSContext*)[self currentContext] xDisplay];
}

- (Display*)xDisplay
{
  return XDPY;
}

- (DPSContext)xDPSContext
{
  return dps_context;
}

- (void)createDPSContext
{
  int x, y;
  char version[255];
  unsigned long valuemask;
  XGCValues values;
  XClassHint classhint; 

  /* Create a GC for the initial window */
  values.foreground = context->black;
  values.background = context->white;
  values.function = GXcopy;
  values.plane_mask = AllPlanes;
  values.clip_mask = None;
  valuemask = (GCFunction | GCPlaneMask | GCClipMask 
         | GCForeground|GCBackground);
  context->copy_gc = XCreateGC(XDPY, xAppRootWindow, valuemask, &values);

  if (windowName == nil) 
    {
      windowName = [[NSProcessInfo processInfo] processName];
    }
  classhint.res_name = (char*)[windowName cString];
  classhint.res_class = "PXKWindow";
  XSetClassHint(XDPY, xAppRootWindow, &classhint);

  /* Create the context if need be */
  if (!dps_context)
    {
      /* Pass None as the drawable argument; the program will execute correctly
         but will not render any text or graphics. */
      dps_context = XDPSCreateSimpleContext(XDPY, None, context->copy_gc, 0, 0,
				       DPSDefaultTextBackstop,
				       (DPSErrorProc)myDPSErrorProc, NULL);
      if (dps_context == NULL)
	{
	  NSLog(@"Could not connect to DPS\n");
	  NSLog(@"Trying again...\n");
       	  dps_context = XDPSCreateSimpleContext(XDPY, None, context->copy_gc, 0, 0,
					   DPSDefaultTextBackstop,
					   (DPSErrorProc)myDPSErrorProc, NULL);

	  if (dps_context == NULL)
	    {
	      NSLog(@"DPS is not available\n");
	      exit(1);
	    }
	}

      // Make it the active context
      DPSSetContext(dps_context);
      XDPSRegisterContext(dps_context, NO);

      // Use pass-through event handling
      XDPSSetEventDelivery(XDPY, dps_event_pass_through);
    }

  PSWinitcontext (XGContextFromGC (context->copy_gc), xAppRootWindow, 0, 0);
  PSWGetTransform (dps_context, ctm, invctm, &x, &y);
  PSWinitcontext (XGContextFromGC (context->copy_gc), None, 0, 0);
  PSWVersion(version);
  dps_version = [NSString stringWithCString: version];
  NSDebugLLog(@"NSDPSContext", @"Using DPS Version: %s\n", version);
  NSDebugLLog(@"NSDPSContext", @"DPS Default Matrix: [%f %f %f %f %f %f]\n", 
	ctm[0], ctm[1], ctm[2], ctm[3], ctm[4], ctm[5]);
  if (GSDebugSet(@"DPSNormalMatrix") == YES)
    {
      NSLog(@"Reseting default matrix\n");
      ctm[0] /= fabs(ctm[0]);
      ctm[3] /= fabs(ctm[3]);
      PSWSetMatrix(ctm);
    }
}

- (void) flush
{
  XFlush(XDPY);
}

- (Window) xDisplayRootWindow
{
  return RootWindow(XDPY, XSCR);
}

- (Window) xAppRootWindow
{
  return xAppRootWindow;
}

- (void) _localTransform: (float *)local_ctm inverse: (float *)local_inv
           offset: (NSPoint *)offset
{
  int x, y;
  if (local_ctm == NULL || local_inv == NULL)
    return;
  PSWGetTransform (dps_context, local_ctm, local_inv, &x, &y);
  if (offset)
    {
      offset->x = x;
      offset->y = y;
    }
}

- (NSPoint)userPointFromXPoint:(NSPoint)xPoint
{
  float lctm[6], linv[6];
  NSPoint offset, userPoint;

  [self _localTransform: lctm inverse: linv offset: &offset];
  //xPoint.x -= offset.x;
  //xPoint.y -= offset.y;
  userPoint.x = linv[A_COEFF] * xPoint.x + linv[C_COEFF] * xPoint.y
  		+ linv[TX_CONS];
  userPoint.y = linv[B_COEFF] * xPoint.x + linv[D_COEFF] * xPoint.y
                + linv[TY_CONS];
  return userPoint;
}

- (NSPoint)XPointFromUserPoint:(NSPoint)userPoint
{
  float lctm[6], linv[6];
  NSPoint offset, xPoint;

  [self _localTransform: lctm inverse: linv offset: &offset];
  xPoint.x = lctm[A_COEFF] * userPoint.x + lctm[C_COEFF] * userPoint.y
  		+  lctm[TX_CONS];
  xPoint.y = lctm[B_COEFF] * userPoint.x + lctm[D_COEFF] * userPoint.y
                +  lctm[TY_CONS];
  xPoint.x = floor (xPoint.x);
  xPoint.y = floor (xPoint.y);
  return xPoint;
}

- (NSSize)userSizeFromXSize:(NSSize)xSize
{
  float lctm[6], linv[6];
  NSSize userSize;

  [self _localTransform: lctm inverse: linv offset: NULL];
  userSize.width = linv[A_COEFF] * xSize.width + linv[C_COEFF] * xSize.height;
  userSize.height = linv[B_COEFF] * xSize.width + linv[D_COEFF] * xSize.height;
  userSize.height = fabs (userSize.height);
  return userSize;
}

- (NSSize)XSizeFromUserSize:(NSSize)userSize
{
  float lctm[6], linv[6];
  NSSize xSize;

  [self _localTransform: lctm inverse: linv offset: NULL];
  xSize.width = lctm[A_COEFF] * userSize.width + lctm[C_COEFF] * userSize.height;
  xSize.height = lctm[B_COEFF] * userSize.width + lctm[D_COEFF] * userSize.height;
  xSize.width = fabs (floor (xSize.width));
  xSize.height = fabs (floor (xSize.height));
  return xSize;
}

- (BOOL) _setFrame: (NSRect)rect forWindow: (int)winNum
{
  int			xScreen;
  NSAutoreleasePool	*arp;
  NSRect		xVal;
  NSRect		last;
  NSEvent		*event;
  NSDate		*limit;
  NSMutableArray	*tmpQueue;
  unsigned		pos;
  float			xdiff;
  float			ydiff;
  PXKWindow		*window;
  XSizeHints            siz_hints;
  NSRect		frame;
  BOOL                  resize = NO;
  BOOL                  move = NO;

  window = (PXKWindow*)[PXKWindow _windowWithTag: winNum];
  frame = [window frame];
  if (NSEqualRects(rect, frame) == YES)
    {
      return (NO);
    }
  if (NSEqualSizes(rect.size, frame.size) == NO)
    {
      resize = YES;
    }
  if (NSEqualPoints(rect.origin, frame.origin) == NO)
    {
      move = YES;
    }
  xdiff = rect.origin.x - frame.origin.x;
  ydiff = rect.origin.y - frame.origin.y;

  xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  xVal.origin.x = rect.origin.x;
  xVal.origin.y = rect.origin.y + rect.size.height;
  xVal.origin.y = (float)DisplayHeight(XDPY, xScreen) - xVal.origin.y;
  xVal.size.width = rect.size.width;
  xVal.size.height = rect.size.height;

  last = [window xFrame];

  XMoveResizeWindow (XDPY, [window xWindow],
    (int)xVal.origin.x, (int)xVal.origin.y,
    (int)xVal.size.width, (int)xVal.size.height);
  siz_hints.x = xVal.origin.x;
  siz_hints.y = xVal.origin.y;
  XSetNormalHints(XDPY, [window xWindow], &siz_hints);

  /*
   * Having told the X system to move the window, we must now
   * wait for it to actually do the job.
   */
  XSync(XDPY, 0);

  /*
   * Soak up any X events that have arrived.
   */
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];

  /*
   * Now massage all the events currently in the queue to make sure
   * mouse locations in our window are adjusted as necessary.
   */
  arp = [NSAutoreleasePool new];
  limit = [NSDate distantPast]; /* Don't wait for new events.   */
  tmpQueue = [NSMutableArray arrayWithCapacity: 8];
  for (;;)
    {
      NSEventType	type;

      event = DPSGetEvent(self, NSAnyEventMask, limit,
	NSEventTrackingRunLoopMode);
      if (event == nil)
	break;
      type = [event type];
      if (type == NSAppKitDefined && [event windowNumber] == winNum)
	{
	  GSAppKitSubtype	sub = [event subtype];

	  /*
	   *	Window movement or resize events for the window we are
	   *	watching are posted immediately, so they can take effect.
	   */
	  if (sub == GSAppKitWindowMoved || sub == GSAppKitWindowResized)
	    {
	      [window sendEvent: event];
              if (sub == GSAppKitWindowResized)
                [window update];
	    }
	  else
	    {
	      [tmpQueue addObject: event];
	    }
	}
      else if (type != NSPeriodic && type != NSLeftMouseDragged
	&& type != NSRightMouseDragged && type != NSMouseMoved)
	{
	  /*
	   * Save any events that arrive before our window is moved - excepting
	   * periodic events (which we can assume will be outdated) and mouse
	   * movement events (which might flood us).
	   */
	  [tmpQueue addObject: event];
	}
      if (NSEqualRects(rect, [window frame]) == YES)
	{
	  break;
	}
      if (NSEqualRects(last, [window xFrame]) == NO)
	{
	  NSDebugLog(@"From: %@\nWant %@\nGot %@",
	    NSStringFromRect(last),
	    NSStringFromRect(xVal),
	    NSStringFromRect([window xFrame]));
	  last = [window xFrame];
	}
    }
  /*
   * If we got any events while waiting for the window movement, we
   * may need to adjust their locations to match the new window position.
   */
  pos = [tmpQueue count];
  while (pos-- > 0) 
    {
      event = [tmpQueue objectAtIndex: pos];
      if ([event windowNumber] == winNum)
	{
	  NSPoint	loc = [event locationInWindow];

	  loc.x -= xdiff;
	  loc.y -= ydiff;
	  [event _patchLocation: loc];
	}
      DPSPostEvent(self, event, YES);
    }
  [arp release];
  /*
   * Failsafe - if X hasn't told us it has moved/resized the window, we
   * fake the notification.
   */
  if (NSEqualRects([window frame], rect) == NO)
    {
      NSEvent	*e;
      int	width = xVal.size.width;
      int	height = xVal.size.height;
      int	x = xVal.origin.x;
      int	y = xVal.origin.y;
      int	xScreen;
      int	h;
      Window	qRoot;
      Window	qParent;
      Window	*qChildren;
      unsigned	qNum;

      xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
      h = DisplayHeight(XDPY, xScreen);

      XQueryTree(XDPY, [window xWindow],
	&qRoot, &qParent, &qChildren, &qNum);

      xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
      h = DisplayHeight(XDPY, xScreen);

      if (qParent != qRoot)
	{
	  Window	*r;
	  int		X;
	  int		Y;
	  /*
	   * window has a title bar so have X server translate from display
	   * coordinates to parent window coordinates.
	   */
	  XTranslateCoordinates(XDPY, RootWindow(XDPY, xScreen),
	    [window xWindow], x, y, &X, &Y, &r);
	  xVal.origin.x = X;
	  xVal.origin.y = Y;
	}
      [window xSetFrameFromXFrame: xVal];
      if (resize)
	{
	  e = [NSEvent otherEventWithType: NSAppKitDefined
				 location: NSZeroPoint
			    modifierFlags: 0
				timestamp: 0
			     windowNumber: winNum
				  context: self
				  subtype: GSAppKitWindowResized
				    data1: width
				    data2: height];
	  [window sendEvent: e];
	}
      if (move)
	{
	  y = h - (y + height);

	  e = [NSEvent otherEventWithType: NSAppKitDefined
				 location: NSZeroPoint
			    modifierFlags: 0
				timestamp: 0
			     windowNumber: winNum
				  context: self
				  subtype: GSAppKitWindowMoved
				    data1: x
				    data2: y];
	  [window sendEvent: e];
	}
    }
  if (resize)
    {
      [window update];
    }
  return YES;
}

- (void) _orderWindow: (NSWindowOrderingMode)place
           relativeTo: (int)otherWin
            forWindow: (int)winNum
{
  PXKWindow	*window;
  Window	xWin;
  Display	*xDpy;
  XSizeHints	hints;
  NSRect	frame;
  int		xScrn;
  int		level;

  window = (PXKWindow*)[PXKWindow _windowWithTag: winNum];
  if (window == nil)
    return;

  xWin = [window xWindow];
  xDpy = [self xDisplay];
  frame = [window frame];
  level = [window level];
  xScrn = [(PXKScreen *)[NSScreen mainScreen] xScreen];

  hints.flags = PPosition | USPosition;
  hints.x = frame.origin.x;
  hints.y = DisplayHeight(xDpy, xScrn) - (frame.origin.y + frame.size.height);

  if (place != NSWindowOut)
    {
      /*
       * At this point, we should make sure that the current window-level
       * is set in the X server, so it can stack windows correctly.
       * Unfortunately, I don't know how to do this.
       */
    }

  switch (place)
    {
      case NSWindowBelow:
	XLowerWindow(xDpy, xWin);
	XMapWindow(xDpy, xWin);
	XSetNormalHints(xDpy, xWin, &hints);
	break;

      case NSWindowAbove:
	XMapRaised(xDpy, xWin);
	XSetNormalHints(xDpy, xWin, &hints);
	break;

      case NSWindowOut:
	XSetNormalHints(xDpy, xWin, &hints);
	XUnmapWindow(xDpy, xWin);
	break;
    }
  XSync(xDpy, 0);

  /*
   * Now do general window management stuff.
   */
  if (place == NSWindowOut)
    {
      if ([NSApp isActive])
	{
	  if ([window isKeyWindow])
	    {
	      [window resignKeyWindow];
	    }
	  if ([window isMainWindow])
	    {
	      [window resignMainWindow];
	    }
	}
    }
  else
    {
      [[window contentView] setNeedsDisplay: YES];
      [window update];
    }
  if ([windowList indexOfObjectIdenticalTo: window] == NSNotFound)
    {
      [windowList addObject: window];
    }
}

/*
 * Return mouse location in base coords ignoring the event loop
 */
- (void) DPSmouselocation: (float*)x :(float*)y
{
  Window	rootWin;
  Window	childWin;
  int		currentX;
  int		currentY;
  int		winX;
  int		winY;
  unsigned	mask;
  BOOL		ok;

  ok = XQueryPointer (context->dpy, [self xDisplayRootWindow],
    &rootWin, &childWin, &currentX, &currentY, &winX, &winY, &mask);
  if (ok)
    {
      int	xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];

      *x = currentX;
      *y = DisplayHeight(context->dpy, xScreen) - currentY;
    }
  else
    {
      *x = 0;
      *y = 0;
    }
}

- (NSEvent*) DPSGetEventMatchingMask: (unsigned)mask
                          beforeDate: (NSDate*)limit
                              inMode: (NSString*)mode
                             dequeue: (BOOL)flag
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
  return [super DPSGetEventMatchingMask: mask
                      beforeDate: limit
                          inMode: mode
                         dequeue: flag];
}

- (void) DPSDiscardEventsMatchingMask: (unsigned)mask
                          beforeEvent: (NSEvent*)limit
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
  [super DPSDiscardEventsMatchingMask: mask
                          beforeEvent: limit];


}

/* Drag and Drop */
- (void) _resetDragTypes: (NSArray*)types toWindow: (int)winNum
{
  int count, i;
  Atom *typelist;
  PXKWindow  *window;
  NSString *type;
  NSCountedSet *drag_set = [self _dragTypesForWindow: winNum];
  NSEnumerator *drag_enum = [drag_set objectEnumerator];

  window = (PXKWindow*)[PXKWindow _windowWithTag: winNum];
  count = [drag_set count];
  typelist = NSZoneMalloc([self zone], (count+1) * sizeof(Atom));
  i = 0;
  while ((type = [drag_enum nextObject]))
    {
      NSString *mime = [NSPasteboard mimeTypeForPasteboardType: type];
      NSDebugLog(@"  registering %@ on window\n", mime);
      typelist[i] = XInternAtom(XDPY, [mime cString], False);
      i++;
    }
  typelist[i] = 0;
  xdnd_set_dnd_aware(&dnd, [(PXKWindow*)window xWindow], typelist);
  NSZoneFree([self zone], typelist);
}

- (BOOL) _addDragTypes: (NSArray*)types toWindow: (int)winNum
{
  BOOL did_add;
  PXKWindow  *window;

  window = (PXKWindow*)[PXKWindow _windowWithTag: winNum];
  if (!xDndInitialized && window)
    {
      xDndInitialized = YES;
      xdnd_init(&dnd, XDPY);
      xdnd_set_dnd_aware(&dnd, [(PXKWindow*)window xWindow], NULL);
    }
  did_add = [super _addDragTypes: types toWindow: winNum];
  if (did_add)
    [self _resetDragTypes: types toWindow: winNum];
  return did_add;
}

- (BOOL) _removeDragTypes: (NSArray*)types fromWindow: (int)winNum
{
  BOOL did_add;

  did_add = [super _addDragTypes: types toWindow: winNum];
  if (did_add)
    [self _resetDragTypes: types toWindow: winNum];
  return did_add;
}

- (id <NSDraggingInfo>)_dragInfo
{
  return [PXKDragView sharedDragView];
}

- (void) _postExternalEvent: (NSEvent *)theEvent
{
  if ([theEvent subtype] == GSAppKitDraggingStatus)
    {
      Atom xaction;
      NSDragOperation action = [theEvent data2];
      xaction = GSActionForDragOperation(action);
      xdnd_send_status(&dnd, 
		       [theEvent data1],
		       [(PXKWindow *)[theEvent window] xWindow], 
		       (action != NSDragOperationNone),
		       0,
		       0, 0, 0, 0,
		       xaction);
    }
  else if ([theEvent subtype] == GSAppKitDraggingFinished)
    {
      xdnd_send_finished(&dnd, 
			 [theEvent data1],
			 [(PXKWindow *)[theEvent window] xWindow], 
			 0);
    }
  else
    {
      NSLog(@"Internal: unhandled post external event\n");
    }

}

@end



@interface PXKRawWindow : PXKWindow
@end

@implementation PXKRawWindow

- (BOOL) canBecomeMainWindow
{
  return NO;
}

- (BOOL) canBecomeKeyWindow
{
  return NO;
}

- (void) initDefaults
{
  [super initDefaults];
  menu_exclude = YES;           // Don't show in windows menu.
  is_released_when_closed = NO;
}

- (id) initWithContentRect: (NSRect)contentRect
		 styleMask: (unsigned int)aStyle
		   backing: (NSBackingStoreType)bufferingType
		     defer: (BOOL)flag
		    screen: (NSScreen*)aScreen
{
  XSetWindowAttributes winattrs;
  unsigned long valuemask;

  self = [super initWithContentRect: contentRect
			  styleMask: aStyle
			    backing: bufferingType
			      defer: flag
			     screen: aScreen];

  valuemask = (CWSaveUnder|CWOverrideRedirect);
  winattrs.save_under = True;
  winattrs.override_redirect = True;
  XChangeWindowAttributes ([NSDPSContext currentXDisplay], [self xWindow],
                            valuemask, &winattrs);
  [[NSApplication sharedApplication] removeWindowsItem: self];

  return self;
}
@end



@implementation PXKDragView

#define	DWZ	48

static	PXKDragView	*sharedDragView = nil;

+ (PXKDragView*) sharedDragView
{
  if (!sharedDragView)
    {
      NSRect			winRect = {{0, 0}, {DWZ, DWZ}};
      PXKRawWindow		*sharedDragWindow = [PXKRawWindow alloc];

      [sharedDragWindow initWithContentRect: winRect
				  styleMask: NSBorderlessWindowMask
				    backing: NSBackingStoreNonretained
				      defer: YES];

      sharedDragView = [PXKDragView new];
      [sharedDragWindow setContentView: sharedDragView];
      RELEASE(sharedDragView);
    }

  return sharedDragView;
}

- (void) _sendLocalEvent: (GSAppKitSubtype)subtype
		  action: (NSDragOperation)action
	        position: (NSPoint)eventLocation
	       timestamp: (NSTimeInterval)time
	        toWindow: (NSWindow *)dWindow
{
  NSEvent *e;
  NSGraphicsContext *context = GSCurrentContext();
  NSRect xFrame = [(PXKWindow *)dWindow xFrame];

  eventLocation = NSMakePoint(eventLocation.x - NSMinX(xFrame),
			      eventLocation.y - NSMinY(xFrame));
  eventLocation.y = xFrame.size.height - eventLocation.y;

  e = [NSEvent otherEventWithType: NSAppKitDefined
	  location: eventLocation
	  modifierFlags: 0
	  timestamp: time
	  windowNumber: [dWindow windowNumber]
	  context: context
	  subtype: subtype
	  data1: [(PXKWindow *)window xWindow]
	  data2: action];
  [dragWindow sendEvent: e];
}

- (void) setDragInfoFromEvent: (NSEvent *)event
{
  dragSource = nil;
  dragPasteboard = [[NSPasteboard pasteboardWithName: NSDragPboard] retain];
  dragPoint = [event locationInWindow];
  dragSequence = [event timestamp];
  dragWindow = [event window];
  dragMask = [event data2];
  dragLocal = NO;
}

- (void) drawRect: (NSRect)rect
{
  [dragCell drawWithFrame: rect inView: self];
}

- (void)dragImage:(NSImage *)anImage
               at:(NSPoint)viewLocation
           offset:(NSSize)initialOffset
            event:(NSEvent *)event
       pasteboard:(NSPasteboard *)pboard
           source:(id)sourceObject
        slideBack:(BOOL)slideFlag
{
  int i, count;
  NSArray *types;
  Display *xDisplay = [NSDPSContext currentXDisplay];

  if (!anImage)
    anImage = [NSImage imageNamed: @"common_Close"];

  if (!dragCell)
    {
      dragCell = [[NSCell alloc] initImageCell: anImage];
      [dragCell setBordered: NO];
    }
  [dragCell setImage: anImage];
  DESTROY(dragPasteboard);
  dragPasteboard = [pboard retain];
  dragSource = sourceObject;

  types = [pboard types];
  count = [types count];
  typelist = NSZoneMalloc([self zone], (count+1) * sizeof(Atom));
  for (i = 0; i < count; i++)
    {
      NSString *mime = [NSPasteboard mimeTypeForPasteboardType: 
		       [types objectAtIndex: i]];
      typelist[i] = XInternAtom(xDisplay, [mime cString], False);
    }
  typelist[count] = 0;

  [self mouseDown: event];
  NSZoneFree([self zone], typelist);
  typelist = NULL;
}

- (void)mouseDown: (NSEvent*)theEvent
{
  Display	*xDisplay = [NSDPSContext currentXDisplay];
  int		xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  unsigned int	eventMask = NSLeftMouseDownMask | NSLeftMouseUpMask
			    | NSLeftMouseDraggedMask | NSMouseMovedMask
			    | NSPeriodicMask | NSAppKitDefinedMask;
  NSPoint	point, lastPoint;
  NSDate	*theDistantFuture = [NSDate distantFuture];
  NSEventType	eventType;
  Window	xWindow = [window xWindow];
  Window	rootWindow;
  float		wx;
  float		wy;
  int		init = 0;
  NSRect	dragWindowXFrame;
  PXKWindow	*eWindow = [theEvent window];
  BOOL		dnd_aware = NO;
  Window        mouseWindow;
  Window	lastMouseWindow = None;
  Window	eXWindow = [eWindow xWindow];
  NSSize	imageSize;

  XColor	xg = [(PXKColor*)[NSColor greenColor] xColor];
  XColor	xb = [(PXKColor*)[NSColor blackColor] xColor];
  XColor	xw = [(PXKColor*)[NSColor whiteColor] xColor];

  rootWindow = RootWindow(xDisplay, xScreen);

  [self unregisterDraggedTypes];

  /* Need to lockFocus to do this. FIXME when image caching works */
  [self lockFocus];
  XShapeCombineMask(xDisplay, xWindow, ShapeBounding, 0, 0,
    [(PXKBitmapImageRep*)[[dragCell image] bestRepresentationForDevice: nil]
    xPixmapMask], ShapeSet);
  [self unlockFocus];

  /* convert point and lastPoint into base coordinates (NOT X Coordinates) */
  lastPoint = [[eWindow contentView] convertPoint: [theEvent locationInWindow]
					 fromView: nil];
  lastPoint = [eWindow convertBaseToScreen: lastPoint];
  imageSize = [[dragCell image] size];
  [window setFrame: NSMakeRect(lastPoint.x-imageSize.width/2, 
		    lastPoint.y-imageSize.height/2, 
		    imageSize.width, imageSize.height)
	   display: YES];
  [window orderFront: nil];
  //[window _captureMouse: self];
  dragWindowXFrame = [window xFrame];	// in X screen coordinates
  wx = dragWindowXFrame.origin.x;
  wy = dragWindowXFrame.origin.y;
  dragSequence = [theEvent timestamp];
  dragLocal = NO;

  [NSEvent startPeriodicEventsAfterDelay: 0.02 withPeriod: 0.03];

  while ((eventType = [theEvent type]) != NSLeftMouseUp)
    {
      if (eventType == NSAppKitDefined)
	{
	  GSAppKitSubtype	sub = [theEvent subtype];
	  switch (sub)
	    {
	    case GSAppKitDraggingStatus:
	      NSDebugLLog(@"NSDragging", @"got GSAppKitDraggingStatus\n");
	      if ([theEvent data1] == lastMouseWindow)
		{
		  /* Check the type of operation the target will perform
		     and change the cursor accordingly */
		  /* FIXME: Implement */
		}
	      break;

	    case GSAppKitDraggingFinished:
	      NSLog(@"Internal: got GSAppKitDraggingFinished out of seq\n");
	      break;

	    default:
	      NSLog(@"Internal: dropped NSAppKitDefined (%d) event\n", sub);
	      break;
	    }
	}
      else if (eventType != NSPeriodic)
	{
	  point = [theEvent locationInWindow];
	  if (init < 1)				// dump the first point
	    {					// because it is in coords
	      init++;				// of the dragging source
	      point = lastPoint;
	    }
	  else
	    {
  	      point = [[eWindow contentView] 
				convertPoint: [theEvent locationInWindow]
				    fromView: nil];
  	      point = [eWindow convertBaseToScreen: point];
	    }
	}
      else
	{
	  if (point.x != lastPoint.x || point.y != lastPoint.y)
	    {
	      BOOL oldDragLocal;
	      NSWindow *oldDragWindow;
	      NSDragOperation action;
	      Atom xAction;

	      /* Move drag window (point/lastPoint are reversed since
		 X Coordinates are flipped) */
	      wx += (point.x - lastPoint.x);
	      wy += (lastPoint.y - point.y);
	      XMoveWindow(xDisplay, xWindow, (int)wx, (int)wy);
	      lastPoint = point;

	      mouseWindow = _findXWindow(xDisplay, rootWindow, rootWindow,
		wx+imageSize.width+1, wy+imageSize.height/2);
	      oldDragWindow = dragWindow;
	      oldDragLocal = dragLocal;
	      dragWindow = [PXKWindow _windowForXWindow: mouseWindow];
	      dragLocal = (dragWindow) ? YES : NO;
	      dragPoint = lastPoint;
	      if (dragLocal)
		dragPoint = [dragWindow convertScreenToBase: dragPoint];
	      action = [dragSource 
		       draggingSourceOperationMaskForLocal: dragLocal];
	      xAction = GSActionForDragOperation(action);

	      if (mouseWindow != lastMouseWindow && dnd_aware)
		{
		  if (oldDragLocal)
		    [self _sendLocalEvent: GSAppKitDraggingExit
		                   action: action
		                 position: NSZeroPoint
		                timestamp: dragSequence
		                 toWindow: oldDragWindow];
		  else
		    xdnd_send_leave(&dnd, lastMouseWindow, xWindow);
		  lastMouseWindow = None;
		  dnd_aware = NO;

		  XDefineCursor(xDisplay, xWindow,
		    [(PXKCursor*)[NSCursor arrowCursor] xCursor]);
		  XRecolorCursor(xDisplay,
		    [(PXKCursor*)[NSCursor arrowCursor] xCursor], &xb,&xw);
		}

	      if (mouseWindow != rootWindow && mouseWindow != xWindow)
		{
		  if (mouseWindow == lastMouseWindow)
		    {
		      NSDebugLLog(@"NSDragging", @"sending dnd pos\n");
		      if (dragLocal)
			[self _sendLocalEvent: GSAppKitDraggingUpdate
		                   action: action
		                 position: NSMakePoint(wx+imageSize.width+1,
						       wy+imageSize.height/2)
		                timestamp: CurrentTime
		                 toWindow: dragWindow];
		      else
			xdnd_send_position(&dnd, mouseWindow, xWindow,
					 xAction, wx+imageSize.width+1, 
					 wy+imageSize.height/2, CurrentTime);
		    }
		  else
		    {
		      dnd_aware = xdnd_is_dnd_aware(&dnd, mouseWindow,
					    &dnd.dragging_version, typelist);
		      if (dnd_aware)
			{
			  XDefineCursor(xDisplay, xWindow,
				[(PXKCursor*)[NSCursor arrowCursor] xCursor]);
			  XRecolorCursor(xDisplay,
				[(PXKCursor*)[NSCursor arrowCursor] xCursor], &xg,&xb);
			  
			  NSDebugLLog(@"NSDragging", @"sending dnd enter/pos\n");
			  if (dragLocal)
			    [self _sendLocalEvent: GSAppKitDraggingEnter
		                   action: action
		                 position: NSMakePoint(wx+imageSize.width+1,
						       wy+imageSize.height/2)
		                timestamp: CurrentTime
		                 toWindow: dragWindow];
			  else
			    {
			      xdnd_send_enter(&dnd, mouseWindow, xWindow, 
					      typelist);
			      xdnd_send_position(&dnd, mouseWindow, xWindow,
						 xAction, wx+imageSize.width+1,
						 wy+imageSize.height/2, 
						 CurrentTime);
			    }
			  lastMouseWindow = mouseWindow;
			}
		    }
		}
	    }
	}
      theEvent = [NSApp nextEventMatchingMask: eventMask
				    untilDate: theDistantFuture
				       inMode: NSEventTrackingRunLoopMode
				      dequeue: YES];
    }

  [NSEvent stopPeriodicEvents];
  //[window _releaseMouse: self];
  [window orderOut: nil];
  XDefineCursor(xDisplay, xWindow,
    [(PXKCursor*)[NSCursor arrowCursor] xCursor]);
  XRecolorCursor(xDisplay, [(PXKCursor*)[NSCursor arrowCursor] xCursor],&xb,&xw);

  if (dnd_aware && mouseWindow != (Window) None)
    {
      NSDebugLLog(@"NSDragging", @"sending dnd drop\n");
      XSetSelectionOwner(xDisplay, dnd.XdndSelection, xWindow, CurrentTime);
      if (dragLocal)
	[self _sendLocalEvent: GSAppKitDraggingDrop
		       action: 0
		     position: NSZeroPoint
		    timestamp: CurrentTime
		     toWindow: dragWindow];
      else
	xdnd_send_drop (&dnd, lastMouseWindow, xWindow, CurrentTime);
    }
  else
    {
      if (!(mouseWindow=_findXWindow(xDisplay, rootWindow, rootWindow, wx, wy)))
	return;
      else
	{
	/*
	  id	browser = [NSApp delegate];	// browser, DO NOT rely on this
	  BOOL	exists = NO, is_dir = NO;	// FIX ME s/b moved to Workspac
	  NSFileManager *fm = [NSFileManager defaultManager];
	  BOOL altKeyDown = [theEvent modifierFlags] & NSAlternateKeyMask;
	  NSMutableString *s = AUTORELEASE([[NSMutableString alloc]
				initWithString: [browser path]]);

	  exists = [fm fileExistsAtPath: s isDirectory: &is_dir];
	  // if ALT key was held down user wants to change to dir
	  if ((exists) && (is_dir) && altKeyDown)
	    {
	      [s insertString: @"cd " atIndex: 0];
	      [s appendString: @"\n"];
	    }
	  // send path string to X window under the cursor
	  if (!_sendXString(xDisplay, mouseWindow, [s cString]))
	    NSLog(@"PXKDragWindow: error sending string to X window\n");
	  */
	}
    }
}

- (void) setImage: (NSImage *)anImage
{
  [dragCell setImage: anImage];

  [self lockFocus];
  [dragCell drawWithFrame: NSMakeRect(0,0,48,48) inView: self];
  [self unlockFocus];
}

/* NSDraggingInfo protocol */
- (NSWindow *)draggingDestinationWindow
{
  return dragWindow;
}

- (NSPoint)draggingLocation
{
  return dragPoint;
}

- (NSPasteboard *)draggingPasteboard
{
  return dragPasteboard;
}

- (int)draggingSequenceNumber
{
  return dragSequence;
}

- (id)draggingSource
{
  return dragSource;
}

- (NSDragOperation)draggingSourceOperationMask
{
  if (dragSource)
    return [dragSource draggingSourceOperationMaskForLocal: dragLocal];
  else
    return dragMask;
}

- (NSImage *)draggedImage
{
  if (dragSource)
    return [dragCell image];
  else
    return nil;
}

- (NSPoint)draggedImageLocation
{
  NSPoint loc;
  if (dragSource)
    {
      NSSize size;
      size = [[dragCell image] size];
      loc = NSMakePoint(dragPoint.x-size.width/2, dragPoint.y - size.height/2);
    }
  else
    loc = dragPoint;
  return loc;
}

- (void)slideDraggedImageTo:(NSPoint)screenPoint
{
}

@end

