/* 
   NSDPSContext.h

   Encapsulation of Display Postscript contexts

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Scott Christley <scottc@net-community.com>
   Date: 1996
   
   This file is part of the GNUstep GUI Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#ifndef _GNUstep_H_NSDPSContext
#define _GNUstep_H_NSDPSContext

/* Define this to avoid including redefinitions of ps functions introduced
   by NSGraphicsContext */
#define _PSOperators_h_INCLUDE

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <Foundation/NSObject.h>
#include <AppKit/AppKit.h>
#include <AppKit/NSGraphicsContext.h>
#include <gnustep/xdps/DPSOperators.h>
#include <stdarg.h>

/* HACKS
 *
 * For the time being, we still need to have several backend classes use the
 * poseAs: mechanism.
 * For tidyness, we declare the class interfaces here rather than in separate
 * header files.
 * As the backend classes are removed, their interfaces will be removed too.
 */

@interface PXKColor : NSColor
{
}
- (XColor)xColor;
- (void)setXColor:(XColor)xcolor;
@end

@interface PXKCursor : NSCursor
{
}

- (void)setXCursor:(Cursor)cursor;
- (Cursor)xCursor;
- (BOOL)xCursorIsValid;

@end

@interface PXKScreen : NSScreen
{
}
- (int)xScreen;
@end

@interface	PXKWindow : NSWindow
{
}
+ (PXKWindow*)_windowForXWindow:(Window)xWindow;
+ (NSWindow*)_windowWithTag:(int)windowNumber;

- (void)setContentViewSize:(NSSize)aSize;

- (NSRect)xFrame;
- (Window)xWindow;
- (GC)xGC;
- (void)xSetFrameFromXFrame:(NSRect)rect;
- (void)_setVisible:(BOOL)flag;
- (void)_updateWindowGeometry;
- (Drawable)xDrawable;
- (void)_createBackingStore;

//
// Dealing with expose events
//
- (void)_addExposedRectangle:(XRectangle)rectangle;
- (void)_processExposedRectangles;

@end

@interface PXKBitmapImageRep : NSBitmapImageRep
{
}

- (Pixmap) xPixmapMask;

@end

@interface	PXKDragView : NSView <NSDraggingInfo>
{
  
  NSCell         *dragCell;
  NSPasteboard   *dragPasteboard;
  NSPoint         dragPoint;
  int             dragSequence;
  id              dragSource;
  NSDragOperation dragMask;
  id              dragWindow;
  Atom           *typelist;

  BOOL            dragLocal;
}

+ (PXKDragView*) sharedDragView;

- (void) setDragInfoFromEvent: (NSEvent *)event;
@end


/*
 *	Declare all the backend classes here.
 */

@class  PXKApplication;
@class  PXKCachedImageRep;
@class  PXKEPSImageRep;
@class  PXKColor;
@class  PXKFont;
@class  PXKFontManager;
@class  PXKImage;
@class  PXKMenuWindow;

@class NSData;
@class NSMutableData;
@class NSAffineTransform;

typedef struct RContext {
    Display *dpy;
    int screen_number;
    Colormap cmap;
    Region viewclip;
    
    GC copy_gc;

    Visual *visual;
    int depth;
    Window drawable;		       /* window to pass for XCreatePixmap().*/
				       /* generally = root */
    int vclass;
    
    unsigned long black;
    unsigned long white;
    
    int red_offset;		       /* only used in 24bpp */
    int green_offset;
    int blue_offset;

    /* only used for pseudocolor and grayscale */

    XStandardColormap *std_rgb_map;    /* standard RGB colormap */
    XStandardColormap *std_gray_map;   /* standard grayscale colormap */
    
    int ncolors;		       /* total number of colors we can use */
    int ngrays;
    XColor *colors;		       /* internal colormap */

    /* Extra colors */
    XColor *extra_colors;
    int nextracolors;

    struct {
	unsigned int use_shared_pixmap:1;
    } flags;
} RContext;

//
// NSDPSContext class interface
//
@interface NSDPSContext : NSGraphicsContext
{
  DPSContext dps_context;
  BOOL is_screen_context;
  DPSErrorProc error_proc;
  DPSTextProc text_proc;
  NSDPSContext *chained_parent;
  NSDPSContext *chained_child;
  BOOL is_output_traced;
  BOOL is_synchronized;
  float ctm[6], invctm[6];
  NSString *dps_version;
  Window grabWindow;
  PXKWindow *iconWindow;

@public
  RContext *context;
  Window xAppRootWindow;
  NSMutableArray *windowList;
}

//
// Initializing a Context
//
- initWithMutableData:(NSMutableData *)data
	 forDebugging:(BOOL)debug
     languageEncoding:(DPSProgramEncoding)langEnc
	 nameEncoding:(DPSNameEncoding)nameEnc
	     textProc:(DPSTextProc)tProc
	    errorProc:(DPSErrorProc)errorProc;

- (NSDPSContext *)DPSContext;

//
// Controlling the Context
//
- (void)interruptExecution;
//- (void)notifyObjectWhenFinishedExecuting:(id <NSDPSContextNotification>)obj;
- (void)resetCommunication;

//
// Managing Returned Text and Errors
//
+ (NSString *)stringForDPSError:(const DPSBinObjSeqRec *)error;
- (DPSErrorProc)errorProc;
- (void)setErrorProc:(DPSErrorProc)proc;
- (void)setTextProc:(DPSTextProc)proc;
- (DPSTextProc)textProc;

//
// Sending Raw Data
//
- (void)printFormat:(NSString *)format,...;
- (void)printFormat:(NSString *)format arguments:(va_list)argList;
- (void)writeData:(NSData *)buf;
- (void)writePostScriptWithLanguageEncodingConversion:(NSData *)buf;

//
// Managing Binary Object Sequences
//
- (void)awaitReturnValues;
- (void)writeBOSArray:(const void *)data
		count:(unsigned int)items
	       ofType:(DPSDefinedType)type;
- (void)writeBOSNumString:(const void *)data
		   length:(unsigned int)count
		   ofType:(DPSDefinedType)type
		    scale:(int)scale;
- (void)writeBOSString:(const void *)data
		length:(unsigned int)bytes;
- (void)writeBinaryObjectSequence:(const void *)data
			   length:(unsigned int)bytes;
- (void)updateNameMap;

//
// Managing Chained Contexts
//
- (void)chainChildContext:(NSDPSContext *)child;
- (NSDPSContext *)childContext;
- (NSDPSContext *)parentContext;
- (void)unchainContext;

//
// Debugging Aids
//
+ (BOOL)areAllContextsOutputTraced;
+ (BOOL)areAllContextsSynchronized;
+ (void)setAllContextsOutputTraced:(BOOL)flag;
+ (void)setAllContextsSynchronized:(BOOL)flag;
- (BOOL)isOutputTraced;
- (BOOL)isSynchronized;
- (void)setOutputTraced:(BOOL)flag;
- (void)setSynchronized:(BOOL)flag;

@end

@interface NSDPSContext (GNUstepXDPS)

+ (Display*) currentXDisplay;
- (Display *)xDisplay;
- (DPSContext)xDPSContext;
- (void)createDPSContext;
- (Window)xDisplayRootWindow;
- (Window)xAppRootWindow;

- (NSPoint) userPointFromXPoint: (NSPoint)xPoint;
- (NSPoint) XPointFromUserPoint: (NSPoint)userPoint;
- (NSSize) userSizeFromXSize: (NSSize)xSize;
- (NSSize) XSizeFromUserSize: (NSSize)userSize;

@end

#endif /* _GNUstep_H_NSDPSContext */
