/* 
   PXKBitmapImageRep.m

   NSBitmapImageRep for GNUstep GUI X/DPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Adam Fedor <fedor@colorado.edu>
   Author:  Scott Christley <scottc@net-community.com>
   Date: Feb 1996
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <stdlib.h>
#include <tiff.h>

#include <Foundation/NSLock.h>
#include <Foundation/NSData.h>

#include <gnustep/xdps/PXKBitmapImageRep.h>
#include <gnustep/xdps/PXKWindow.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKScreen.h>

static	char *hex = "0123456789abcdef";
#define	PUTHEX(cont, c)	DPSPrintf(cont, "%c%c", hex[((c)>>4)&0xf], hex[(c)&0xf]);

//
// Class variables
//
static BOOL gnustep_xdps_bitmap_dict;
static int gnustep_xdps_bitmap_cache_num;
static NSLock *gnustep_xdps_bitmap_lock;

//
// Backend structure for PXKBitmapImageRep
//
typedef struct _PXKBitmapImageRep_struct
{
  BOOL is_cached;
  int cache_num;
} PXKBitmapImageRep_struct;

#define PXKCACHE (((PXKBitmapImageRep_struct *)back_end_reserved)->is_cached)
#define PXKNUM (((PXKBitmapImageRep_struct *)back_end_reserved)->cache_num)

#if 0
void
PSDataColorSeparate(NSMutableData* tif, uint32 w, uint32 h, 
		    int nc, int samplesperpixel)
{
  uint32 row;
  int breaklen = MAXLINE, cc, s, maxs;
  unsigned char *tf_buf;
  unsigned char *cp, c;
  tsize_t tf_bytesperrow;

  (void) w;
  tf_bytesperrow = TIFFScanlineSize(tif);
  tf_buf = (unsigned char *) _TIFFmalloc(tf_bytesperrow);
  if (tf_buf == NULL) {
    NSLog(@"No space for scanline buffer");
    return;
  }
  maxs = (samplesperpixel > nc ? nc : samplesperpixel);
  for (row = 0; row < h; row++) {
    for (s = 0; s < maxs; s++) {
      if (TIFFReadScanline(tif, tf_buf, row, s) < 0)
	break;
      for (cp = tf_buf, cc = 0; cc < tf_bytesperrow; cc++) {
	DOBREAK(breaklen, 1);
	c = *cp++;
	PUTHEX(c);
      }
    }
  }
  _TIFFfree((char *) tf_buf);
}
#endif

@implementation PXKBitmapImageRep

//
// Class methods
//
+ (void) initialize
{
  if (self == [PXKBitmapImageRep class])
    {
      // Register ourselves
      [self registerImageRepClass: [self class]];

      // Allocate the cache number lock
      gnustep_xdps_bitmap_lock = [[NSRecursiveLock alloc] init];
      gnustep_xdps_bitmap_cache_num = 1;

      // We need to create the postscript dictionary
      gnustep_xdps_bitmap_dict = NO;
    }
}

//
// Instance methods
//

//
// Initialization
//
- (id) initWithBitmapDataPlanes: (unsigned char **)planes
		pixelsWide: (int)width
		pixelsHigh: (int)height
		bitsPerSample: (int)bps
		samplesPerPixel: (int)spp
		hasAlpha: (BOOL)alpha
		isPlanar: (BOOL)isPlanar
		colorSpaceName: (NSString *)colorSpaceName
		bytesPerRow: (int)rowBytes
		bitsPerPixel: (int)pixelBits;
{
  [super initWithBitmapDataPlanes: planes
	 pixelsWide: width
	 pixelsHigh: height
	 bitsPerSample: bps
	 samplesPerPixel: spp
	 hasAlpha: alpha
	 isPlanar: isPlanar
	 colorSpaceName: colorSpaceName
	 bytesPerRow: rowBytes
	 bitsPerPixel: pixelBits];

  // Allocate the back-end structure
  back_end_reserved = malloc(sizeof(PXKBitmapImageRep_struct));
  PXKCACHE = NO;
  PXKNUM = 0;

  return self;
}

- (void)dealloc
{
  // Free the back-end structure
  free(back_end_reserved);

  [super dealloc];
}

//
// Create the postscript dictionary
//
- (void)createBitmapCacheDictionary
{
  DPSContext ctxt;

  // Don't recreate it if its already been created
  if (!gnustep_xdps_bitmap_dict)
    {
      ctxt = DPSGetCurrentContext();
      DPSPrintf(ctxt, "/PXKBitmapImages 100 dict def\n");
      gnustep_xdps_bitmap_dict = YES;
    }
}

//
// RGB color image with color values separate
// Use colorimage postscript operator
//
- (BOOL)drawColorSeparate
{
  return YES;
}

//
// RGB color image with color values not separate
// Use colorimage postscript operator
//
- (BOOL)drawColorContig
{
  uint32 row, col, i, j, pos;
  const unsigned char *tf_buf = [imageData bytes];
  unsigned char dot;
  float adjust;
  DPSContext ctxt;

  // Create the cache dictionary if necessary
  [self createBitmapCacheDictionary];

  ctxt = DPSGetCurrentContext();

  // If image isn't cached then put in cache dictionary
  if (!PXKCACHE)
    {
      // Get next cache number
      [gnustep_xdps_bitmap_lock lock];
      PXKNUM = gnustep_xdps_bitmap_cache_num++;
      [gnustep_xdps_bitmap_lock unlock];

      // Name of image in cache dictionary is the
      // word 'image' followed by the unique cache number
      DPSPrintf(ctxt, "PXKBitmapImages begin\n");
      DPSPrintf(ctxt, "/image%d ", PXKNUM);

      // Write the data
      DPSPrintf(ctxt, "<");
      j = 0;
      pos = 0;
      for (row = 0; row < size.height; row++)
	for (col = 0; col < size.width; col++)
	  for (i = 0; i < numColors; i++)
	    {
	      if (hasAlpha)
		{
		  adjust = tf_buf[pos + (3 - (pos % 4))];
		  adjust = adjust/255;
		  NSDebugLog(@"alpha %f", adjust);
		}
	      else
		adjust = 1;
	      if ((hasAlpha) && ((pos % 4) != 3))
		{
		  dot = 255 + (tf_buf[pos] - 255)*adjust;
		  PUTHEX(ctxt, dot);
		  NSDebugLog(@"%c%c", hex[((dot)>>4)&0xf], 
			     hex[(dot)&0xf]);
		  ++j;
		}
	      ++pos;
	      if (j == 33)
		{
		  //DPSPrintf(ctxt, "\n");
		  NSDebugLog(@"\n");
		  j = 0;
		}
	    }
      DPSPrintf(ctxt, "> def\n");
      DPSPrintf(ctxt, "end\n");

      // The image is cached!
      PXKCACHE = YES;
    }

  // Draw the image from the cache
  if (PXKCACHE)
    {
      DPSPrintf(ctxt, "%f %f scale\n", size.width, size.height);

      DPSPrintf(ctxt, "%d %d 8\n", (int)size.width, (int)size.height);
      DPSPrintf(ctxt, "[%lu 0 0 -%lu 0 %lu]\n", (unsigned long)size.width, 
		(unsigned long)size.height, (unsigned long)size.height);

      DPSPrintf(ctxt, "PXKBitmapImages begin\n");
      DPSPrintf(ctxt, "image%d false 3 colorimage\n", PXKNUM);
      DPSPrintf(ctxt, "end\n");

      return YES;
    }
  else
    return NO;
}

- (BOOL) draw
{
  NSDebugLog(@"PXKBitmapImageRep: draw\n");

  NSDebugLog(@"PXKBitmapImageRep: color space %s\n", [_colorSpace cString]);
  NSDebugLog(@"PXKBitmapImageRep: size %f %f\n", size.width, size.height);
  NSDebugLog(@"%d %d %d %d\n", bitsPerSample, _pixelsWide, 
	     _pixelsHigh, hasAlpha);
  NSDebugLog(@"%d %d %d %d\n", bytesPerRow, numColors, 
	     bitsPerPixel, compression);

  if ([_colorSpace compare: NSDeviceRGBColorSpace] == NSOrderedSame)
    {
      if (_isPlanar)
	[self drawColorSeparate];
	  //	  PSColorSeparatePreamble(fd, w, h, 3);
	  //	  PSDataColorSeparate(imageData, _pixelsWide, _pixelsHigh, 
	  //			      3, numColors);
      else
	[self drawColorContig];
    }

#if 0
  switch (space) {
	case PHOTOMETRIC_RGB:
		if (planarconfiguration == PLANARCONFIG_CONTIG) {
			fprintf(fd, "%s", RGBcolorimage);
			PSColorContigPreamble(fd, w, h, 3);
			PSDataColorContig(fd, tif, w, h, 3);
		} else {
			PSColorSeparatePreamble(fd, w, h, 3);
			PSDataColorSeparate(fd, tif, w, h, 3);
		}
		break;
	case PHOTOMETRIC_SEPARATED:
		/* XXX should emit CMYKcolorimage */
		if (planarconfiguration == PLANARCONFIG_CONTIG) {
			PSColorContigPreamble(fd, w, h, 4);
			PSDataColorContig(fd, tif, w, h, 4);
		} else {
			PSColorSeparatePreamble(fd, w, h, 4);
			PSDataColorSeparate(fd, tif, w, h, 4);
		}
		break;
	case PHOTOMETRIC_PALETTE:
		fprintf(fd, "%s", RGBcolorimage);
		PhotoshopBanner(fd, w, h, 1, 3, "false 3 colorimage");
		fprintf(fd, "/scanLine %ld string def\n",
		    (long) ps_bytesperrow);
		fprintf(fd, "%lu %lu 8\n",
		    (unsigned long) w, (unsigned long) h);
		fprintf(fd, "[%lu 0 0 -%lu 0 %lu]\n",
		    (unsigned long) w, (unsigned long) h, (unsigned long) h);
		fprintf(fd, "{currentfile scanLine readhexstring pop} bind\n");
		fprintf(fd, "false 3 colorimage\n");
		PSDataPalette(fd, tif, w, h);
		break;
	case PHOTOMETRIC_MINISBLACK:
	case PHOTOMETRIC_MINISWHITE:
		PhotoshopBanner(fd, w, h, 1, 1, "image");
		fprintf(fd, "/scanLine %ld string def\n",
		    (long) ps_bytesperrow);
		fprintf(fd, "%lu %lu %d\n",
		    (unsigned long) w, (unsigned long) h, bitspersample);
		fprintf(fd, "[%lu 0 0 -%lu 0 %lu]\n",
		    (unsigned long) w, (unsigned long) h, (unsigned long) h);
		fprintf(fd,
		    "{currentfile scanLine readhexstring pop} bind\n");
		fprintf(fd, "image\n");
		PSDataBW(fd, tif, w, h);
		break;
	}
#endif

  return YES;
}

@end
