/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright(C) 2000, 2001 Edgar Denny(e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <ctype.h>

#include "food.h"
#include "recipe.h"
#include "plan.h"
#include "base_win.h"
#include "recipe_win.h"
#include "food_win.h"
#include "nutr_comp_dlg.h"
#include "food_srch_dlg.h"
#include "nutr_goal_dlg_ui.h"

typedef struct Food_View_s {
	GtkButton *clear_button;
	GtkButton *compute_button;
	GtkButton *goals_button;
	GtkButton *search_button;

	GtkMenuItem *exit;
	GtkMenuItem *clear;
	GtkMenuItem *meal_plan;
	GtkMenuItem *recipe;
	GtkMenuItem *food;
	GtkMenuItem *nutr_goal;
	GtkMenuItem *about;

	GtkCombo *combo;

	GtkEntry *food_entry;
	GtkEntry *amount_entry;
	GtkEntry *combo_entry;

	GnomeAppBar *appbar;

	GtkWidget *menubar_container;
	GtkWidget *toolbar_container;
	GtkWidget *appbar_container;
	GtkWidget *notebook_container;
	GtkWidget *table_container;
} Food_View_t;


static GladeXML *load_xml( void);
static void connect_signals( Food_View_t *);

static Food_View_t *create_food_view( void);
static Food_View_t *get_food_view( void);

/* callbacks. */
static void on_clear_button_released( GtkButton *, gpointer);
static void on_clear_button_enter( GtkButton *, gpointer);
static void on_clear_button_leave( GtkButton *, gpointer);
static void on_compute_button_released( GtkButton *, gpointer);
static void on_compute_button_enter( GtkButton *, gpointer);
static void on_compute_button_leave( GtkButton *, gpointer);
static void on_goals_button_released( GtkButton *, gpointer);
static void on_goals_button_enter( GtkButton *, gpointer);
static void on_goals_button_leave( GtkButton *, gpointer);
static void on_search_button_released( GtkButton *, gpointer);

static void on_exit_activate( GtkMenuItem *, gpointer);
static void on_clear_activate( GtkMenuItem *, gpointer);
static void on_meal_plan_activate( GtkMenuItem *, gpointer);
static void on_recipe_activate( GtkMenuItem *, gpointer);
static void on_food_activate( GtkMenuItem *, gpointer);
static void on_nutr_goal_activate( GtkMenuItem *, gpointer);
static void on_about_activate ( GtkMenuItem *, gpointer);

/* load the glade xml. */
static GladeXML *
load_xml()
{
	GladeXML *xml = NULL;

	xml = glade_xml_new( GNUTRITION_GLADEDIR "/food_win.glade", NULL);

	if ( !xml) {
		g_log( "Gnutrition", G_LOG_LEVEL_ERROR,
			"cannot load glade file: food_win.glade\n");
		return NULL;
	}
	return xml;
}

static Food_View_t *
create_food_view()
{
	static Food_View_t *view = NULL;
	static GladeXML *xml = NULL;

	if ( !xml) xml = load_xml();

	if ( !view) {
		view = (Food_View_t *)g_malloc( sizeof( Food_View_t ));

		view->clear_button = GTK_BUTTON( glade_xml_get_widget( xml, 
			"clear_button"));
		view->compute_button = GTK_BUTTON( glade_xml_get_widget( xml,
			"compute_button"));
		view->goals_button = GTK_BUTTON( glade_xml_get_widget( xml, 
			"goals_button"));
		view->search_button = GTK_BUTTON( glade_xml_get_widget( xml, 
			"search_button"));

		view->exit = GTK_MENU_ITEM( glade_xml_get_widget( xml, 
			"exit"));
		view->clear = GTK_MENU_ITEM( glade_xml_get_widget( xml, 
			"clear"));
		view->meal_plan = GTK_MENU_ITEM( glade_xml_get_widget( xml, 
			"meal_plan"));
		view->recipe = GTK_MENU_ITEM( glade_xml_get_widget( xml, 
			"recipe"));
		view->food = GTK_MENU_ITEM( glade_xml_get_widget( xml, 
			"food"));
		view->nutr_goal = GTK_MENU_ITEM( glade_xml_get_widget( xml, 
			"nutr_goal"));
		view->about = GTK_MENU_ITEM( glade_xml_get_widget( xml, 
			"about"));

		view->combo = GTK_COMBO( glade_xml_get_widget( xml, "combo"));

		view->food_entry = GTK_ENTRY( glade_xml_get_widget( xml, 
			"food_entry"));
		view->amount_entry = GTK_ENTRY( glade_xml_get_widget( xml, 
			"amount_entry"));
		view->combo_entry = GTK_ENTRY( glade_xml_get_widget( xml, 
			"combo_entry"));

		view->appbar = GNOME_APPBAR( glade_xml_get_widget( xml, 
			"appbar"));

		view->menubar_container = glade_xml_get_widget( xml, 
			"menubar_container");
		view->toolbar_container = glade_xml_get_widget( xml, 
			"toolbar_container");
		view->notebook_container = glade_xml_get_widget( xml, 
			"notebook_container");
		view->table_container = glade_xml_get_widget( xml, 
			"table_container");
		view->appbar_container = glade_xml_get_widget( xml, 
			"appbar_container");

		connect_signals( view);
	}
	return view;
}

static Food_View_t *
get_food_view()
{
	return create_food_view();
}

/* connect the signals. */
static void
connect_signals( Food_View_t *view)
{
	gtk_signal_connect( GTK_OBJECT( view->clear_button), "released", 
		GTK_SIGNAL_FUNC( on_clear_button_released), (gpointer)view);
	gtk_signal_connect( GTK_OBJECT( view->clear_button), "enter", 
		GTK_SIGNAL_FUNC( on_clear_button_enter), (gpointer)view);
	gtk_signal_connect(GTK_OBJECT( view->clear_button), "leave", 
		GTK_SIGNAL_FUNC( on_clear_button_leave), (gpointer)view);
			
	gtk_signal_connect( GTK_OBJECT( view->compute_button), "released", 
		GTK_SIGNAL_FUNC( on_compute_button_released), (gpointer)view);
	gtk_signal_connect( GTK_OBJECT( view->compute_button), "enter", 
		GTK_SIGNAL_FUNC( on_compute_button_enter), (gpointer)view);
	gtk_signal_connect(GTK_OBJECT( view->compute_button), "leave", 
		GTK_SIGNAL_FUNC( on_compute_button_leave), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->goals_button), "released", 
		GTK_SIGNAL_FUNC( on_goals_button_released), (gpointer)view);
	gtk_signal_connect( GTK_OBJECT( view->goals_button), "enter", 
		GTK_SIGNAL_FUNC( on_goals_button_enter), (gpointer)view);
	gtk_signal_connect(GTK_OBJECT( view->goals_button), "leave", 
		GTK_SIGNAL_FUNC( on_goals_button_leave), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->search_button), "released", 
		GTK_SIGNAL_FUNC( on_search_button_released), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->exit), "activate", 
		GTK_SIGNAL_FUNC( on_exit_activate), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->clear), "activate", 
		GTK_SIGNAL_FUNC( on_clear_activate), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->meal_plan), "activate", 
		GTK_SIGNAL_FUNC( on_meal_plan_activate), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->recipe), "activate", 
		GTK_SIGNAL_FUNC( on_recipe_activate), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->food), "activate", 
		GTK_SIGNAL_FUNC( on_food_activate), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->nutr_goal), "activate", 
		GTK_SIGNAL_FUNC( on_nutr_goal_activate), (gpointer)view);

	gtk_signal_connect( GTK_OBJECT( view->about), "activate", 
		GTK_SIGNAL_FUNC( on_about_activate), (gpointer)view);
}

/* return the menubar widget. */
GtkWidget *
gnutr_get_food_win_menubar()
{
	Food_View_t *view = get_food_view();
	return view->menubar_container;
}

/* return the toolbar widget. */
GtkWidget *
gnutr_get_food_win_toolbar()
{
	Food_View_t *view = get_food_view();
	return view->toolbar_container;
}

/* return the main table widget. */
GtkWidget *
gnutr_get_food_win_notebook_container()
{
	Food_View_t *view = get_food_view();
	return view->notebook_container;
}

/* return the main table widget. */
GtkWidget *
gnutr_get_food_win_table()
{
	Food_View_t *view = get_food_view();
	return view->table_container;
}

/* return the appbar widget. */
GtkWidget *
gnutr_get_food_win_appbar()
{
	Food_View_t *view = get_food_view();
	return view->appbar_container;
}

/* callback when the "Clear" button is released. */
static void
on_clear_button_released( GtkButton *button,
                          gpointer   data)
{
	Food_View_t *view;
	GList *list = NULL;
	view = (Food_View_t *)data;

	list = g_list_append( list, "");

	gtk_entry_set_text( view->food_entry, "");
	gtk_combo_set_popdown_strings( view->combo, list);
	gtk_entry_set_text( view->amount_entry, "");
	gnutr_reset_nutr_comp();

	g_list_free( list);
}

/* callback when the pointer enters the "Clear" button. */
static void
on_clear_button_enter( GtkButton *button,
                       gpointer   data)
{
	Food_View_t *view;
	view = (Food_View_t *)data;
	gnome_appbar_push( view->appbar, " Clear the selected food data.");
}

/* callback when the pointer leaves the "Clear" button. */
static void
on_clear_button_leave( GtkButton *button,
                       gpointer   data)
{
	Food_View_t *view;
	view = (Food_View_t *)data;
	gnome_appbar_clear_stack( view->appbar);
}

/* callback when the "Compute" button is released. */
static void
on_compute_button_released( GtkButton *button,
                            gpointer   data)
{
	char *amount = NULL, *msre_no, *fd_no;
	char *food_desc, *msre_desc = NULL;
	GHashTable *htbl1, *htbl2;
	Food_View_t *view;

	view = (Food_View_t *)data;

	htbl1 = get_htbl_fd_desc_fd_no();
	htbl2 = get_htbl_msre_desc_msre_no();

	food_desc = gtk_entry_get_text( view->food_entry);
	if ( !food_desc || strcmp( food_desc, "") == 0) return;
	fd_no = (char *)g_hash_table_lookup( htbl1, (gpointer)food_desc);

	msre_desc = gtk_entry_get_text( view->combo_entry);
	msre_no = (char *)g_hash_table_lookup( htbl2, (gpointer)msre_desc);

	amount = gtk_entry_get_text( view->amount_entry);
	if ( !amount || strcmp( amount, "") == 0) return;

	gnutr_calc_nutr_comp( amount, msre_no, fd_no);
}

/* callback when the pointer enters the "Compute" button. */
static void
on_compute_button_enter( GtkButton *button,
                         gpointer   data)
{
	Food_View_t *view;
	
	view = (Food_View_t *)data;
	gnome_appbar_push( view->appbar,
		" Compute the food nutrient composition.");
}

/* callback when the pointer leaves the "Compute" button. */
static void
on_compute_button_leave( GtkButton *button,
                         gpointer   data)
{
	Food_View_t *view;
	
	view = (Food_View_t *)data;
	gnome_appbar_clear_stack( view->appbar);
}

/* callback when the "Goals" button is released. */
static void
on_goals_button_released( GtkButton *button,
                          gpointer   data)
{
	gnutr_show_nutr_goal_dlg();
}

/* callback when the pointer enters the "Goals" button. */
static void
on_goals_button_enter( GtkButton *button,
                       gpointer   data)
{
	Food_View_t *view;
	
	view = (Food_View_t *)data;
	gnome_appbar_push( view->appbar, " Open the nutrient goal dialog.");
}

/* callback when the pointer leaves the "Goals" button. */
static void
on_goals_button_leave( GtkButton *button,
                       gpointer   data)
{
	Food_View_t *view;
	
	view = (Food_View_t *)data;
	gnome_appbar_clear_stack( view->appbar);
}

/* callback when the "Search" button is released. */
static void
on_search_button_released( GtkButton *button,
                           gpointer   data)
{
	enum { PLAN_VIEW, RECIPE_VIEW, FOOD_VIEW };
	gnutr_show_srch_dlg( FOOD_VIEW);
}

void
gnutr_update_food_win( char *food_no)
{
	GHashTable *htbl1;
	GList *msre_list = NULL;
	char *food_desc;
	Food_View_t *view;

	g_return_if_fail( food_no);

	view = get_food_view();

	htbl1 = get_htbl_fd_no_fd_desc();
	msre_list = gnutr_get_fd_no_msre_desc_list( food_no);

	food_desc = g_hash_table_lookup( htbl1, (gpointer)food_no);
	gtk_entry_set_text( view->food_entry, food_desc);

	gtk_combo_set_popdown_strings( view->combo, msre_list);

	gtk_entry_set_text( view->amount_entry, "1.00");
}

/* exit gnutrition when "Exit" is selected from the menubar. */
static void
on_exit_activate( GtkMenuItem *menuitem,
                  gpointer     data)
{
	gnutr_exit_app();
}

/* clear the food when "Clear" is selected from the menubar. */
static void
on_clear_activate( GtkMenuItem *menuitem,
                   gpointer     data)
{
	on_clear_button_released( NULL, data);
}

/* show the meal plan view when "Meal Plan" is selected from the menubar. */
static void
on_meal_plan_activate( GtkMenuItem *menuitem,
                       gpointer     data)
{
	gnutr_switch_to_meal_plan_view();
}

/* show the recipe view when "Recipe" is selected from the menubar. */
static void
on_recipe_activate( GtkMenuItem *menuitem,
                    gpointer     data)
{
	gnutr_switch_to_recipe_view();
}

/* show the food view when "Food" is selected from the menubar. */
static void
on_food_activate( GtkMenuItem *menuitem,
                  gpointer     data)
{
}

/* show the nutrient goal dialog when "Nutrient Goal" is selected 
 * from the menubar. */
static void
on_nutr_goal_activate( GtkMenuItem *menuitem,
                       gpointer     data)
{
	on_goals_button_released( NULL, data);
}

/* show the about dialog when "About" is selected from the menubar. */
static void
on_about_activate ( GtkMenuItem *menuitem,
                    gpointer     data)
{
	gnutr_show_about_dlg();
}

