/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef INCLUDED_USRP1_SINK_BASE_H
#define INCLUDED_USRP1_SINK_BASE_H

#include <gr_sync_block.h>
#include <stdexcept>

class usrp_standard_tx;


/*!
 * \brief abstract interface to Universal Software Radio Peripheral Tx path (Rev 1)
 */
class usrp1_sink_base : public gr_sync_block {
 private:
  usrp_standard_tx	*d_usrp;
  int			 d_nunderruns;
  
 protected:
  usrp1_sink_base (const std::string &name,
		   gr_io_signature_sptr input_signature,
		   int which_board,
		   unsigned int interp_rate
		   ) throw (std::runtime_error);

  /*!
   * \brief convert between input item format and usrp native format
   *
   * \param input_items[in]		stream(s) of input items
   * \param input_index[in]		starting index in input_items
   * \param input_items_available[in]	number of items available starting at item[index]
   * \param input_items_consumed[out]  	number of input items consumed by copy
   * \param usrp_buffer[out]		destination buffer
   * \param usrp_buffer_length[in] 	\p usrp_buffer length in bytes
   * \param bytes_written[out]		number of bytes written into \p usrp_buffer
   */
  virtual void copy_to_usrp_buffer (gr_vector_const_void_star &input_items,
				    int  input_index,
				    int	 input_items_available,
				    int  &input_items_consumed,
				    void *usrp_buffer,
				    int  usrp_buffer_length,
				    int	 &bytes_written) = 0;

 public:
  ~usrp1_sink_base ();

  int work (int noutput_items,
	    gr_vector_const_void_star &input_items,
	    gr_vector_void_star &output_items);

  /*!
   * \brief Set interpolator rate.  \p rate must be in [4, 1024] and a multiple of 4.
   *
   * The final complex sample rate across the USB is
   *   dac_freq () / interp_rate ()
   */
  bool set_interp_rate (unsigned int rate);

  /*!
   * \brief set the frequency of the digital up converter.
   *
   * \p channel must be 0.  \p freq is the center frequency in Hz.
   * It must be in the range [-44M, 44M].  The frequency specified is
   * quantized.  Use tx_freq to retrieve the actual value used.
   */
  bool set_tx_freq (int channel, double freq);

  void set_verbose (bool verbose);

  // ACCESSORS

  long dac_freq () const;
  unsigned int interp_rate () const;
  double tx_freq (int channel) const;
  int nunderruns () const { return d_nunderruns; }

  /*!
   * \brief Write auxiliary digital to analog converter.
   *
   * \param slot	Which Tx or Rx slot to write.
   *    		N.B., SLOT_TX_A and SLOT_RX_A share the same AUX DAC's.
   *          		SLOT_TX_B and SLOT_RX_B share the same AUX DAC's.
   * \param which_dac	[0,3] RX slots must use only 0 and 1.  TX slots must use only 2 and 3.
   * \param value	[0,4095]
   * \returns true iff successful
   */
  bool write_aux_dac (int slot, int which_dac, int value);

  /*!
   * \brief Read auxiliary analog to digital converter.
   *
   * \param slot	2-bit slot number. E.g., SLOT_TX_A
   * \param which_adc	[0,1]
   * \returns value in the range [0,4095] if successful, else READ_FAILED.
   */
  int read_aux_adc (int slot, int which_adc);

  /*!
   * \brief Write EEPROM on motherboard or any daughterboard.
   * \param i2c_addr		I2C bus address of EEPROM
   * \param eeprom_offset	byte offset in EEPROM to begin writing
   * \param buf			the data to write
   * \returns true iff sucessful
   */
  bool write_eeprom (int i2c_addr, int eeprom_offset, const std::string buf);

  /*!
   * \brief Write EEPROM on motherboard or any daughterboard.
   * \param i2c_addr		I2C bus address of EEPROM
   * \param eeprom_offset	byte offset in EEPROM to begin reading
   * \param len			number of bytes to read
   * \returns the data read if successful, else a zero length string.
   */
  std::string read_eeprom (int i2c_addr, int eeprom_offset, int len);

  // internal routines...

  /*!
   * \brief Write FPGA register.
   * \param regno	7-bit register number
   * \param value	32-bit value
   * \returns true iff successful
   */
  bool _write_fpga_reg (int regno, int value);	//< 7-bit regno, 32-bit value

  /*!
   * \brief Read FPGA register.
   * \param regno	7-bit register number
   * \returns register value if successful, else READ_FAILED
   */
  int  _read_fpga_reg (int regno);
};

#endif /* INCLUDED_USRP1_SINK_BASE_H */
