/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Sample application to illustrate the use of a GrSimpleScopeSink.
 * The input here is provided by the VrSigSource signal generator.
 */

#include <make_GrMC4020Source.h>
#include <VrFileSource.h>
#include <GrFFTSink.h>
#include <VrFixOffset.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include "VrGUI.h" 
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <atsc_consts.h>

#define SAMPLING_FREQUENCY                   20e6
#define	DAUGHTER_CARD_SAMPLING_FREQUENCY     (ATSC_SYMBOL_RATE * 2)

#define IOTYPE short


static void
usage (const char *name)
{
  fprintf (stderr, "usage: %s [-f <filename>] [-r] [-n] [-x]\n", name);
  fprintf (stderr, "  -f <filename> : read from file instead of ADC\n");
  fprintf (stderr, "  -r : continuously repeat the file contents (loop)\n");
  fprintf (stderr, "  -n : don't subtract the ADC DC offset\n");
  fprintf (stderr, "  -D : clock input from daughter card\n");
  exit (1);
}

int main(int argc, char **argv) {
  VrGUI *guimain = new VrGUI(argc, argv);
  VrGUILayout *horiz = guimain->top->horizontal();
  VrGUILayout *vert = horiz->vertical();
  VrSource<short> *source;
  VrSink<IOTYPE> *sink;
  VrFixOffset<short,IOTYPE> *offset_fixer;

  int	c;
  char	*filename = 0;		// if 0, then use ADC for input
  bool  repeat_p = false;	// continuous loop through file
  bool  fix_offset_p = true;	// subtract constant ADC DC offset?
  bool	clk_from_daughter_card_p = false;

  while ((c = getopt (argc, argv, "rf:nD")) != EOF){
    switch (c){

    case 'f':
      filename = optarg;
      break;

    case 'r':
      repeat_p = true;
      break;

    case 'n':
      fix_offset_p = false;
      break;

    case 'D':
      clk_from_daughter_card_p = true;
      break;
      
    case '?':
    default:
      usage (argv[0]);
      break;
    }
  }

  if(filename)
    source = new VrFileSource<short>(SAMPLING_FREQUENCY, filename, repeat_p);

  else {
    if (clk_from_daughter_card_p)
      source = make_GrMC4020SourceS(DAUGHTER_CARD_SAMPLING_FREQUENCY,
					 MCC_CH3_EN
					 | MCC_ALL_5V
					 | MCC_CLK_AD_START_TRIG_IN);
    else
      source = make_GrMC4020SourceS(SAMPLING_FREQUENCY,
					 MCC_CH3_EN
					 | MCC_ALL_5V
					 | MCC_CLK_INTERNAL);
  }
  
  sink = new GrFFTSink<IOTYPE>(vert, 20, 100, 1024);

  if (fix_offset_p){
    offset_fixer = new VrFixOffset<short,IOTYPE>();
    NWO_CONNECT (source, offset_fixer);
    NWO_CONNECT (offset_fixer, sink);
  }
  else
    NWO_CONNECT (source, sink);

  VrMultiTask *m = new VrMultiTask ();
  m->add (sink);
  m->start();

  guimain->start();

  while (1) {
    guimain->processEvents(10 /*ms*/);
    m->process();
  }
}
