/*
     This file is part of GNUnet.
     (C) 2005, 2006, 2007 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/namespace_search.c
 * @brief code for namespace-related operations on the search frame
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "fs.h"
#include "helper.h"
#include "meta.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <GNUnet/gnunet_util_pseudonym.h>
#include <GNUnet/gnunet_uritrack_lib.h>
#include <GNUnet/gnunet_namespace_lib.h>
#include <extractor.h>


/**
 * The spin button giving the rating for a particular namespace
 * has been changed.  Store the new rating for the namespace.
 */
void
on_namespaceRatingSpinButton_changed_fs (GtkWidget * dummy,
                                         GtkWidget * dummy2)
{
  GtkWidget *spin;
  GtkWidget *ncbe;
  GtkTreeModel *model;
  GtkTreeIter iter;
  char *nsName;
  char *description;
  int rating;
  int newrating;
  GNUNET_HashCode nsid;

  GNUNET_GTK_DEBUG_BEGIN ();
  spin =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "namespaceRatingSpinButton");
  ncbe =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "searchNamespaceComboBoxEntry");
  model = gtk_combo_box_get_model (GTK_COMBO_BOX (ncbe));
  description = NULL;
  nsName = NULL;
  if (TRUE == gtk_combo_box_get_active_iter (GTK_COMBO_BOX (ncbe), &iter))
    {
      gtk_tree_model_get (model,
                          &iter,
                          NS_SEARCH_DESCRIPTION, &description,
                          NS_SEARCH_NAME, &nsName,
                          NS_SEARCH_RATING, &rating, -1);
      if ((description != NULL) && (0 == strcmp (description, "")))
        {
          /* just to be sure */
          gtk_widget_set_sensitive (spin, FALSE);
        }
      else
        {
          if (nsName != NULL)
            {
              GNUNET_GE_ASSERT (NULL,
                                GNUNET_OK ==
                                GNUNET_pseudonym_name_to_id (ectx, cfg,
                                                             nsName, &nsid));
              newrating = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin));
              rating = GNUNET_pseudonym_rank (ectx,
                                              cfg, &nsid, newrating - rating);
              if (rating != newrating)
                {
                  /* concurrent modification? */
                  gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), rating);
                  GNUNET_GE_BREAK (ectx, 0);
                }
              gtk_list_store_set (GTK_LIST_STORE (model),
                                  &iter, NS_SEARCH_RATING, rating, -1);
            }
        }
    }
  else
    {
      /* FIXME: if GNUNET_enc_to_hash succeeds, we may want to keep this
         active */
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), 0);
      gtk_widget_set_sensitive (spin, FALSE);
    }
  if (description != NULL)
    free (description);
  if (nsName != NULL)
    free (nsName);
  GNUNET_GTK_DEBUG_END ();
}


/**
 * The namespace in the search window has changed.
 * Update the trust level (possibly changing sensitivity)
 * and set the search string to the root (if available).
 */
void
on_searchNamespaceComboBoxEntry_changed_fs (GtkWidget * dummy,
                                            GtkWidget * dummy2)
{
  GtkWidget *keyword;
  GtkWidget *spin;
  GtkWidget *ncbe;
  GtkTreeModel *model;
  GtkTreeIter iter;
  int rating;
  char *nsName;
  char *descStr;
  GNUNET_HashCode ns;
  char *root;

  GNUNET_GTK_DEBUG_BEGIN ();
  spin =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "namespaceRatingSpinButton");
  ncbe =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "searchNamespaceComboBoxEntry");
  model = gtk_combo_box_get_model (GTK_COMBO_BOX (ncbe));
  descStr = NULL;
  nsName = NULL;
  if (TRUE == gtk_combo_box_get_active_iter (GTK_COMBO_BOX (ncbe), &iter))
    {
      gtk_tree_model_get (model,
                          &iter,
                          NS_SEARCH_DESCRIPTION, &descStr,
                          NS_SEARCH_NAME, &nsName,
                          NS_SEARCH_RATING, &rating, -1);
      if ((descStr != NULL) && (0 == strcmp (descStr, "")))
        {
          gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), 0);
          gtk_widget_set_sensitive (spin, FALSE);
        }
      else if (nsName != NULL)
        {
          GNUNET_GE_ASSERT (NULL,
                            GNUNET_OK ==
                            GNUNET_pseudonym_name_to_id (ectx, cfg,
                                                         nsName, &ns));
          gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), rating);
          gtk_widget_set_sensitive (spin, TRUE);
          keyword
            = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                                    "fssearchKeywordComboBoxEntry");
          root = GNUNET_NS_namespace_get_root (ectx, cfg, &ns);
          if (root != NULL)
            {
              gtk_entry_set_text (GTK_ENTRY
                                  (gtk_bin_get_child (GTK_BIN (keyword))),
                                  root);
              GNUNET_free (root);
            }
          else
            {
              gtk_entry_set_text (GTK_ENTRY
                                  (gtk_bin_get_child (GTK_BIN (keyword))),
                                  "");
            }
        }
    }
  else
    {
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), 0);
      gtk_widget_set_sensitive (spin, FALSE);
    }
  if (descStr != NULL)
    free (descStr);
  if (nsName != NULL)
    free (nsName);
  GNUNET_GTK_DEBUG_END ();
}

struct NewNamespaceInfo
{
  char *namespaceName;
  const GNUNET_HashCode *namespaceId;
  const struct GNUNET_MetaData *md;
  int rating;
};

static void *
saveDiscovery (void *cls)
{
  struct NewNamespaceInfo *nni = cls;
  GtkListStore *model;
  GtkTreeIter iter;
  struct GNUNET_MetaData *dmd;
  GtkWidget *ncbe;
  char *desc;
  char *haveName;
  int found;

  GNUNET_GTK_DEBUG_BEGIN ();
  ncbe =
    glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                          "searchNamespaceComboBoxEntry");
  model = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (ncbe)));
  if (nni->md == NULL)
    {
      dmd = NULL;
      desc = GNUNET_strdup ("");
    }
  else
    {
      dmd = GNUNET_meta_data_duplicate (nni->md);
      desc = GNUNET_meta_data_get_first_by_types (dmd,
                                                  EXTRACTOR_DESCRIPTION,
                                                  EXTRACTOR_TITLE,
                                                  EXTRACTOR_AUTHOR,
                                                  EXTRACTOR_GENRE,
                                                  EXTRACTOR_SUBJECT,
                                                  EXTRACTOR_CREATOR,
                                                  EXTRACTOR_PRODUCER,
                                                  EXTRACTOR_GROUP,
                                                  EXTRACTOR_CREATED_FOR,
                                                  EXTRACTOR_SUMMARY,
                                                  EXTRACTOR_OWNER, -1);
      if (desc == NULL)
        desc = GNUNET_strdup ("");
    }

  /* check if present! */
  found = GNUNET_NO;
  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
    {
      do
        {
          gtk_tree_model_get (GTK_TREE_MODEL (model),
                              &iter, NS_SEARCH_NAME, &haveName, -1);
          if ((haveName != NULL) &&
              (0 == strcmp (haveName, nni->namespaceName)))
            found = GNUNET_YES;
          GNUNET_free_non_null (haveName);
        }
      while ((found == GNUNET_NO) &&
             (gtk_tree_model_iter_next (GTK_TREE_MODEL (model), &iter)));
    }
  if (found == GNUNET_NO)
    {
      GNUNET_GTK_show_info_message (_("Namespace `%s' found: %s.\n"),
                                    nni->namespaceName, desc);
      gtk_list_store_append (model, &iter);
      gtk_list_store_set (model,
                          &iter,
                          NS_SEARCH_DESCRIPTION, desc,
                          NS_SEARCH_NAME, nni->namespaceName,
                          NS_SEARCH_METADATA, dmd,
                          NS_SEARCH_RATING, nni->rating, -1);
    }
  else
    {
      GNUNET_meta_data_destroy (dmd);
    }
  GNUNET_free (desc);
  GNUNET_GTK_DEBUG_END ();

  return NULL;
}



/**
 * Call this function to inform the user about
 * newly found namespaces and to add them to the
 * list of available namespaces in the search
 * dialog.
 *
 * @param cls pass NULL
 * @param rating the local rating of the namespace
 * @return GNUNET_OK (always)
 */
int
namespace_discovered_cb (void *cls,
                         const GNUNET_HashCode * namespaceId,
                         const struct GNUNET_MetaData *md, int rating)
{
  struct NewNamespaceInfo nni;

  nni.namespaceName = GNUNET_pseudonym_id_to_name (ectx, cfg, namespaceId);
  nni.namespaceId = namespaceId;
  nni.md = md;
  nni.rating = rating;
  GNUNET_GTK_save_call (&saveDiscovery, &nni);
  GNUNET_free (nni.namespaceName);
  return GNUNET_OK;
}
