# GNU Enterprise Navigator - GTK Frontent
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: UIgtk2.py 9957 2009-10-11 18:55:59Z reinhard $

import pygtk
pygtk.require('2.0')

import gtk
import gtkhtml2
import urllib
import urlparse
import os

from gnue.common.apps import GConfig
from gnue.common.utils.FileUtils import openResource
from gnue.navigator import VERSION
from gnue.navigator import GNParser

try:
  from gnue.forms.GFInstance import GFInstance
  from gnue.forms.uidrivers import gtk2 as ui

except ImportError:
  FORMS_SUPPORT = False
  print 'GNUe Forms is not installed on your system'

images_dir = GConfig.getInstalledBase ('forms_images','common_images') + '/'


# =============================================================================
# This class implements the GTK-UI of the navigator client
# =============================================================================

class Instance:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, processes):
    """
    @param processes: GNObjects tree describing the current loaded gpd
    """

    self.processes         = processes
    self._lastSerialNumber = 0
    self._treeModel        = None

    self.__currentURL      = None
    self.__opener          = urllib.FancyURLopener ()

    self.app = ui.getApp ()

         
  # ---------------------------------------------------------------------------
  # Build the UI and start the navigator client
  # ---------------------------------------------------------------------------

  def run (self, instance):
    """
    This function creates the user interface and starts the main loop
    """

    self.__instance = instance

    self.__buildInterface ()

    # Transform the GNObjects tree into a tree store and activate it
    self.__treeStore = self.__buildTreeModel (self.processes)
    self.treeView.set_model (self.__treeStore)
    self.treeView.expand_row ((0,), False)
    self.__lastItem = None

    self.processes.setClientHandlers ({'form': self.runForm})

    self.treeView.set_cursor ((0,))

    self.mainWindow.show_all ()
    self.app.mainLoop ()
  

  # ---------------------------------------------------------------------------
  # display a message in the status bar
  # ---------------------------------------------------------------------------

  def setStatus (self, message = None):
    """
    This function removes the last message from the status bar and adds a new
    one if specified.

    @param message: message to put into the status bar or None
    """

    self.statusbar.pop (0)
    if message is not None:
      self.statusbar.push (0, message)


  # ---------------------------------------------------------------------------
  # Build up the user interface
  # ---------------------------------------------------------------------------

  def __buildInterface (self):
    """
    This function creates the user interface and connects all signal handlers
    """

    self.mainWindow = gtk.Window ()
    self.mainWindow.set_resizable (True)
    self.content_table = gtk.Table (3, 1, False)

    self.mainWindow.add (self.content_table)
    self.mainWindow.set_title ('GNUe Navigator')
    self.mainWindow.set_default_size (600, 400)
    self.mainWindow.connect ('delete_event', self.__windowExit)

    # StatusBar
    self.statusbar = gtk.Statusbar ()
    self.content_table.attach (self.statusbar,
                 # X direction           Y direction
                 0, 1,                   2, 3,
                 gtk.EXPAND | gtk.FILL,  0,
                 0,                      0)

    # Menu-Bar
    self.handleBox = gtk.HandleBox ()
    self.menu = self.__createMenuBar ()
    self.handleBox.add (self.menu)
    self.menu.show ()

    self.content_table.attach (self.handleBox,
                 # X direction           Y direction
                 0, 1,                   0, 1,
                 gtk.EXPAND | gtk.FILL,  0,
                 0,                      0)

    # main part of window
    self.splitter = gtk.HPaned ()
    self.content_table.attach (self.splitter,
                 # X direction           Y direction
                 0, 1,                   1, 2,
                 gtk.EXPAND | gtk.FILL,  gtk.EXPAND | gtk.FILL,
                 0,                      0)

    self.treePane = gtk.ScrolledWindow ()
    self.treePane.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    self.treePane.set_shadow_type (gtk.SHADOW_IN)

    self.treeView = gtk.TreeView ()
    self.treeView.set_headers_visible (False)

    col = gtk.TreeViewColumn ('Title', gtk.CellRendererText (), text = 0)
    self.treeView.append_column (col)
    self.treeView.connect ('row_activated', self.__row_activated)
    self.treeView.connect ('cursor_changed', self.__row_selected)

    self.treePane.add (self.treeView)


    self.splitter.add1 (self.treePane)

    self.viewPane = gtk.ScrolledWindow ()
    self.viewPane.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    self.viewPane.set_shadow_type (gtk.SHADOW_IN)
    self.splitter.add2 (self.viewPane)

    self.view = gtkhtml2.View ()
    self.view.connect ('on_url', self.__onURL)
    self.viewPane.add (self.view)

    self.__startNewDocument ()

    self.splitter.set_position (200)


  # ---------------------------------------------------------------------------
  # Create the menu bar
  # ---------------------------------------------------------------------------

  def __createMenuBar (self):
    """
    This function creates the menu bar for the navigator.
    """

    menu_items = ( \
        (u_('/_File'), None, None, 0, '<Branch>'),
        (u_('/_File/_Open'), '<control>O', self.__openFile, 0, '<StockItem>',
                                                               gtk.STOCK_OPEN),
        (u_('/_File/_Quit'), '<control>Q', self.__windowExit, 0, '<StockItem>',
                                                               gtk.STOCK_QUIT),

        (u_('/_Favorites'), None, None, 0, '<Branch>'),
        (u_('/_Favorites/_Add Favorite'), None, None, 0, ''),
        (u_('/_Favorites/_Organize Favorites'), None, None, 0, ''),
 
        (u_('/_Help'), None, None, 0, '<Branch>'),
        (u_('/Help/_About'), None, self.__about, 0, ''),
      )

    self.accel_group = gtk.AccelGroup ()
    self.mainWindow.add_accel_group (self.accel_group)
    
    self.item_factory = gtk.ItemFactory (gtk.MenuBar, '<main>',
                                         self.accel_group)
    
    self.item_factory.create_items (menu_items, self.mainWindow)
    
    return self.item_factory.get_widget ('<main>')


  # ---------------------------------------------------------------------------
  # Close the navigator
  # ---------------------------------------------------------------------------

  def __windowExit (self, widget, event, data = None):
    """
    This function quits the main loop
    """

    # TODO: can we keep track of all windows opened ?
    self.app.quit ()


  # ---------------------------------------------------------------------------
  # Create a new tree model
  # ---------------------------------------------------------------------------

  def __buildTreeModel (self, GNTree):
    """
    This function creates a new tree store and populates the given GNObjects
    tree into that store

    @param GNTree: GNObjects tree to be added to the gtk.TreeStore
    @return: gtk.TreeStore holding the given GNObjects tree. The tree store has
        two columns: Title of the element, the element instance itself.
    """

    result = gtk.TreeStore (str, object)
    GNTree.walk (self.__addToTreeModel, store = result)

    return result


  # ---------------------------------------------------------------------------
  # Add an item of a GNObjects tree to the TreeStore
  # ---------------------------------------------------------------------------

  def __addToTreeModel (self, gnObject, store):
    """
    This function adds an item of a GNObjects tree to the given tree store

    @param gnObject: GNObject instance to be added
    @param store: gtk.TreeStore the gnObject should be added to
    """

    if gnObject._type == 'GNProcesses':
      node = store.append (None, [gnObject.title, gnObject])

    elif gnObject._type in ['GNStep', 'GNProcess']:
      node = store.append (gnObject.getParent ().__node,
                           [gnObject.title, gnObject])

    else:
      return

    # Remember the iterator of the current node
    gnObject.__node = node


  # ---------------------------------------------------------------------------
  # Activate a row in the tree view
  # ---------------------------------------------------------------------------

  def __row_activated (self, tree, path, column):
    """
    This function is called on the 'row-activated' signal of the tree view,
    which happens if an element of the tree get's selected by the enter-key or
    a double-click.

    @param tree: the tree view widget which emitted the signal
    @param path: the path tuple of the selected item
    @param column: the tree view column instance which has been activated
    """

    try:
      self.beginWait ()

      tree.expand_row (path, False)
      item = self.__treeStore.get_iter (path)

      # Fetch the associated GN* instance, which is hold in the first column
      gnObject = self.__treeStore.get_value (item, 1)
      if gnObject._type == 'GNStep':
        gnObject.run ()

    finally:
      self.endWait ()


  # ---------------------------------------------------------------------------
  # A row has been selected in the tree
  # ---------------------------------------------------------------------------

  def __row_selected (self, tree):
    """
    This function get's called when the focus in the tree view widget has
    changed.

    @param tree: the tree view widget which emitted the signal
    """

    try:
      self.beginWait ()
    
      (path, column) = tree.get_cursor ()
      if path != self.__lastItem:
        self.__lastItem = path

        item     = self.__treeStore.get_iter (path)
        gnObject = self.__treeStore.get_value (item, 1)
        descr    = gnObject.findChildOfType ('GNDescription')

        if descr is not None:
          stream = descr.getChildrenAsContent ()
        else:
          stream = self.__getTitlePage ()

        self.__startNewDocument ()
        self.__loadDocument ('text/html', stream)

    finally:
      self.endWait ()


  # ---------------------------------------------------------------------------
  # Create a HTML stream with the title page
  # ---------------------------------------------------------------------------

  def __getTitlePage (self):
    """
    This function creates a HTML string containing the title page

    @return: stream with the HTML code for the titlepage
    """

    return '<HTML><BODY><CENTER><B>GNUe Navigator</B>' \
           '<p><img src="%s"></p>' \
           '<p>A part of the <a href="http://www.gnuenterprise.org/">' \
           'GNU Enterprise Project</a></p>' \
           '</center></body></html>' % (images_dir + "/ship2.png")


  # --------------------------------------------------------------------------
  # Change the mouse pointer in an hour-glass
  # --------------------------------------------------------------------------

  def beginWait (self):
    """
    This function changes the mouse pointer to an hour glass.
    """

    if self.mainWindow.window is not None:
      self.mainWindow.window.set_cursor (gtk.gdk.Cursor (gtk.gdk.WATCH))


  # --------------------------------------------------------------------------
  # Change the mouse pointer back to it's normal apperance
  # --------------------------------------------------------------------------

  def endWait (self):
    """
    This function changes the mouse pointer back to the normal state.
    """

    if self.mainWindow.window is not None:
      self.mainWindow.window.set_cursor (gtk.gdk.Cursor (gtk.gdk.LEFT_PTR))


  # ===========================================================================
  # gtkhtml stuff
  # ===========================================================================


  # ---------------------------------------------------------------------------
  # Start a fresh document
  # ---------------------------------------------------------------------------

  def __startNewDocument (self):
    """
    This function creates a new instance of a fresh document, connects the
    signal handlers and resets the global URL.
    """

    self.__currentURL = None
    self.document = gtkhtml2.Document ()
    self.document.connect ('request_url' , self.__requestURL)
    self.document.connect ('link_clicked', self.__linkClicked)

    self.view.set_document (self.document)


  # ---------------------------------------------------------------------------
  # Load a given stream into the current document
  # ---------------------------------------------------------------------------

  def __loadDocument (self, contenttype, stream):
    """
    This function loads a stream of a given type into the current document
    which get's cleared first.

    @param contenttype: type of the contents, e.g. 'text/html'
    @param stream: string with the actual data
    """
    stream = '''<html><head><meta http-equiv="Content-Type" content="text/html; charset=utf8">
                      </head><body>%s</body></html>''' % stream

    self.document.clear ()
    self.document.open_stream (contenttype)
    self.document.write_stream (stream)
    self.document.close_stream ()


  # ---------------------------------------------------------------------------
  # Check if a given URL is relative or absolute
  # ---------------------------------------------------------------------------

  def __isRelativeToServer (self, url):
    """
    This function determines wether the given URL is absolute or relative

    @param url: URL to be checked
    @return: True if URL is relative, False if it is absolute
    """

    parts = urlparse.urlparse (url)
    if parts [0] or parts [1]:
      return False

    return True


  # ---------------------------------------------------------------------------
  # Resolve a URI so we have an absolute one afterwards
  # ---------------------------------------------------------------------------

  def __resolveURI (self, uri):
    """
    This function returns a resolved URI. If it is a relative URL it will be
    joined with the current 'global' URL.

    @param uri: URI to be resolved
    @return: complete URI
    """

    if self.__isRelativeToServer (uri):
      return urlparse.urljoin (self.__currentURL, uri)

    return uri


  # ---------------------------------------------------------------------------
  # Resolve a given URL and open it
  # ---------------------------------------------------------------------------

  def __openURL (self, url):
    """
    This function opens a given URL and returns a file-like object.
    @param url: URL to be opened. This URL will be resolved.

    @return: file object for the url.
    """

    uri = self.__resolveURI (url)
    return self.__opener.open (uri)


  # ---------------------------------------------------------------------------
  # download a given url into a stream
  # ---------------------------------------------------------------------------

  def __requestURL (self, document, url, stream):
    """
    This function get's called when the given document needs data from an URL.
    It opens the given URL and writes it's contents into the given stream.

    @param document: the document which requested the resource
    @param url: the URL from where we can get the resource
    @param stream: the stream where the result get's written to
    """

    try:
      self.beginWait ()

      self.setStatus (u_("Requesting %s" % url))
      stream.write (self.__openURL (url).read ())

    finally:
      self.setStatus ()
      self.endWait ()


  # ---------------------------------------------------------------------------
  # Process the click to a link in the current document
  # ---------------------------------------------------------------------------

  def __linkClicked (self, document, link):
    """
    This function get's called when the user clicks a link in the given
    document. The specified URL will then be loaded and displayed in the
    current document.

    @param document: the document where the link should be displayed in
    @param link: URL of the link which has been selected
    """

    try:
      self.beginWait ()

      fhd = self.__openURL (link)
      self.__currentURL = self.__resolveURI (link)

      headers = fhd.info ()
      document.clear ()

      ctype = headers.getheader ('Content-type') or 'text/plain'
      document.open_stream (ctype)
      document.write_stream (fhd.read ())
      document.close_stream ()

    finally:
      self.endWait ()


  # ---------------------------------------------------------------------------
  # On URL
  # ---------------------------------------------------------------------------

  def __onURL (self, view, url):
    """
    This function get's called when the mouse is over an url. This url get's
    displayed in the status bar. If the mouse has left the url this function
    will be called with an url value of None.
    """

    self.setStatus (url)

  # ---------------------------------------------------------------------------
  # Run a form
  # ---------------------------------------------------------------------------

  def runForm (self, step, parameters = {}):
    """
    This function runs a GNStep with a GNUe Form definition.

    @param step: the GNStep instance describing the form
    @param paramters: dictionary with user parameters for the form
    """

    # TODO: this code *should* be executed in a new thread.
    if os.path.basename (step.location) == step.location:
      try:
        formdir = gConfigNav ('FormDir')

      except KeyError:
        formdir = ""

      formfile = os.path.join (formdir, step.location)

    else:
      formfile = step.location

    try:
      self.beginWait ()
      self.setStatus ('running form %s' % formfile)
      self._runForm (formfile, parameters)

    finally:
      self.endWait ()
      self.setStatus ()


  # ---------------------------------------------------------------------------
  # Run a form from a trigger
  # ---------------------------------------------------------------------------

  def runFormFromTrigger (self, form, parameters = {}):
    """
    This function runs a form from trigger code.

    @param form: URL of the form
    @param parameters: dictionary with user parameters for the form
    """

    try:
      self.beginWait ()
      self.setStatus ('running form %s from trigger' % formfile)

      self._runForm (form, self._params)

    finally:
      self.endWait ()
      self.setStatus ()


  # ---------------------------------------------------------------------------
  # Do the dirty work of running a form
  # ---------------------------------------------------------------------------

  def _runForm (self, formfile, parameters):
    """
    This function starts a new form.

    @param formfile: URL of the form to be started
    @param parameters: dictionary with user parameters for the form
    """

    self.__instance.run_from_file(formfile, parameters)


  # ---------------------------------------------------------------------------
  # Open another process definition file
  # ---------------------------------------------------------------------------

  def __openFile (self, obj, more, muchmore):
    """
    This function displays a 'FileSelection' dialog to select other GPD files.
    If a file was selected it will be loaded into a new process tree.
    """

    self.filesel = gtk.FileSelection ( \
        u_("Select another process definition file"))
    self.filesel.ok_button.connect ('clicked', \
        lambda w, self: self.__reload (self.filesel.get_filename ()), self)
    self.filesel.cancel_button.connect ('clicked', \
        lambda obj, self: self.filesel.destroy (), self)
    self.filesel.hide_fileop_buttons ()
    self.filesel.complete ('*.gpd')

    self.filesel.show ()


  # ---------------------------------------------------------------------------
  # Load another process definition file
  # ---------------------------------------------------------------------------

  def __reload (self, filename):
    """
    This function loads a new process tree from the given filename and replaces
    the current tree view with the new process tree.

    @param filename: name of the GPD file to be loaded
    """

    if self.filesel is not None:
      self.filesel.destroy ()

    fhd = openResource (filename)

    try:
      processes = GNParser.loadProcesses (fhd)
      processes._connections   = self.processes._connections
      processes._configManager = self.processes._configManager
      processes._ui_type       = self.processes._ui_type

      newModel  = self.__buildTreeModel (processes)
      self.processes = processes
      self.__treeStore = newModel
      self.treeView.set_model (newModel)
      self.treeView.expand_row ((0,), False)
      self.treeView.set_cursor ((0,))
      self.treeView.grab_focus ()

    finally:
      fhd.close ()


  # ---------------------------------------------------------------------------
  # Display a nice about message
  # ---------------------------------------------------------------------------

  def __about (self, obj, more, muchmore):
    """
    This function displays an about box.
    """

    text = _("GNUE Navigator")+"\n"+      \
    _("    Version : ")+"%s\n"+         \
    _("    Driver  : UIgtk2")+"\n"+ \
    _("-= Process Info =-                  ")+"\n"+        \
    _("    Name   : ")+"%s\n"+          \
    _("    Version: ")+"%s\n"+          \
    _("    Author : ")+"%s\n"+          \
    _("    Description: ")+"%s\n"
    dlg = gtk.MessageDialog(self.mainWindow,
                            gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_MODAL,
                            gtk.MESSAGE_INFO,
                            gtk.BUTTONS_CLOSE,
                            message_format = text % (VERSION,'-','-','-','-'))
    dlg.connect ('response', lambda dlg, response: dlg.destroy ())
    dlg.show ()
