# GNU Enterprise Forms - Display handler - Numeric values
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: $
"""
DisplayHandler classes for Forms input validation
"""
__revision__ = "$Id$"

import re

from gnue.common.utils import xlocale
from gnue.forms.input.displayHandlers.Cursor import BaseCursor

# =============================================================================
# Display handler for numeric values
# =============================================================================

class Numeric(BaseCursor):
    """
    Class to handle the display and entry of numeric fields.
    """

    # -------------------------------------------------------------------------
    # Constructor
    # -------------------------------------------------------------------------

    def __init__(self, entry, eventHandler, subEventHandler, display_mask,
            input_mask):

        BaseCursor.__init__(self, entry, eventHandler, subEventHandler,
                display_mask, input_mask)
        if display_mask:
            self._display_mask = display_mask
        else:
            if self.field.scale:
                self._display_mask = '%%.%sf' % self.field.scale
            elif self.field.length:
                self._display_mask = '%d'
            else:
                self._display_mask = '%s'

        self.__grouping = gConfigForms('numeric_grouping') or None


    # -------------------------------------------------------------------------
    # Try to guess what value is meant by a given string
    # -------------------------------------------------------------------------

    def parse_display(self, display):
        """
        Try to convert the given display string into a number
        """

        if not len(display):
            return None

        match = re.match('^(\D*)(\d*)(\D*)(\d*)$', display)
        if match:
            (sign, whole, dec, frac) = match.groups()
            display = sign + whole
            if frac:
                display += "." + frac

        return float(display)


    # -------------------------------------------------------------------------
    # Create a display string for the current value
    # -------------------------------------------------------------------------

    def build_display(self, value, editing):
        """
        Create a display string for the current value
        """

        if value is None:
            return ''

        if editing:
            result = self._display_mask % value
        else:
            result = xlocale.format_numeric(self._display_mask, float(value),
                    self.__grouping)

        return result
