/* from.c -- Show from whom your messages are. */

/* GNU From -- show from whom your messages are.

   Copyright (C) 1990 Free Software Foundation

   Authored by Brian Fox.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdio.h>
#include <pwd.h>
#include <sys/errno.h>
#include "general.h"
#include "unixmail.h"

extern int errno;

#if !defined (SPOOLDIR)
#define SPOOLDIR "/usr/spool/mail"
#endif

extern struct passwd *getpwuid ();

/* Non-NULL is a regexp to match against FILTER_FIELD. */
char *filter_regexp = (char *)NULL;

/* Which field to do regexp matching on. */
char *filter_field = "From";

/* Non-zero means just count the messages, don't print headers. */
int count_only = 0;

/* Non-zero means include all messages in /usr/spool/mail/USER, not just
   those that haven't been read yet. */
int show_all = 0;

/* Non-zero means to extract matching messages. */
int extracting = 0;

/* The default specification string for header formatting. */
char *format_spec = "%18From %11Date  %47Subject";

/* The name of the program as taken from argv[0]. */
char *progname;

main (argc, argv)
     int argc;
     char **argv;
{
  int arg_index = 1;
  char *arg;

  arg = rindex (argv[0], '/');

  if (arg && arg != argv[0])
    progname = arg + 1;
  else
    progname = argv[0];

  /* Parse simple arguments. */
  while (arg_index < argc && argv[arg_index][0] == '-')
    {
      arg = argv[arg_index++];

      if (strcmp (arg, "-c") == 0)
	count_only = 1;
      else if (strcmp (arg, "-a") == 0)
	show_all = 1;
      else if (strcmp (arg, "-s") == 0)
	filter_regexp = argv[arg_index++];
      else if (strcmp (arg, "-field") == 0)
	filter_field = argv[arg_index++];
      else if (strcmp (arg, "-format") == 0)
	format_spec = argv[arg_index++];
      else if (strcmp (arg, "-extract") == 0)
	extracting = 1;
      else
	usage ();
    }

  if (arg_index < argc)
    {
      while (arg_index < argc)
	from (argv[arg_index++]);
    }
  else
    {
      struct passwd *entry;

      entry = getpwuid (getuid ());

      if (!entry)
	error ("You have no username");

      from (entry->pw_name);
    }
  exit (0);
}

/* Count and print messages in /usr/spool/mail/USER that contain the
   regular expression FILTER_REGEXP in the FILTER_FIELD.  If FILTER_REGEXP
   is NULL, then all messages match.  If the global variable count_only is
   non-zero, then just count the number of messages that would be displayed,
   and output that. */
from (user)
     char *user;
{
  register int i;
  FILE *stream;
  char *path, *line;
  char **headers;
  
  path = (char *)xmalloc (3 + strlen (user) + strlen (SPOOLDIR));

  if (*user != '/')
    sprintf (path, "%s/%s", SPOOLDIR, user);
  else
    strcpy (path, user);

  stream = fopen (path, "r");

  /* If the file couldn't be opened, maybe print error, then give up. */
  if (!stream)
    {
      if (errno != ENOENT)
	perror (path);

      free (path);
      return;
    }

  fclose (stream);

  if (extracting)
    {
      from_message_bodies
	(path, format_spec, filter_regexp, filter_field, show_all, stdout);

      free (path);
      return;
    }
  
  headers = from_headers
    (path, format_spec, filter_regexp, filter_field, show_all);

  if (!headers)
    {
      perror (path);
      free (path);
      return;
    }

  free (path);

  /* Count or display the headers. */
  for (i = 0; line = headers[i]; i++)
    if (!count_only)
      printf ("%s\n", line);

  if (count_only)
    printf ("%d\n", i);

  free_array (headers);
}

/* Release the memory allocated to the array of char *. */
free_array (array)
     char **array;
{
  register int i;

  for (i = 0; array && array[i]; i++)
    free (array[i]);

  if (array)
    free (array);
}

error (fmt, arg1, arg2)
     char *fmt, *arg1, *arg2;
{
  fprintf (stderr, "%s: ", progname);
  fprintf (stderr, fmt, arg1, arg2);
  fprintf (stderr, ".\n");
  exit (1);
}

usage ()
{
  fprintf (stderr, "\
Usage: %s [-c] [-a] [-field NAME] [-s SENDER] [-format SPEC] [USER]\n",
	   progname);
  fprintf (stderr, "\n\
    -a  means show ALL messages, not just unread ones.\n\
    -c  means show only the count of messages that would be displayed.\n\
    -s  takes a regular expression as an argument, mapped over entire From.\n\
    -extract means print the messages whose headers are printed as well.\n\
    -field specifies the name of the field to find `-s' argument in.\n"
	   );
  fprintf (stderr, "%s","\n\
    SPEC consists of `%' style escapes and normal text.\n\
    The default specification is \"%18From %11Date %48Subject\"\n");

  exit (1);
}
