// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of ccRTP.
// 
// The exception is that, if you link the ccRTP library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccRTP library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccRTP.  If you copy code from other releases into a copy of
// ccRTP, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccRTP, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

/** 
 * @file private.h 
 * @brief Declaration of ccRTP internal stuff.
 *
 * This header includes the autogenerated config.h, the general header 
 * (rtp.h) and the nana assertion checking library headers.
 */

#include "config.h"
#undef	PACKAGE
#undef	VERSION
#include <cc++/config.h>
#include <cc++/socket.h>
#include <cc++/export.h>
#include "rtp.h"
#include "rtpext.h"

// This is only of interest for developers, so it is included only if
// --with-nana flag is given to configure
#ifdef HAVE_LIBNANA
#   include "nana.h"
#   include "DL.h"
#else
#   define I(e)    /* empty */
#   define DL(e)   /* empty */
#   define VDL(e)  /* empty */
#endif

#ifdef   HAVE_CSTDLIB
#include <cstdlib>
#else
#   ifdef   HAVE_STDLIB_H
#      include <stdlib.h>
#   else
#      warning "Your system does not have cstdlib nor stdlib.h"
#define getenv(w) ("unidentifiable_user")
#   endif //HAVE_STDLIB_H
#endif //HAVE_CSTDLIB

#ifdef HAVE_SYS_UTSNAME_H
#   include <sys/utsname.h>
#else
#   error "Portability problem: you do not have sys/utsname.h"
#endif

#ifndef timeradd
#define timeradd(a, b, result)                                               \
 do {                                                                        \
   (result)->tv_sec = (a)->tv_sec + (b)->tv_sec;                             \
   (result)->tv_usec = (a)->tv_usec + (b)->tv_usec;                          \
   if ((result)->tv_usec >= 1000000)                                         \
     {                                                                       \
       ++(result)->tv_sec;                                                   \
       (result)->tv_usec -= 1000000;                                         \
     }                                                                       \
 } while (0)
#endif

#ifndef timersub
#define timersub(a, b, result)                                               \
 do {                                                                        \
   (result)->tv_sec = (a)->tv_sec - (b)->tv_sec;                             \
   (result)->tv_usec = (a)->tv_usec - (b)->tv_usec;                          \
   if ((result)->tv_usec < 0) {                                              \
     --(result)->tv_sec;                                                     \
     (result)->tv_usec += 1000000;                                           \
   }                                                                         \
 } while (0)
#endif

#ifdef  CCXX_NAMESPACES
namespace ost {
#endif

// This is for the main page in Doxygen documentation
/**
 * @mainpage ccRTP Library Reference
 * @section overview Overview (README)
 * @verbinclude README
 * @section todolist TODO List
 * @verbinclude TODO
 */

// This will make doxygen to generate example descriptions
/**
 * @example rtpduphello.cpp
 *
 * A basic example of how to use RTPDuplex
 **/
/** 
 * @example rtphello.cpp
 *
 * A basic example of how to use RTPPacket
 **/
/**
 * @example audio.h
 */
/**
 * @example audiotx.cpp
 */
/**
 * @example audiorx.cpp
 */

#pragma pack(1)

/**
 * @typedef ReceiverInfo
 *
 * Struct for the data contained in a receiver info block. Receiver
 * info blocks can be found in SR (sender report) or RR (receiver
 * report) RTCP packets.
 **/
 struct ReceiverInfo
{
	uint8 flost;
	uint8 lost_msb;
	uint16 lost_lsb;
	uint32 highest_seqnum;
	uint32 jitter;
	uint32 lsr;
	uint32 dlsr;
}       ;

/**
 * @typedef RRBlock
 *
 * Struct for a receiver info block in a SR (sender report) or an RR
 * (receiver report) RTCP packet.
 **/
typedef struct 
{
	uint32 ssrc;                   ///< source identifier
	ReceiverInfo rinfo;
}       RRBlock;

/**
 * @typedef RecvReport
 *
 * @short raw structure of the source and every receiver report in an
 * SR or RR RTCP packet.  
 **/
typedef struct 
{
	uint32 ssrc;                    ///< source identifier
	RRBlock blocks [0];             ///< receiver report blocks
}       RecvReport;

/**
 * @typedef SenderInfo
 *
 * Struct for the sender info block in a SR (sender report) RTCP packet.
 **/
 struct SenderInfo
{
	uint32 NTP_msb;
	uint32 NTP_lsb;
	uint32 RTPtimestamp;
	uint32 packet_count;
	uint32 octet_count;
}       ;

/**
 * @typedef SendReport
 *
 * Struct for SR (sender report) RTCP packets. 
 **/
typedef struct 
{
	uint32 ssrc;              ///< source identifier
	SenderInfo sinfo;         ///< actual sender info
	RRBlock blocks [0];        ///< possibly several receiver info blocks 
}       SendReport;

/**
 * @typedef SDESItem
 *
 * Struct for an item description from a SDES packet.
 **/
typedef struct 
{
	uint8 type;
	uint8 len;
}       SDESItem;

/**
 * @typedef SDESChunk
 *
 * Struct for a chunk of items
 **/
typedef struct 
{
	uint32 ssrc;
	SDESItem item;
}       SDESChunk;

/**
 * @typedef BYEPacket
 *
 * @short Struct for RTCP BYE (leaving session) packets.
 **/
typedef struct 
{
	uint32 ssrc;          ///< ssrc identifier of source leaving
	uint8 length;         ///< [optional] length of reason
	char reason [0];      ///< [optional] reason text (not null terminated)
}       BYEPacket;

/**
 * @typedef APPPacket
 *
 * @short Struct for RTCP APP (application specific) packets.
 **/
typedef struct 
{
	uint32 ssrc;          ///< ssrc identifier of source leaving
	char name [4];
	char* data;
}       APPPacket;

/**
 * @typedef RTCPPacket 
 *
 * @short Struct representing general RTCP packet headers as they are
 * sent through the network.
 * 
 * This struct consists of fixed header always present at the
 * beginning of any RTCP packet and a union for SR, RR, SDES, BYE and
 * APP packets.
 **/
struct RTCPPacket
{
	RTCPFixedHeader fh;           ///< Fixed RTCP header
	union 
	{
		SendReport SR;
		RecvReport RR;
		SDESChunk SDES; 
		BYEPacket BYE;
		APPPacket APP;
	}       info;                 ///< Union for SR, RR, SDES, BYE and APP
}       ;

#pragma pack()

#ifdef  CCXX_NAMESPACES
};
#endif

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
