/*	ccot.c
 *
 *	Complex circular cotangent
 *
 *
 *
 * SYNOPSIS:
 *
 * double complex ccot();
 * double complex z, w;
 *
 * w = ccot (z);
 *
 *
 *
 * DESCRIPTION:
 *
 * If
 *     z = x + iy,
 *
 * then
 *
 *           sin 2x  -  i sinh 2y
 *     w  =  --------------------.
 *            cosh 2y  -  cos 2x
 *
 * On the real axis, the denominator has zeros at even
 * multiples of PI/2.  Near these points it is evaluated
 * by a Taylor series.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       -10,+10      3000       6.5e-17     1.6e-17
 *    IEEE      -10,+10     30000       9.2e-16     1.2e-16
 * Also tested by ctan * ccot = 1 + i0.
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004

#include "complex.h"
#include <float.h>
#include <math.h>

#include "redupi.c"
#include "ctans.c"

double  complex
ccot(z)
double complex z;
{
    double complex w;
    double  d;

    d = cosh(2.0 * cimag(z)) - cos(2.0 * creal(z));

    if (fabs(d) < 0.25)
	d = ctans(z);

    if (d == 0.0)
    {
	//mtherr ("ccot", OVERFLOW);
	w = DBL_MAX + DBL_MAX * I;
	return (w);
    }

    w = sin(2.0 * creal(z)) / d - (sinh(2.0 * cimag(z)) / d) * I;
    return w;
}
