/*	tgamma.c
 *
 *	Gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, tgamma();
 * extern int signgam;
 *
 * y = tgamma( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns gamma function of the argument.  The result is
 * correctly signed, and the sign (+1 or -1) is also
 * returned in a global (extern) variable named signgam.
 * This variable is also filled in by the logarithmic gamma
 * function lgamma().
 *
 * Arguments |x| <= 34 are reduced by recurrence and the function
 * approximated by a rational function of degree 6/7 in the
 * interval (2,3).  Large arguments are handled by Stirling's
 * formula. Large negative arguments are made positive using
 * a reflection formula.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC      -34, 34      10000       1.3e-16     2.5e-17
 *    IEEE    -170,-33      20000       2.3e-15     3.3e-16
 *    IEEE     -33,  33     20000       9.4e-16     2.2e-16
 *    IEEE      33, 171.6   20000       2.3e-15     3.2e-16
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */

/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1987, 1989, 1992, 2000 by Stephen L. Moshier
*/
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, February 2004

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

static double polevl(double, double *, int);
static double stirf(double);

static double
polevl(double x, double *coef, int N)
{
    double  ans;
    int     i;
    double *p;

    p = coef;
    ans = *p++;
    i = N;

    do
	ans = ans * x + *p++;
    while (--i);

    return (ans);
}

/* *INDENT-OFF* */
static double P[] = 
{
  1.60119522476751861407E-4,
  1.19135147006586384913E-3,
  1.04213797561761569935E-2,
  4.76367800457137231464E-2,
  2.07448227648435975150E-1,
  4.94214826801497100753E-1,
  9.99999999999999996796E-1
};
static double Q[] = 
{
-2.31581873324120129819E-5,
 5.39605580493303397842E-4,
-4.45641913851797240494E-3,
 1.18139785222060435552E-2,
 3.58236398605498653373E-2,
-2.34591795718243348568E-1,
 7.14304917030273074085E-2,
 1.00000000000000000320E0
};

static double STIR[5] = 
{
 7.87311395793093628397E-4,
-2.29549961613378126380E-4,
-2.68132617805781232825E-3,
 3.47222221605458667310E-3,
 8.33333333333482257126E-2,
};
/* *INDENT-ON* */

//int     signgam = 0;
extern int signgam;

#define MAXGAM 171.624376956302725
#define MAXSTIR 143.01608
static const double SQTPI = 2.50662827463100050242E0;

/* Gamma function computed by Stirling's formula.
 * The polynomial STIR is valid for 33 <= x <= 172.
 */
static double
stirf(double x)
{
    double  y, w, v;

    w = 1.0 / x;
    w = 1.0 + w * polevl(w, STIR, 4);
    y = exp(x);
    if (x > MAXSTIR)
    {					/* Avoid overflow in pow() */
	v = pow(x, 0.5 * x - 0.25);
	y = v * (v / y);
    }
    else
    {
	y = pow(x, x - 0.5) / y;
    }
    y = SQTPI * y * w;
    return (y);
}


double
tgamma(x)
double  x;
{
    double  p, q, z;
    int     i;

    signgam = 1;

    if (isnand(x))
    {
	return (x);
    }
    if (isinfd(x))
    {
	if (x > 0.0)
	{
	    return (x);
	}
	else
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    return (NAN);
	}
    }

    q = fabs(x);
    p = floor(q);

    if ((p == q) && (x <= 0.0))
    {
	if (x)				// Negative integer
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    return (NAN);
	}
	else
	{
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_DIVBYZERO);
	    return copysign(HUGE_VAL, x);
	}
    }
    if ((q < 1.0 / DBL_MAX) || (q > MAXGAM))
    {
	__math_set_errno(ERANGE);
	__fp_raise_except(FE_OVERFLOW);
	return copysign(HUGE_VAL, x);
    }
    if (q > 33.0)
    {
	if (x < 0.0)
	{
	    i = p;
	    if ((i & 1) == 0)
		signgam = -1;
	    z = q - p;
	    if (z > 0.5)
	    {
		p += 1.0;
		z = q - p;
	    }
	    z = q * sin(PI * z);
	    if (z == 0.0)
	    {
		__math_set_errno(ERANGE);
		__fp_raise_except(FE_OVERFLOW);
		return (signgam * HUGE_VAL);
	    }
	    z = fabs(z);
	    z = PI / (z * stirf(q));
	}
	else
	{
	    z = stirf(x);
	}
	return (signgam * z);
    }

    z = 1.0;
    while (x >= 3.0)
    {
	x -= 1.0;
	z *= x;
    }

    while (x < 0.0)
    {
	if (x > -1.E-9)
	    goto small;
	z /= x;
	x += 1.0;
    }

    while (x < 2.0)
    {
	if (x < 1.e-9)
	    goto small;
	z /= x;
	x += 1.0;
    }

    if (x == 2.0)
	return (z);

    x -= 2.0;
    p = polevl(x, P, 6);
    q = polevl(x, Q, 7);
    return (z * p / q);

  small:
    if (x == 0.0)
    {
	return (HUGE_VAL);
    }
    else
	return (z / ((1.0 + 0.5772156649015329 * x) * x));
}
