// ----------
// splitflt.c
// ----------
//
// Function SplitFlt
//
// Breaks up float floating point argument into two 32-bit
// numbers of the same type.
//
// int SplitFlt(Arg, Exp, Head, Tail)
//
//      float Arg;                	// Input
//      int   *Exp;			// Exponent of Arg
//      float *Head;              	// Upper 32 bits of Arg
//      float *Tail;			// Lower 32 bits of Arg
//
// Returns -1 if Arg is +-Inf or NaN
//
// Procedure is:
//
// 1. Scale Arg in [.5, 1), store exponent in Exp
// 2. Get fraction bits as a 24-bit unsigned long integer value
// 3. Break into two equally sized unsigned long int values
//    HeadBits - Upper 12 bits are left justified in 32-bit ULONG
//    TailBits - Lower 12 bits are right justified in 32-bit ULONG
// 4. Convert Head and Tail to double, correctly signed and
//    scaled in [.5, 1) with exponent on Tail 12 less that Head's
//

#include <fdlibml.h>

int
SplitFlt(float  Arg,  int *Exp,
         float *Head, float *Tail)
{
    int     Retval = 0;

    if (!isfinitef(Arg))
    {
	Retval = -1;
    }
    else
    {
    	ULONG	FracBits, HeadBits, TailBits;

	FracBits = ldexpf(frexpf(fabsf(Arg), Exp), 24);
	HeadBits = FracBits & 0xfffff000UL;
	TailBits = FracBits ^ HeadBits;

	*Head = copysignf(ldexpf((float)HeadBits, -24), Arg);
	*Tail = copysignf(ldexpf((float)TailBits, -24), Arg);
    }

    return Retval;
}
