// ----------
// splitdbl.c
// ----------
//
// Function SplitDbl
//
// Breaks up double floating point argument into two 32-bit
// numbers of the same type.
//
// int SplitDbl(Arg, Exp, Head, Tail)
//
//      double Arg;                // Input
//      int	    *Exp;		// Exponent of Arg
//      double *Head;              // Upper 32 bits of Arg
//      double *Tail;              // Lower 32 bits of Arg
//
// Returns -1 if Arg is +-Inf or NaN
//
// Procedure is:
//
// 1. Scale Arg in [.5, 1), store exponent in Exp
// 2. Get fraction bits as a 64-bit unsigned long long integer value
// 3. Break into two nearly equally sized unsigned long long int values
//    HeadBits - Upper 27 bits are left justified in 64-bit ULLONG
//    TailBits - Lower 26 bits are right justified in 64-bit ULLONG
// 4. Convert Head and Tail to double, correctly signed and
//    scaled in [.5, 1) with exponent on Tail 53 less that Head's
//

#include <fdlibml.h>

int
SplitDbl(double Arg, int *Exp,
    double *Head, double *Tail)
{
    int     Retval = 0;

    if (!isfinited(Arg))
    {
	Retval = -1;
    }
    else
    {
    	ULLONG	FracBits, HeadBits, TailBits;

	FracBits = ldexp(frexp(fabs(Arg), Exp), 53);
	HeadBits = FracBits & 0xfffffffffc000000ULL;
	TailBits = FracBits ^ HeadBits;

	*Head = copysign(ldexp((double)HeadBits, -53), Arg);
	*Tail = copysign(ldexp((double)TailBits, -53), Arg);
    }

    return Retval;
}
