/*							sinl.c
 *
 *	Circular sine, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, sinl();
 *
 * y = sinl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Range reduction is into intervals of pi/4.  The reduction
 * error is nearly eliminated by contriving an extended precision
 * modular arithmetic.
 *
 * Two polynomial approximating functions are employed.
 * Between 0 and pi/4 the sine is approximated by the Cody
 * and Waite polynomial form
 *	x + x**3 P(x**2) .
 * Between pi/4 and pi/2 the cosine is represented as
 *	1 - .5 x**2 + x**4 Q(x**2) .
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	    # trials	  peak	       rms
 *    IEEE     +-5.5e11	     200,000	1.2e-19	    2.9e-20
 *
 * ERROR MESSAGES:
 *
 *   message	       condition	value returned
 * sin total loss   x > 2**39		    0.0
 *
 * Loss of precision occurs for x > 2**39 = 5.49755813888e11.
 * The routine as implemented flags a TLOSS error for
 * x > 2**39 and returns 0.0.
 */
/*							cosl.c
 *
 *	Circular cosine, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, cosl();
 *
 * y = cosl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Range reduction is into intervals of pi/4.  The reduction
 * error is nearly eliminated by contriving an extended precision
 * modular arithmetic.
 *
 * Two polynomial approximating functions are employed.
 * Between 0 and pi/4 the cosine is approximated by
 *	1 - .5 x**2 + x**4 Q(x**2) .
 * Between pi/4 and pi/2 the sine is represented by the Cody
 * and Waite polynomial form
 *	x  +  x**3 P(x**2) .
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	    # trials	  peak	       rms
 *    IEEE     +-5.5e11	      50000	 1.2e-19     2.9e-20
 */

/*							sin.c	*/

/*
Cephes Math Library Release 2.7:  May, 1998
Copyright 1985, 1990, 1998 by Stephen L. Moshier
*/
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, December 2003

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

static long double S[7] =
{
    -7.5785404094842805756289E-13L,
    +1.6058363167320443249231E-10L,
    -2.5052104881870868784055E-8L,
    +2.7557319214064922217861E-6L,
    -1.9841269841254799668344E-4L,
    +8.3333333333333225058715E-3L,
    -1.6666666666666666640255E-1L,
};
static long double C[7] =
{
    +4.7377507964246204691685E-14L,
    -1.1470284843425359765671E-11L,
    +2.0876754287081521758361E-9L,
    -2.7557319214999787979814E-7L,
    +2.4801587301570552304991E-5L,
    -1.3888888888888872993737E-3L,
    +4.1666666666666666609054E-2L,
};
static long double DP1	  = 7.853981554508209228515625E-1L;
static long double DP2 	  = 7.946627356147928367136046290398E-9L;
static long double DP3 	  = 3.061616997868382943065164830688E-17L;

#ifdef USE_LOSSTH
static long double LOSSTH = 5.49755813888e11L; /* 2^39 */
#endif

long double
sinl(x)
long double x;
{
    long double y, z, zz;
    int		j, sign;

    if (x == 0.0L)
	return (x);
    if (isnanl(x))
	return (x);

// Check for denormal Argument

    if (fpclassifyl(x) == FP_SUBNORMAL)
    {
    	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);
	return (x);
    }

    if (isinfl(x))
    {
    	__math_set_errno(EDOM);
	__fp_raise_except(FE_INVALID);
	return NAN;
    }

/* make argument positive but save the sign */

    sign = 1;
    if (x < 0.0L)
    {
	x = -x;
	sign = -1;
    }

    y = floorl(x / PIO4L);	/* integer part of x/PIO4 */

/* strip high bits of integer part to prevent integer overflow */
    z = ldexpl(y, -4);
    z = floorl(z);		/* integer part of y/8 */
    z = y - ldexpl(z, 4);	/* y - 16 * (y/16) */

    j = z;			/* convert to integer to test phase angle */
/* map zeros to origin */
    if (j & 1)
    {
	++j;
	++y;
    }
    j = j & 07;				/* octant modulo 360 degrees */
/* reflect in x axis */
    if (j > 3)
    {
	sign = -sign;
	j -= 4;
    }

/* Extended precision modular arithmetic */
    z = ((x - y * DP1) - y * DP2) - y * DP3;

    zz = z * z;
    if ((j == 1) || (j == 2))
    {
	long double c;
	c = C[6]+zz*(C[5]+zz*(C[4]+zz*(C[3]+zz*(C[2]+zz*(C[1]+zz*C[0])))));
	y = 1.0L - ldexpl(zz, -1) + zz * zz * c;
	//printf("Quadrants 1 or 2\n");
    }
    else
    {
	long double s;
	s = S[6]+zz*(S[5]+zz*(S[4]+zz*(S[3]+zz*(S[2]+zz*(S[1]+zz*S[0])))));
	y = z + z * (zz * s);
	//printf("Quadrants 0 or 4\n");
    }

    if (sign < 0)
	y = -y;

    return (y);
}


long double
cosl(x)
long double x;
{
    long double y, z, zz;
    long    i;
    int	    j, sign;

    if (!isfinitel(x))
    {
	if (isinfl(x))
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	}

	return NAN;
    }

    if (x == 0.0L)
    {
	return 1.0L;
    }

/* make argument positive */
    if (x < 0)
	x = -x;

    sign = 1;

    y = floorl(x / PIO4L);
    z = ldexpl(y, -4);
    z = floorl(z);			/* integer part of y/8 */
    z = y - ldexpl(z, 4);		/* y - 16 * (y/16) */

/* integer and fractional part modulo one octant */
    i = z;
    if (i & 1)				/* map zeros to origin */
    {
	i += 1;
	y += 1.0L;
    }
    j = i & 07;
    if (j > 3)
    {
	j -= 4;
	sign = -sign;
    }

    if (j > 1)
	sign = -sign;

/* Extended precision modular arithmetic */
    z = ((x - y * DP1) - y * DP2) - y * DP3;

    zz = z * z;
    if ((j == 1) || (j == 2))
    {
	long double s;

	s = S[6]+zz*(S[5]+zz*(S[4]+zz*(S[3]+zz*(S[2]+zz*(S[1]+zz*S[0])))));
	y = z + z * (zz * s);
    }
    else
    {
	long double c;
	c = C[6]+zz*(C[5]+zz*(C[4]+zz*(C[3]+zz*(C[2]+zz*(C[1]+zz*C[0])))));
	y = 1.0L - ldexpl(zz, -1) + zz * zz * c;
    }

    if (sign < 0)
	y = -y;

    return (y);
}
