// ---------
// remquol.c
// ---------
//
// Function remquol()
//
// Synopsis
//
//	#include <math.h>
//	long double remquol(long double x, long double y,
//		int *quo);
//
// Description
//
// Function remquol computes the same remainder as function
// remainderl.	In the object pointed to by quo, remquol
// stores the signed lowest three bits (range -7 to +7) of the
// integer value closest to the quotient x/y . This partial
// quotient can be used in some argument reduction algorithms.
//
// Returns
//
// Function remquol returns x REM y (as required by IEC 60559).
// Also, in the object pointed to by quo, function remquol stores
// the signed lowest three bits of the integer value closest to
// the quotient x/y.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, October 2003
//
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long double
remquol(long double x, long double y, int *quo)
{
    long double NewX, Retval;
    int		CrntRndDir, IntPart, YexpBits;

    *quo   = 0;

    if (isnanl(x) || isnanl(y))
    {
    	Retval = NAN;
    }
    else if (isinfl(x) || (y == 0))
    {
       	__math_set_errno(EDOM);
    	Retval = NAN;
	__fp_raise_except(FE_INVALID);
    }
    else if (isinfl(y))
    {
    	Retval = x;
    }
    else
    {
        GET_LDOUBLE_EXP(YexpBits, y);

        if ((YexpBits & 0x7fff) < 0x7ffc)	// Reduce x
	    NewX = fmodl(x, 8 * y);		// x < 8 * y
        else
            NewX = x;

        CrntRndDir = fegetround();
        fesetround(FE_TONEAREST);

        IntPart = rintl(NewX / y);		// x/y rounded to int

        fesetround(CrntRndDir);

        Retval = NewX - IntPart * y;

	if (Retval == 0)
	    Retval = copysignl(Retval, x);

	*quo = (IntPart >= 0) ? IntPart & 7	// get lowest 3 bits
			      : -((-IntPart) & 7);
    }

    return  Retval;
}
