// ------------
// nextafterl.c
// ------------
//
//
//	long double nextafterl(long double ArgX, long double ArgY);
//
//	Description
//
//	Function nextafterl(ArgX, ArgY) determines the next representable
//	value after ArgX in the direction of ArgY, in long double precision.
//	Function nextafterl(ArgX, ArgY) returns ArgY if ArgX equals ArgY. A
//	range error occurs if the magnitude of ArgX is the largest finite
//	value representable in the type and the result is infinite or
//	not representable in the type.
//
//      -- nextafterl(ArgX, ArgY) raises the overflow and inexact
//         exceptions if ArgX is finite and the function value is
//         infinite.
//
//      -- nextafterl(ArgX, ArgY) raises the underflow and inexact
//         exceptions if the function value is subnormal or zero
//         and ArgX!=ArgY.
//
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, July 2003
//
//
// pseudocode for nextafterl(ArgX, ArgY)
// * Split arguments into exponent and fraction via GET64_LDOUBLE
// * if ArgX == ArgY, return ArgY
// * if either argument is NaN, return NaN
// * if ArgX == 0, return minimum subnormal with sign of ArgY
// * if ArgX > 0 and ArgX > ArgY, ArgX = ArgX - 1 ULP
// * if ArgX > 0 and ArgX < ArgY, ArgX = ArgX + 1 ULP
// * if ArgX < 0 and ArgX < ArgY, ArgX = ArgX - 1 ULP
// * if ArgX < 0 and ArgX > ArgY, ArgX = ArgX + 1 ULP
// * At end, Test for whether final ArgX is either Infinity or denormal
// * if isinfl(Retval) and isfinite(ArgX),
// *	raise overflow & inexact exceptions
// * if Retval is subnormal or zero
// *	raise underflow & inexact exceptions

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

static const long double Basic_ULP = (1.0L/(TWO_32*TWO_32));

# if defined __STDC__
long double
nextafterl(long double ArgX, long double ArgY)
# else
long double
nextafterl(ArgX, ArgY)
long double ArgX, ArgY;
# endif
{
    LDBL	Retval;

    if (ArgX == ArgY)
    {
	Retval = ArgY;
    }
    else if (isnanl(ArgX) || isnanl(ArgY))
    {
    	Retval = ArgX + ArgY;
    }
    else if (ArgX == 0.0L)
    {
	SET_LDOUBLE_WORDS(Retval, 0, 0, 1);
	Retval = copysignl(Retval, ArgY);
    }
    else
    {
    	LDBL	ArgX_ULP;
	int	ExpX;
	LDBL	FracX;
    	USHORT	SgnExpX;
	ULLONG  FracBitsX;

	GET64_LDOUBLE(SgnExpX, FracBitsX, ArgX);

	if ((SgnExpX & 0x7fff) == 0)
	{
	    FracX = ArgX;
	    SET64_LDOUBLE(ArgX_ULP, SgnExpX, 1);
	}
	else
	{
	    FracX = frexpl(ArgX, &ExpX);
	    ArgX_ULP = copysignl(Basic_ULP, FracX);
	}

// * if ArgX > 0 and ArgX > ArgY, ArgX = ArgX - 1 ULP
// * if ArgX > 0 and ArgX < ArgY, ArgX = ArgX + 1 ULP
// * if ArgX < 0 and ArgX < ArgY, ArgX = ArgX - 1 ULP
// * if ArgX < 0 and ArgX > ArgY, ArgX = ArgX + 1 ULP

	if (ArgX > 0)
	{
	    if (ArgX > ArgY)
	    {
	    	Retval = FracX - ArgX_ULP;

	    	if (FracBitsX == (ULLONG)(1LL<<63))
		{
		    GET64_LDOUBLE(SgnExpX, FracBitsX, Retval);
		    SET64_LDOUBLE(Retval, SgnExpX, (ULLONG)(-1));
		}
	    }
	    else
	    {
	    	Retval = FracX + ArgX_ULP;
	    }
	}
	else			// ArgX < 0
	{
	    if (ArgX < ArgY)
	    {
	    	Retval = FracX - ArgX_ULP;

	    	if (FracBitsX == (ULLONG)(1LL<<63))
		{
		    GET64_LDOUBLE(SgnExpX, FracBitsX, Retval);
		    SET64_LDOUBLE(Retval, SgnExpX, (ULLONG)(-1));
		}
	    }
	    else
	    {
	    	Retval = FracX + ArgX_ULP;
	    }
	}

	if ((SgnExpX & 0x7fff) != 0)
	    Retval = ldexpl(Retval, ExpX);
    }

// * if isinfl(Retval) and isfinite(ArgX), set errno to ERANGE,
// *	raise overflow & inexact exceptions
// * if Retval is subnormal or zero and since ArgX != ArgY
// *	raise underflow & inexact exceptions

    if (isinfl(Retval) && isfinitel(ArgX))
    {
    	__math_set_errno(ERANGE);
        __fp_raise_except(FE_OVERFLOW | FE_INEXACT);
    }
    else
    {
	USHORT	SgnExpRet;
	ULLONG	FrcBitsRet;
        GET64_LDOUBLE(SgnExpRet, FrcBitsRet, Retval);
        if ((FrcBitsRet == 0) || ((SgnExpRet & 0x7fff) == 0))
        {
	    __fp_raise_except(FE_UNDERFLOW | FE_INEXACT);
	}
    }

    return Retval;
}
