/* modfl.c
 *
 * Split argument x into integral and fractional parts.
 *
 * Calling sequence:
 *
 *   #include <fdlibml.h>
 *
 *	fx = modfl(x, *ix);
 *
 * Prototype:
 *
 *     long double modfl(long double value, long double *iptr);
 *
 *  Description (from ISO C99)
 *
 *  The modfl function breaks the argument value into integral
 *  and fractional parts, each of which has the same type and
 *  sign as the argument. The integral part is stored in the
 *  object pointed to by iptr (in floating-point format).
 *
 *  Returns
 *
 *  The modfl function returns the value of the signed fractional
 *  part of the argument.
 *
 *  modfl(+-Inf, *iptr) returns +-0 and stores +-Inf in the object
 *  pointed to by *iptr.
 *
 *  modfl of a NaN argument returns a NaN and stores a NaN in
 *  the object pointed to by *iptr.
 *
 *
 * Written for DJGPP/GCC by KB Williams,
 * kbwms@aol.com, July 2003
 *
 */

#include <fdlibml.h>

long double __modfl(long double Arg, long double *Iptr);
long double __modfl(long double Arg, long double *Iptr)
{
    return modfl(Arg, Iptr);
}    		       

long double modfl(long double Arg, long double *Iptr)
{
    LDBL    Retval;

    if (isnanl(Arg))
    {					/* Arg is NaN */
	*Iptr = NAN;
	Retval = NAN;
    }
    else if (isinfl(Arg))
    {					/* Arg is Inf */
	*Iptr = Arg;
	Retval = copysignl(0.0L, Arg);
    }
    else
    {
	*Iptr = truncl(Arg);
	Retval = copysignl(Arg - *Iptr, Arg);
    }

    return Retval;
}
