// ------------
// lrintl.c
// ------------
//
// From ISO C99:
//
// Function lrintl
//
// Synopsis
//
//	#include <math.h>
//	long int lrintl(long double x);
//
// Description
//
// The lrintl function rounds its argument to the nearest
// integer value, rounding according to the current rounding
// direction. If the rounded value is outside the range of
// the return type, the numeric result is unspecified.	A
// range error may occur if the magnitude of x is too large.
// In this case, global variable errno is set to ERANGE.
//
// Returns
//
// The lrintl function returns the rounded integer value.
//
// Remarks
//
// The lrintl function provides floating-to-integer conversion
// as prescribed by IEC 60559. It rounds according to the current
// rounding direction.	If the rounded value is outside the range
// of the return type, the numeric result is undefined and the
// invalid exception is raised.	 When no other exception is raised
// during the conversion and the result differs from the argument,
// the inexact exception is raised.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, March 2002, revised September 2003
//
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long int lrintl(long double Arg)
{
    int		CrntExcepts, Exponent;
    long double LdblIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    ROUND_TO_INT(Arg, LdblIntVal);

    (void)frexpl(Arg, &Exponent);

    if ((Exponent > 31) || ! isfinitel(Arg))
    {
	__math_set_errno(EDOM);
	LdblIntVal = -LONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != LdblIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long int) LdblIntVal;
}
