/*							log1pl.c
 *
 *	Relative error logarithm
 *	Natural logarithm of 1+x, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, log1pl();
 *
 * y = log1pl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of 1+x.
 *
 * The argument 1+x is separated into its exponent and fractional
 * parts.  If the exponent is between -1 and +1, the logarithm
 * of the fraction is approximated by
 *
 *     log(1+x) = x - 0.5 x^2 + x^3 P(x)/Q(x).
 *
 * Otherwise, setting  z = 2(x-1)/(x+1),
 *
 *     log(x) = z + z^3 P(z)/Q(z).
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE     -1.0, 9.0    100000	8.2e-20	   2.5e-20
 *
 * ERROR MESSAGES:
 *
 * log singularity:  x-1 = 0; returns -HUGE_VALL
 * log domain:	     x-1 < 0; returns NaN
 */

/*
Cephes Math Library Release 2.8:  April, 2001
Copyright 2001 by Stephen L. Moshier
Converted to DJGPP/GCC by KB Williams,
kbwms@aol.com, December 2001 & October 2003
*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* Coefficients for log(1+x) = x - x^2 / 2 + x^3 P(x)/Q(x)
 * 1/sqrt(2) <= x < sqrt(2)
 * Theoretical peak relative error = 2.32e-20
 */

/* *INDENT-OFF* */

static long double
    P6 = 4.5270000862445199635215E-5L,
    P5 = 4.9854102823193375972212E-1L,
    P4 = 6.5787325942061044846969E0L,
    P3 = 2.9911919328553073277375E1L,
    P2 = 6.0949667980987787057556E1L,
    P1 = 5.7112963590585538103336E1L,
    P0 = 2.0039553499201281259648E1L;

static long double
/*  Q6 = 1.00000000000000000000E0L,*/
    Q5 = 1.5062909083469192043167E1L,
    Q4 = 8.3047565967967209469434E1L,
    Q3 = 2.2176239823732856465394E2L,
    Q2 = 3.0909872225312059774938E2L,
    Q1 = 2.1642788614495947685003E2L,
    Q0 = 6.0118660497603843919306E1L;

/* Coefficients for log(x) = z + z^3 P(z^2)/Q(z^2),
 * where z = 2(x-1)/(x+1)
 * 1/sqrt(2) <= x < sqrt(2)
 * Theoretical peak relative error = 6.16e-22
 */

static long double
    R3 = 1.9757429581415468984296E-3L,
    R2 = -7.1990767473014147232598E-1L,
    R1 = 1.0777257190312272158094E1L,
    R0 = -3.5717684488096787370998E1L;

static long double
/*  S3 = 1.00000000000000000000E0L,*/
    S2 = -2.6201045551331104417768E1L,
    S1 = 1.9361891836232102174846E2L,
    S0 = -4.2861221385716144629696E2L;

static long double C1 = 6.9314575195312500000000E-1L;
static long double C2 = 1.4286068203094172321215E-6L;

// ----------------------------------------------------
/* *INDENT-ON* */

# if defined __STDC__
long double log1pl(long double xm1)
# else
long double log1pl(xm1)
long double xm1;
# endif
{
    long double r, s;
    long double x, y, z, Retval;
    int	    e;
					//printf("log1pl(%Lg)\n", xm1);
    if ((isinfl(xm1) > 0) || (isnanl(xm1)) || (xm1 == 0))
    {
	Retval = xm1;
    }
    else if (fpclassifyl(xm1) == FP_SUBNORMAL)
    {
    	__math_set_errno(ERANGE);
	Retval = xm1;
	__fp_raise_except(FE_UNDERFLOW);
    }
    else
    {
	x = xm1 + 1.0L;

	/* Test for domain errors.  */
	if (x <= 0.0L)
	{
	    if (x == 0.0L)
	    {
		__math_set_errno(ERANGE);
		Retval = -HUGE_VALL;
		__fp_raise_except(FE_DIVBYZERO);
	    }
	    else
	    {
		__math_set_errno(EDOM);
		Retval = NAN;
		__fp_raise_except(FE_INVALID);
	    }
	}
	else
	{
	    /* Separate mantissa from exponent.
	       Use frexp so that denormal numbers
	       will be handled properly.  */

	    x = frexpl(x, &e);

	    /* logarithm using log(x) = z + z^3 P(z)/Q(z),
	       where z = 2(x-1)/(x+1)  */

	    if ((e > 2) || (e < -2))
	    {
		if (x < SQRTH)
		{			/* 2( 2x-1 )/( 2x+1 ) */
		    e -= 1;
		    z = x - 0.5L;
		    y = 0.5L * z + 0.5L;
		}
		else
		{			/*  2 (x-1)/(x+1)   */
		    z = x - 0.5L;
		    z -= 0.5L;
		    y = 0.5L * x + 0.5L;
		}

		x = z / y;
		z = x * x;

/* *INDENT-OFF* */
		r = R0 + z * (R1+z*(R2+z*R3));
		s = S0 + z * (S1+z*(S2+z));
/* *INDENT-ON* */
		z = x * (z * r / s);
		z = z + e * C2;
		z = z + x;
		z = z + e * C1;
		Retval = z;
	    }

	    else
	    {
		/* logarithm using log(1+x) = x - .5x**2 + x**3 P(x)/Q(x) */

		if (x < SQRTH)
		{
		    e -= 1;
		    if (e != 0)
			x = 2.0 * x - 1.0L;
		    else
			x = xm1;
		}
		else
		{
		    if (e != 0)
			x = x - 1.0L;
		    else
			x = xm1;
		}

/* *INDENT-OFF* */
		r = P0 + x * (P1+x*(P2+x*(P3+x*(P4+x*(P5+x*P6)))));
		s = Q0 + x * (Q1+x*(Q2+x*(Q3+x*(Q4+x*(Q5+x)))));
/* *INDENT-ON* */

		z = x * x;
		y = x * (z * r / s);
		y = y + e * C2;
		z = y - 0.5 * z;
		z = z + x;
		z = z + e * C1;
		Retval = z;
	    }
	}
    }
    return Retval;
}
