// ------------
// llrintl.c
// ------------
//
// From ISO C99:
//
// Function llrintl
//
// Synopsis
//
//	#include <math.h>
//	long long int llrintl(long double x);
//
// Description
//
// Using the current rounding direction, function llrintl
// rounds its argument to the nearest integer value and
// returns a numeric value of type long long int. If the
// rounded value is outside the range of type long long
// int, the value of the result is undefined and global
// variable errno is set to ERANGE.
//
// Returns
//
// The llrintl function returns the rounded integer value.
//
// Remarks (C99 paragraph 7.9.6.5)
//
// The llrintl function provides floating-to-integer conversion
// as prescribed by IEC 60559. It rounds according to the current
// rounding direction.	If the rounded value is outside the range
// of the return type, the numeric result is undefined and the
// invalid exception is raised.	 When no other exception is raised
// during the conversion and the result differs from the argument,
// the inexact exception is raised.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, March 2002, revised September 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long long llrintl(long double Arg)
{
    int		CrntExcepts, Exponent;
    long double LdblIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    ROUND_TO_INT(Arg, LdblIntVal);

    (void)frexpl(LdblIntVal, &Exponent);

    if ((Exponent > 63) || ! isfinitel(Arg))
    {
	__math_set_errno(EDOM);
	LdblIntVal = -LLONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != LdblIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long long) LdblIntVal;
}
