// Function fmodl()
//
// From ISO C99:
//
// Synopsis
//
//	#include <math.h>
//	long double fmodl(long double x, long double y);
//
// Description
//
// Function fmodl computes the floating-point remainder
// of x/y.
//
// Returns
//
// Function fmodl() returns the value x-ny, for some integer
// n such that, if y is nonzero, the result has the same sign
// as x and magnitude less than the magnitude of y.
//
// Special Values
//
//	o fmodl(+-0, y) returns +-0 if y is not zero.
//
//	o fmodl(x, y) returns a NaN and raises the invalid
//	  exception if x is infinite or y is zero.  In either
//	  case errno is set to EDOM.
//
//	o fmodl(x, +-Inf) returns x if x is not infinite.
//
//
// Written by J.T. Conklin <jtc@netbsd.org>.
// Public domain.
//
// Adapted for `long double' by Ulrich Drepper <drepper@cygnus.com>.
//
// Modified for DJGPP/GNU by KB Williams, kbwms@aol.com,
// February 2002
//
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

#define AV	__asm__ __volatile__

long double
fmodl(long double x, long double y)
{
    long double Retval;

    if (isinfl(x) || y == 0)
    {
    	Retval = NAN;
    	__math_set_errno(EDOM);
	__fp_raise_except(FE_INVALID);
    }
    else if (isnanl(x) || isnanl(y))
    {
    	Retval = NAN;
    }
    else
    {
	feclearexcept(FE_UNDERFLOW);

        AV ("1:\tFPREM      \n"
	    "FNSTSW   %%ax\n"
	    "SAHF	       \n"
	    "JP	     1b\n"
	    "FSTP	   %%st(1)"
            : "=t" (Retval) : "0" (x), "u" (y) : "ax", "st(1)");

        if (fetestexcept(FE_UNDERFLOW))
        {
            __math_set_errno(ERANGE);
        }
    }
    return (Retval);
}
