// ----------
// fesetenv.c
// ----------
//
// Prepared for DJGPP/GCC by KB Williams, kbwms@aol.com,
// February 2002
//
// From ISO C99:
//
// Function fesetenv
//
// Synopsis
//
//	#include <fenv.h>
//	int fesetenv(const fenv_t *envp);
//
// Description
//
// Function fesetenv establishes the floating-point environment
// represented by the object pointed to by envp.  The argument
// envp shall point to an object set by a call to fegetenv or
// feholdexcept, or equal either to the macro FE_DFL_ENV or to
// an implementation-defined environment macro.	 Note that
// fesetenv merely installs the state of the exception flags
// represented through its argument, and does not raise these
// exceptions.
//
// Return
//
// Function fesetenv returns zero if and only if the floating-point
// environment was successfully established.
//

#include <fenv.h>

#define AV	__asm__ __volatile__
#define LOAD_FPU_ENV(FE)\
    {					\
	AV ("FWAIT");			\
	AV ("FLDENV %0" : : "m" (FE));	\
    }

int
fesetenv(const fenv_t * envp)
{
    int	    Retval;
    fenv_t  FPU_Env;

    if (envp)
    {
	Retval = 0;			// Non-null pointer

	if (envp == FE_DFL_ENV)
	{
	    FPU_Env = *(FE_DFL_ENV);
	}
	else
	{
	    FPU_Env = *envp;
# if 0
	    FPU_Env.Control_Word &=
		~(FE_ALL_EXCEPT | FE_TONEAREST | FE_PC_EXTENDED);
	    FPU_Env.Control_Word |=
		(FE_ALL_EXCEPT | FE_TONEAREST | FE_PC_EXTENDED);

	    FPU_Env.Status_Word &= ~FE_ALL_EXCEPT;
# endif
	}

	LOAD_FPU_ENV(FPU_Env)
    }
    else
    {
	Retval = -1;			// Null pointer
    }

    return Retval;
}
# if 0
#include <stdio.h>

int
main(void)
{
    long double x, y, z;
    fenv_t  FPU_Env;

    x = 1; y = 3; z = x / y;

    // Get Environment after Flt. Pt. Calc.
    AV (" FSTENV %0":"=m"(*&FPU_Env));

    printf("Initial:  FPU_Env.Control_Word	  = %.4hx\n",
       (short)FPU_Env.Control_Word);
    printf("Initial:  FPU_Env.Status_Word	  = %.4hx\n",
	(short)FPU_Env.Status_Word);
    printf("Initial:  FPU_Env.Tag_Word		  = %.4hx\n",
       (short)FPU_Env.Tag_Word);
    printf("Initial:  FPU_Env.IP_Offset		  = %.8x\n",
       FPU_Env.IP_Offset);
    printf("Initial:  FPU_Env.CS_Selector	  = %.4hx\n",
       (short)FPU_Env.CS_Selector);
    printf("Initial:  FPU_Env.Opcode		  = %.4hx\n",
       (short)FPU_Env.Opcode	 );
    printf("Initial:  FPU_Env.Data_Operand_Offset = %.8x\n",
       FPU_Env.Data_Operand_Offset);
    printf("Initial:  FPU_Env.Operand_Selector	  = %.4hx\n",
       (short)FPU_Env.Operand_Selector);

    // Set default environment in FPU
    fesetenv(FE_DFL_ENV);

    // Get it
    AV ("FSTENV %0":"=m"(*&FPU_Env));

    printf("Default:  FPU_Env.Control_Word	  = %.4hx\n",
       (short)FPU_Env.Control_Word);
    printf("Default:  FPU_Env.Status_Word	  = %.4hx\n",
	(short)FPU_Env.Status_Word);
    printf("Default:  FPU_Env.Tag_Word		  = %.4hx\n",
       (short)FPU_Env.Tag_Word);
    printf("Default:  FPU_Env.IP_Offset		  = %.8x\n",
       FPU_Env.IP_Offset);
    printf("Default:  FPU_Env.CS_Selector	  = %.4hx\n",
       (short)FPU_Env.CS_Selector);
    printf("Default:  FPU_Env.Opcode		  = %.4hx\n",
       (short)FPU_Env.Opcode	 );
    printf("Default:  FPU_Env.Data_Operand_Offset = %.8x\n",
       FPU_Env.Data_Operand_Offset);
    printf("Default:  FPU_Env.Operand_Selector	  = %.4hx\n",
       (short)FPU_Env.Operand_Selector);

    // Modify Default with Flt. Pt. Calc.
    x = 1; y = 3; z = x / y;

    AV ("FSTENV %0":"=m"(*&FPU_Env));
    fesetenv(&FPU_Env);

    printf("(After)  FPU_Env.Control_Word = %.8x\n",
	FPU_Env.Control_Word);
    printf("(After)  FPU_Env.Status_Word  = %.8x\n",
	FPU_Env.Status_Word);
}
# endif
