/*							expl.c
 *
 *	Exponential function, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, expl();
 *
 * y = expl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns e (2.71828...) raised to the x power.
 *
 * Range reduction is accomplished by separating the argument
 * into an integer k and fraction f such that
 *
 *     x    k  f
 *    e	 = 2  e.
 *
 * A Pade' form of degree 2/3 is used to approximate exp(f) - 1
 * in the basic range [-0.5 ln 2, 0.5 ln 2].
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	+-10000	    50000	1.12e-19    2.81e-20
 *
 *
 * Error amplification in the exponential function can be
 * a serious matter.  The error propagation involves
 * exp( X(1+delta) ) = exp(X) ( 1 + X*delta + ... ),
 * which shows that a 1 lsb error in representing X produces
 * a relative error of X times 1 lsb in the function.
 * While the routine gives an accurate result for arguments
 * that are exactly represented by a long double precision
 * computer number, the result contains amplified roundoff
 * error for large arguments not exactly represented.
 *
 *
 * ERROR MESSAGES:
 *
 *   message	     condition	    value returned
 * exp underflow    x < MINLOG	       0.0
 * exp overflow	    x > MAXLOG	       MAXNUM
 *
 */
//
// Cephes Math Library Release 2.7:  May, 1998
// Copyright 1984, 1991, 1998 by Stephen L. Moshier
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, December 2001 & October 2003
//

/*	Exponential function	*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* *INDENT-OFF* */
static long double P[3] =
{
    1.2617719307481059087798E-4L,
    3.0299440770744196129956E-2L,
    9.9999999999999999991025E-1L,
};
static long double Q[4] =
{
    3.0019850513866445504159E-6L,
    2.5244834034968410419224E-3L,
    2.2726554820815502876593E-1L,
    2.0000000000000000000897E0L,
};
static long double C1 = 6.9314575195312500000000E-1L;
static long double C2 = 1.4286068203094172321215E-6L;
/* *INDENT-ON* */

long double
expl(long double Arg)
{
    long double Retval;

    if (!isfinitel(Arg))
    {
	if (isnanl(Arg) || (Arg > 0))
	{
	    Retval = (Arg);		/* Arg is NaN or +Inf */
	}
	else
	{
	    Retval = 0;			/* Arg is -Inf */
	}
    }
    else if (Arg >= MAXLOGL)
    {
	if (Arg == MAXLOGL)
	{
	    Retval = LDBL_MAX;
	}
	else
	{
	    __math_set_errno(ERANGE);
	    Retval = (HUGE_VALL);
	    __fp_raise_except(FE_OVERFLOW);
	}
    }
    else if (Arg == 0)
    {
    	Retval = 1.0L;
    }
    else
    {
    /* Express e**Arg = e**g 2**n
     *	 = e**g e**( n loge(2) )
     *	 = e**( g + n loge(2) )
     */
	int	n;
	long double ArgSq, T;

	if (Arg <= MINLOGL)		// Arg <= logl(LDBL_MIN)
    	{
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_UNDERFLOW);
	}
	T = floorl(M_LOG2E * Arg+0.5L); /* floor() truncates downward. */
	n = T;
	Arg -= T * C1;
	Arg -= T * C2;

	/* rational approximation for exponential
	 * of the fractional part:
	 * e**Arg =  1 + 2Arg P(Arg**2)/(Q(Arg**2) - Arg P(Arg**2))
	 */
	ArgSq = Arg * Arg;

	T = Arg * (P[2]+ArgSq*(P[1]+ArgSq*P[0]));

	Retval =  T / (Q[3]+ArgSq*(Q[2]+ArgSq*(Q[1]+ArgSq*Q[0])) - T);
	Retval = 1.0L + ldexpl(Retval, 1);
	Retval = ldexpl(Retval, n);
    }
    return (Retval);
}
