/* coshl.c -- long double version of cosh.c.
 * Conversion to long double by KB Williams,
 * kbwms@aol.com, December 2001 & October 2003
 */
/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */

/* coshl(x)
 * Method :
 * mathematically coshl(x) if defined to be (exp(x)+exp(-x))/2
 *	1. Replace x by |x| (coshl(x) = coshl(-x)).
 *	2.
 *							[ exp(x) - 1 ]^2
 *	    0	     <= x <= ln2/2  :  coshl(x) := 1 + -------------------
 *							   2*exp(x)
 *
 *						   exp(x) +  1/exp(x)
 *	    ln2/2    <= x <= 22	    :  coshl(x) := -------------------
 *							   2
 *	    22	     <= x <= lnovft :  coshl(x) := expl(x)/2
 *	    lnovft   <= x <= ln2ovft:  coshl(x) := expl(x/2)/2 * expl(x/2)
 *	    ln2ovft  <	x	    :  coshl(x) := huge*huge (overflow)
 *
 * Special cases:
 *	coshl(x) is |x| if x is +INF, -INF, or NaN.
 *	only coshl(0)=1 is exact for finite x.
 */

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long double 
coshl(long double Arg)
{
    long double AbsArg, t, w, Retval;

    AbsArg = fabsl(Arg);

    if (isinfl(Arg))
    {					//printf("cosh() range 1\n");
	Retval = AbsArg;
    }
    else if (isnanl(Arg))
    {
	Retval = Arg;
    }
    else if (AbsArg < 1.0L/TWO_33)
    {					//printf("cosh() range 2\n");
	Retval = 1.0L;
    }
    /* if |Arg| in (0,0.5*ln2],
		return 1+expm1l(|Arg|)^2/(2*expl(|Arg|)) */
    else if (AbsArg < 0.5L*LOGE2L)
    {					//printf("cosh() range 3\n");
	t = expm1l(AbsArg);
	w = 1.0L + t;
	Retval = 1.0L + (t * t) / (w + w);
    }
    /* if |Arg| in [0.5*ln2,22),
		return (exp(|Arg|)+1/exp(|Arg|)/2; */
    else if (AbsArg < 22.0L)
    {					//printf("cosh() range 4\n");
	t = expl(AbsArg);
	Retval = 0.5L * t + 0.5L / t;
    }

    /* if |Arg| in [22, ln(maxdouble)] return 0.5L*exp(|Arg|) */
    else if (AbsArg < MAXLOGL)
    {					//printf("cosh() range 5\n");
	Retval = 0.5L * expl(AbsArg);
    }
    /* if |Arg| in [log(maxdouble), log(overflow threshold)] */
    else if (AbsArg < (MAXLOGL + LOGE2L))
    {					//printf("cosh() range 6\n");
	w = expl(0.5L * AbsArg);	/* avoid overflow */
	Retval = 0.5L * w;
	Retval *= w;
    }
    /* if |Arg| > overflow threshold, coshl(x) overflow */
    else
    {					//printf("cosh() range 7\n");
	__math_set_errno(ERANGE);
	Retval = HUGE_VALL;
	__fp_raise_except(FE_OVERFLOW);

    }

    return Retval;
}
