/*							atanhl.c
 *
 *	Inverse hyperbolic tangent, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, atanhl();
 *
 * y = atanhl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns inverse hyperbolic tangent of argument in the range
 * MINLOGL to MAXLOGL.
 *
 * If |x| < 0.5, the rational form x + x**3 P(x)/Q(x) is
 * employed.  Otherwise,
 *	  atanh(x) = 0.5 * log( (1+x)/(1-x) ).
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	-1,1	    30000	1.1e-19	    3.3e-20
 *
 */



/*
Cephes Math Library Release 2.7:  May, 1998
Copyright (C) 1987, 1991, 1998 by Stephen L. Moshier
Converted to DJGPP/GCC by KB Williams, kbwms@aol.com, December 2001
*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* *INDENT-OFF* */
static long double
    P5 =  2.9647757819596835680719E-3L,
    P4 = -8.0026596513099094380633E-1L,
    P3 =  7.7920941408493040219831E0L,
    P2 = -2.4330686602187898836837E1L,
    P1 =  3.0204265014595622991082E1L,
    P0 = -1.2961142942114056581210E1L;

static long double
/*  Q5 = 1.0000000000000000000000E0L,*/
    Q4 = -1.3729634163247557081869E1L,
    Q3 =  6.2320841104088512332185E1L,
    Q2 = -1.2469344457045341444078E2L,
    Q1 =  1.1394285233959210574352E2L,
    Q0 = -3.8883428826342169425890E1L;

#ifdef __STDC__
long double atanhl(long double);
extern long double fabsl(long double);
extern long double logl(long double);
extern int signbitl(long double);
#else
long double atanhl();
long double fabsl(), logl();
int  signbitl();
#endif
/* *INDENT-ON* */

# ifdef __STDC__
long double atanhl(long double Arg)
# else
long double atanhl(Arg)
long double Arg;
# endif
{
    long double Retval, AbsArg, Sq;

    if (Arg == 0.0L)
    {
	Retval = Arg;
    }
    else
    {
	AbsArg = fabsl(Arg);
	
	if (AbsArg >= 1.0L)
	{
	    if (AbsArg == 1.0L)
	    {			// Set +-Inf, raise div by 0 exception
		__math_set_errno(ERANGE);
		Retval = copysignl(HUGE_VALL, Arg);
		__fp_raise_except(FE_DIVBYZERO);
	    }
	    else
	    {			// Set NaN, raise invalid exception
		__math_set_errno(EDOM);
		Retval = NAN;
		__fp_raise_except(FE_INVALID);
	    }
	}
	else if (AbsArg < 1.0e-8L)
	{
	    Retval = Arg;

	    if (fpclassifyl(Arg) == FP_SUBNORMAL)
	    {
	    	__math_set_errno(ERANGE);
	    	__fp_raise_except(FE_UNDERFLOW);
	    }
	}
	else if (AbsArg < 0.5L)
	{
	    long double p, q;
	    Sq = Arg * Arg;
	    p = P0 + Sq * (P1+Sq*(P2+Sq*(P3+Sq*(P4+Sq*P5))));
	    q = Q0 + Sq * (Q1+Sq*(Q2+Sq*(Q3+Sq*(Q4+Sq))));
	    Retval = Arg + Arg*Sq*(p/q);
	}
	else
	{
	    Retval = 0.5L * logl((1.0L + Arg) / (1.0L - Arg));
	}
    }
    return Retval;
}
