/*							atan2l.c
 *
 *	Quadrant correct inverse circular tangent,
 *	long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, z, atan2l();
 *
 * z = atan2l( y, x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns radian angle whose tangent is y/x.
 * Range -PI < z <= +PI, args (y,x);
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	-10, 10	    60000	1.7e-19	    3.2e-20
 * See atanl.c.
 *
 */

/*						atan2l.c */


/*
Cephes Math Library Release 2.7:  May, 1998
Copyright 1984, 1990, 1998 by Stephen L. Moshier
Modified for DJGPP/GCC by KB Williams, 
kbwms@aol.com, December 2001, October 2003 &
		January 2004
*/
#include <errno.h>
#include <fenv.h>
#include <fdlibml.h>


long double
atan2l(long double y, long double x)
{
    long double z, w;
    int		code, 
    		XisInf = isinfl(x), 
		YisInf = isinfl(y);

    code = 0;

    if (x < 0.0L)
	code = 2;
    if (y < 0.0L)
	code |= 1;

    if (isnanl(x))
	return (x);
    if (isnanl(y))
	return (y);

    if (y == 0.0L)
    {
	if (signbitl(y))
	{
	    if (x > 0.0L)
		z = y;
	    else if (x < 0.0L)
		z = -PIL;
	    else
	    {
		if (signbitl(x))
		    z = -PIL;
		else
		    z = y;
	    }
	}
	else				/* y is +0 */
	{
	    if (x == 0.0L)
	    {
		if (signbitl(x))
		    z = PIL;
		else
		    z = 0.0L;
	    }
	    else if (x > 0.0L)
		z = 0.0L;
	    else
		z = PIL;
	}
	return z;
    }
    if (x == 0.0L)
    {
	if (y > 0.0L)
	    z = PIO2L;
	else
	    z = -PIO2L;
	return z;
    }
    if (XisInf > 0)			// +Inf
    {
    	if (YisInf)			// +-Inf
	{				       
	    z = copysignl(0.25L * PIL, y);
	}
	else
	{
	    z = copysignl(0.0L, y);
	}
	 return z;
    }
    if (XisInf < 0)			// -Inf
    {
    	if (YisInf)			// +-Inf
	{
	    z = copysignl(0.75L * PIL, y);
	}						 	       
	else
	{
	    z = copysignl(PIL, y);
	}
	 return z;
    }
    if (YisInf)				// +-Inf
    {
    	return copysignl(PIO2L, y);
    }
    			       
    if (x == 0.0L)
    {
	if (code & 1)
	{
	    return (-PIO2L);
	}
	if (y == 0.0L)
	    return (0.0L);
	return (PIO2L);
    }

    if (y == 0.0L)
    {
	if (code & 2)
	    return (PIL);
	return (0.0L);
    }


    switch (code)
    {
    default:
    case 0:
    case 1:
	w = 0.0L;
	break;
    case 2:
	w = PIL;
	break;
    case 3:
	w = -PIL;
	break;
    }

    z = y / x;
    if (z == 0.0L)
    {		
	z = w;	    		       
	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);
    }
    else if (fpclassifyl(z) == FP_SUBNORMAL)
    {
    	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);
    }				       
    else
    {		
	z = w + atanl(z);
    }
    if (z == 0.0L && y < 0.0L)
	z = -0.0L;
    
    return (z);
}
