/*
 * dllink - dynamic linking system
 * Version:  0.1
 *
 * Copyright (C) 2005  Daniel Borca   All Rights Reserved.
 *
 * dllink is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * dllink is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Make; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * This code is loosely based on Stonewheel 0.0.8 by Burton Radons
 * (loth@pacificcoast.net), repackaged by Peter Wang (tjaden@psynet.net)
 */


#include <libc/stubs.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <elf.h>
#include <crt0/list.h>
#include <crt0/internal.h>


#define SUB_VADDR(x)	if (x >= (int)vaddr) x = x - vaddr
#define ROUND_UP(x)	((x + 15) & ~15)
#define ELF_ERROR()	goto fail
#define CHECK_STACK()


#define LOADABLE(psh)	(psh.sh_flags & SHF_ALLOC)
#define TYP_RESIDENT	(1 << 0)
#define TYP_STRINGS	(1 << 1)


extern stone_object objects;

static char dl_error_string[256];


unsigned __elf_vaddr (int f, int infile, Elf32_Ehdr *hdr);
stone_object *__elf_scan_symbol (const char *name, stone_sym **ret, stone_object *obj, stone_object *all);
int __elf_resolve (int report, stone_object *obj, stone_object *all);


static int
dos_open (const char *path)
{
    int fd = _open(path, O_RDONLY);
    if (fd == -1) {
	int i;
	for (i = 0; path[i] != '\0'; i++) {
	    if (path[i] == ':' || path[i] == '/' || path[i] == '\\') {
		break;
	    }
	}
	if (path[i] == '\0') {
	    char *env = getenv("LD_LIBRARY_PATH");
	    if (env != NULL) {
		char *tmp = strdup(env);
		if (tmp != NULL) {
		    char *full = malloc(strlen(tmp) + strlen(path) + 2);
		    if (full != NULL) {
			char *last = tmp - 1;
			do {
			    char ch, *p = last + 1;
			    last = strchr(p, ';');
			    if (last != NULL) {
				*last = '\0';
			    }
			    strcpy(full, p);
			    ch = (p[0] == '\0') ? '/' : p[strlen(p) - 1];
			    if (ch != '/' && ch != '\\') {
				strcat(full, "/");
			    }
			    strcat(full, path);
			    fd = _open(full, O_RDONLY);
			} while (fd < 0 && last != NULL);
			free(full);
		    }
		    free(tmp);
		}
	    }
	}
    }
    return fd;
}


/* XXX maybe merge this with elven.c#f_read() */
static int
f_read (void *ptr, int size, int f, long offset)
{
    if (lseek(f, offset, SEEK_SET) == -1) {
	return 0;
    }
    return (_read(f, ptr, size) == size);
}


/* XXX maybe merge this with elven.c#readsection() */
static char *
readsection (int file, char **dat, int idx, Elf32_Shdr *sh, int infile)
{
    if (dat[idx] == NULL) {
	dat[idx] = malloc(sh[idx].sh_size);
    }
    if (dat[idx] != NULL) {
	if (!f_read(dat[idx], sh[idx].sh_size, file, infile + sh[idx].sh_offset)) {
	    return NULL;
	}
    }
    return dat[idx];
}


/* XXX maybe merge this with elven.c#elf_load() */
static stone_object *
elf_load (int f, int infile, int mode, const char *src, stone_object *all)
{
    stone_object *obj;
    Elf32_Ehdr hdr;
    Elf32_Shdr *sh;
    char **dat, *typ;
    int i, c, old;
    int nsym, nrel, nsyn, ndep, len;
    char *base, *strings;
    unsigned vaddr;

    if (!f_read(&hdr, sizeof(hdr), f, infile + 0)) {
	return NULL;
    }
    if (hdr.e_ident[EI_MAG0] != ELFMAG0 ||
	hdr.e_ident[EI_MAG1] != ELFMAG1 ||
	hdr.e_ident[EI_MAG2] != ELFMAG2 ||
	hdr.e_ident[EI_MAG3] != ELFMAG3) {
	return NULL;
    }
    if (hdr.e_ident[EI_CLASS] != ELFCLASS32 ||
	hdr.e_ident[EI_DATA] != ELFDATA2LSB ||
	hdr.e_ident[EI_VERSION] != EV_CURRENT ||
	hdr.e_machine != EM_386 ||
	hdr.e_shentsize != sizeof(Elf32_Shdr)) {
	return NULL;
    }
    if (hdr.e_type != ET_EXEC &&
	hdr.e_type != ET_DYN) {
	return NULL;
    }
    sh = alloca(hdr.e_shnum * sizeof(Elf32_Shdr));
    CHECK_STACK();
    if (!f_read(sh, sizeof(Elf32_Shdr) * hdr.e_shnum, f, infile + hdr.e_shoff)) {
	return NULL;
    }
    dat = alloca(hdr.e_shnum * sizeof(char *));
    CHECK_STACK();
    for (c = 0; c < hdr.e_shnum; c++) {
	dat[c] = NULL;
    }
    typ = alloca(hdr.e_shnum * sizeof(char));
    CHECK_STACK();
    for (c = 0; c < hdr.e_shnum; c++) {
	typ[c] = 0;
    }

    vaddr = __elf_vaddr(f, infile, &hdr);

    /* calculate sizes */
    nsym = nrel = nsyn = ndep = len = 0;
    strings = (char *)(strlen(src) + 1);
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_REL) {
	    if (sh[c].sh_entsize == sizeof(Elf32_Rel) &&
		/* XXX sh_info weirdness */
		(sh[c].sh_info == 0 || LOADABLE(sh[sh[c].sh_info]))) {
		nrel += sh[c].sh_size / sizeof(Elf32_Rel);
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	}
	if (LOADABLE(sh[c])) {
	    sh[c].sh_addr -= vaddr;
	    if (len < (int)sh[c].sh_addr + (int)sh[c].sh_size) {
		len = sh[c].sh_addr + sh[c].sh_size;
		dat[c] = (char *)sh[c].sh_addr;
		typ[c] = TYP_RESIDENT;
	    }
	}
    }
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_SYMTAB) {
	    /* SHT_SYMTAB is only needed by ET_EXEC */
	    if (sh[c].sh_entsize == sizeof(Elf32_Sym) && hdr.e_type == ET_EXEC) {
		/* XXX Attention: can't have multiple SYMTAB/DYNSYM */
		nsym += sh[c].sh_size / sizeof(Elf32_Sym);
		if (!typ[sh[c].sh_link]) {
		    dat[sh[c].sh_link] = strings;
		    strings += sh[sh[c].sh_link].sh_size;
		    typ[sh[c].sh_link] = TYP_STRINGS;
		}
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	} else if (sh[c].sh_type == SHT_DYNSYM) {
	    if (sh[c].sh_entsize == sizeof(Elf32_Sym)) {
		/* XXX Attention: can't have multiple SYMTAB/DYNSYM */
		nsyn += sh[c].sh_size / sizeof(Elf32_Sym);
		if (!typ[sh[c].sh_link]) {
		    dat[sh[c].sh_link] = strings;
		    strings += sh[sh[c].sh_link].sh_size;
		    typ[sh[c].sh_link] = TYP_STRINGS;
		}
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	} else if (sh[c].sh_type == SHT_DYNAMIC) {
	    if (sh[c].sh_entsize == sizeof(Elf32_Dyn)) {
		ndep += sh[c].sh_size / sizeof(Elf32_Dyn);
		if (!typ[sh[c].sh_link]) {
		    dat[sh[c].sh_link] = strings;
		    strings += sh[sh[c].sh_link].sh_size;
		    typ[sh[c].sh_link] = TYP_STRINGS;
		}
	    } else {
		sh[c].sh_type = SHT_NULL;
	    }
	}
    }

    /* allocate data:
     * +------------------+
     * | object structure |
     * +------------------+
     * |      strings     |
     * +------------------+
     * |      symtab      |
     * +------------------+
     * |      dynsym      |
     * +------------------+
     * |      dynamic     |
     * +------------------+
     */
    i = ROUND_UP(sizeof(stone_object) + (long)strings);
    i += ROUND_UP(nsym * sizeof(stone_sym) +
		  nsyn * sizeof(stone_sym) +
		  nrel * sizeof(stone_rel) +
		  ndep * sizeof(char *));
    i += ROUND_UP(len);

    obj = calloc(1, i);
    if (obj == NULL) {
	return NULL;
    }

    /* fill up object */
    obj->len = len;
    obj->nsym = nsym;
    obj->nsyn = nsyn;
    obj->nrel = nrel;
    obj->ndep = ndep;
    obj->mode = mode;
    obj->type = hdr.e_type;
    list_append(all, obj);

    /* set up pointers (ET_DYN will have obj->sym == obj->dyn) */
    base = (char *)obj + sizeof(stone_object);
    obj->src = base;
    obj->sym = (stone_sym *)ROUND_UP(((long)base + (long)strings));
    obj->syn = (stone_sym *)((char *)obj->sym + nsym * sizeof(stone_sym));
    obj->rel = (stone_rel *)((char *)obj->syn + nsyn * sizeof(stone_sym));
    obj->dep = (char **)((char *)obj->rel + nrel * sizeof(stone_rel));
    obj->code = (char *)ROUND_UP((long)obj->dep + ndep * sizeof(char *));
    strings = base;
    strcpy(base, src);

    /* fixup pointers */
    for (c = 0; c < hdr.e_shnum; c++) {
	if (typ[c] == TYP_RESIDENT) {
	    dat[c] += (long)obj->code;
	} else if (typ[c] == TYP_STRINGS) {
	    dat[c] += (long)strings;
	}
    }

    /* load the bulk */
    for (c = 0; c < hdr.e_shnum; c++) {
	if (typ[c]) {
	    if (sh[c].sh_type == SHT_NOBITS) {
		memset(dat[c], 0, sh[c].sh_size);
	    } else {
		if (!f_read(dat[c], sh[c].sh_size, f, infile + sh[c].sh_offset)) {
		    ELF_ERROR();
		}
	    }
	}
    }

    /* now read in the symbols (ET_DYN won't load SHT_SYMTAB) */
    obj->nsym = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_SYMTAB) {
	    Elf32_Sym *sym;
	    stone_sym *sobj;
	    char *str;

	    sym = (Elf32_Sym *)readsection(f, dat, c, sh, infile);
	    str = readsection(f, dat, sh[c].sh_link, sh, infile);
	    if (sym == NULL || str == NULL) {
		ELF_ERROR();
	    }

	    old = obj->nsym;
	    obj->nsym += sh[c].sh_size / sizeof(Elf32_Sym);

	    for (i = old, sobj = obj->sym + old; i < obj->nsym; sym++, i++, sobj++) {
		sobj->name = &str[sym->st_name];
		sobj->obj = NULL;
		sobj->sym = NULL;
		sobj->type = csst_unknown;

		if (ELF32_ST_BIND(sym->st_info) == STB_GLOBAL ||
		    ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
		    sobj->type = csst_export;
		    if (sym->st_shndx == SHN_UNDEF) {
			sobj->type = csst_import;
		    }
		    if (ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
			sobj->type |= csst_weak;
		    }
		}
		if (sym->st_shndx < hdr.e_shnum || sym->st_shndx == SHN_ABS) {
		    sobj->value = sym->st_value;
		} else {
		    ELF_ERROR();
		}
		SUB_VADDR(sobj->value);
	    }
	}
    }

    /* now read in the symbols */
    obj->nsyn = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_DYNSYM) {
	    Elf32_Sym *sym;
	    stone_sym *sobj;
	    char *str;

	    sym = (Elf32_Sym *)readsection(f, dat, c, sh, infile);
	    str = readsection(f, dat, sh[c].sh_link, sh, infile);
	    if (sym == NULL || str == NULL) {
		ELF_ERROR();
	    }

	    old = obj->nsyn;
	    obj->nsyn += sh[c].sh_size / sizeof(Elf32_Sym);

	    for (i = old, sobj = obj->syn + old; i < obj->nsyn; sym++, i++, sobj++) {
		sobj->name = &str[sym->st_name];
		sobj->obj = NULL;
		sobj->sym = NULL;
		sobj->type = csst_unknown;

		if (ELF32_ST_BIND(sym->st_info) == STB_GLOBAL ||
		    ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
		    sobj->type = csst_export;
		    if (sym->st_shndx == SHN_UNDEF) {
			sobj->type = csst_import;
		    }
		    if (ELF32_ST_BIND(sym->st_info) == STB_WEAK) {
			sobj->type |= csst_weak;
		    }
		}
		if (sym->st_shndx < hdr.e_shnum || sym->st_shndx == SHN_ABS) {
		    sobj->value = sym->st_value;
		} else {
		    ELF_ERROR();
		}
		SUB_VADDR(sobj->value);
	    }
	}
    }

    if (obj->sym == obj->syn) {
	obj->nsym = obj->nsyn;
    }

    /* and the links */
    obj->nrel = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_REL) {
	    Elf32_Rel *rel = (Elf32_Rel *)readsection(f, dat, c, sh, infile);
	    stone_rel *robj;
	    stone_sym *tab = (sh[sh[c].sh_link].sh_type == SHT_SYMTAB) ? obj->sym : obj->syn;

	    if (rel == NULL) {
		ELF_ERROR();
	    }

	    old = obj->nrel;
	    obj->nrel += sh[c].sh_size / sizeof(Elf32_Rel);

	    for (i = old, robj = obj->rel + old; i < obj->nrel; rel++, i++, robj++) {
		int offset = rel->r_offset;
		SUB_VADDR(offset);

		robj->off = &obj->code[offset];
		robj->sym = &tab[ELF32_R_SYM(rel->r_info)];
		robj->type = csr_unknown;
		robj->addend = 0;

		if (ELF32_R_TYPE(rel->r_info) == R_386_JMP_SLOT ||	/* S */
		    ELF32_R_TYPE(rel->r_info) == R_386_GLOB_DAT) {	/* S */
		    robj->type = csr_type1;
		    /*plt = *(int *)&obj->code[*(int *)robj->off + 6 - vaddr] + *(int *)robj->off + 0xA - vaddr;*/
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_RELATIVE) {/* B + A */
		    robj->type = csr_type2;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_32) {	/* S + A */
		    robj->type = csr_type3;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_PC32) {	/* S + A - P */
		    robj->type = csr_type4;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_GOTOFF) {	/* S + A - GOT */
		    /* XXX ET_EXEC:
		     * robj->addend + &_GLOBAL_OFFSET_TABLE_ -> symbol address
		     * no further relocation is required
		     */
		    robj->type = csr_fixed;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_GOTPC) {	/* GOT + A - P */
		    /* XXX ET_EXEC:
		     * robj->addend + EBX -> _GLOBAL_OFFSET_TABLE_
		     * no further relocation is required
		     */
		    robj->type = csr_fixed;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_GOT32) {	/* G + A - P */
		    /* XXX ET_EXEC:
		     * _GLOBAL_OFFSET_TABLE_[robj->addend] = &symbol
		     * no further relocation is required
		     */
		    robj->type = csr_fixed;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_PLT32) {	/* L + A - P */
		    /* XXX FIXME: this call will never go through PLT */
		    robj->type = csr_type4;
		    robj->addend = *(int *)robj->off;
		    SUB_VADDR(robj->addend);
		} else if (ELF32_R_TYPE(rel->r_info) == R_386_NONE) {
		    robj->type = csr_fixed;
		}
	    }
	}
    }

    /* collect the dependencies */
    obj->ndep = 0;
    for (c = 0; c < hdr.e_shnum; c++) {
	if (sh[c].sh_type == SHT_DYNAMIC) {
	    Elf32_Dyn *dyn;
	    char **dobj;
	    char *str;

	    dyn = (Elf32_Dyn *)readsection(f, dat, c, sh, infile);
	    str = readsection(f, dat, sh[c].sh_link, sh, infile);

	    if (dyn == NULL || str == NULL) {
		ELF_ERROR();
	    }

	    old = obj->ndep;
	    ndep = old + sh[c].sh_size / sizeof(Elf32_Dyn);

	    for (i = old, dobj = obj->dep + old; i < ndep; i++) {
		if (dyn[i].d_tag == DT_NEEDED) {
		    dobj[0] = &str[dyn[i].d_un.d_val];
		    dobj++;
		    obj->ndep++;
		} else if (dyn[i].d_tag == DT_INIT) {
		    obj->init = (void (*) (void))(dyn[i].d_un.d_ptr - vaddr + obj->code);
		} else if (dyn[i].d_tag == DT_FINI) {
		    obj->fini = (void (*) (void))(dyn[i].d_un.d_ptr - vaddr + obj->code);
		/*} else if (dyn[i].d_tag == DT_PLTGOT) {
		    got = dyn[i].d_un.d_ptr - vaddr;*/
		}
	    }
	}
    }

    /* finish allocation now */
    for (c = 0; c < hdr.e_shnum; c++) {
	if (dat[c] != NULL && !typ[c]) {
	    free(dat[c]);
	}
    }

    /* external modules */
    for (i = 0; i < obj->ndep; i++) {
	stone_object *dep = dlopen(obj->dep[i], RTLD_GLOBAL | RTLD_LAZY);
	if (dep == NULL) {
	    ELF_ERROR();
	}
    }

    return obj;

  fail:
    list_remove(obj);
    for (c = 0; c < hdr.e_shnum; c++) {
	if (dat[c] != NULL && !typ[c]) {
	    free(dat[c]);
	}
    }
    free(obj);
    return NULL;
}


static void
elf_free (stone_object *obj)
{
    int i;
    stone_rel *rel;

    for (rel = obj->rel, i = 0; i < obj->nrel; rel++, i++) {
	stone_sym *sym = rel->sym;
	if (sym->obj != NULL && sym->obj != obj) {
	    sym->obj->user--;
	}
    }

    list_remove(obj);
    free(obj);
}


/* XXX maybe merge this with elven.c#do_load() */
void *
dlopen (const char *filename, int mode)
{
    int fd;
    stone_object *obj;

    /* check if already loaded */
    list_foreach (obj, &objects) {
	if (!strcmp(obj->src, filename)) {
	    return obj;
	}
    }

    fd = dos_open(filename);
    if (fd < 0) {
	sprintf(dl_error_string, "1cannot open `%s'", filename);
	return NULL;
    }

    obj = elf_load(fd, 0, mode & RTLD_GLOBAL, filename, &objects);
    _close(fd);

    /* link if RTLD_NOW or have to fire ctors, else wait for first `dlsym' */
    if (obj != NULL && ((mode & RTLD_NOW) || obj->init)) {
	__elf_resolve(0, obj, &objects);
	if (!obj->link) {
	    elf_free(obj);
	    obj = NULL;
	    sprintf(dl_error_string, "1cannot link `%s'", filename);
	} else if (obj->init) {
	    obj->init();
	}
    }

    return obj;
}


int
dlclose (void *handle)
{
    stone_object *obj;
    list_foreach (obj, &objects) {
	if (obj == handle) {
	    if (!obj->user) {
		/* try to invoke destructors */
		if (obj->fini && (obj->link || __elf_resolve(0, obj, &objects))) {
		    obj->fini();
		}
		elf_free(obj);
		return 0;
	    }
	    break;
	}
    }
    return -1;
}


const char *
dlerror (void)
{
    const char *err = dl_error_string[0] ? (dl_error_string + 1) : NULL;
    dl_error_string[0] = 0;
    return err;
}


void *
dlsym (void *handle, const char *symbol_name)
{
    stone_sym *ret;
    void *ptr = NULL;
    stone_object *obj = __elf_scan_symbol(symbol_name, &ret, handle, &objects);
    if (obj != NULL && (obj->link || __elf_resolve(0, obj, &objects))) {
	ptr = &obj->code[ret->value];
    }
    return ptr;
}
