/* -*- coding: cp852 -*-

  DJGPP Socket Wrapper for Microsoft Network Client  
  written by Tomasz Zbroek (2007/2008) / UPOS System Sp. z o.o.
  Released to the Public Domain.

*/

#include <dpmi.h>
#include <go32.h>
#include <string.h>
#include <sys/farptr.h>
#include <errno.h>
#include <libc/dosio.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "drventry.h"
#include "result.h"
#include "basmac.h"

static const char* aErrors[] = {
  /* 100 */ "Socket operation on non-socket", 
  /* 101 */ "Destination address required",
  /* 102 */ "Message too long",
  /* 103 */ "Protocol wrong type for socket",
  /* 104 */ "Protocol not available",
  /* 105 */ "Protocol not supported",
  /* 106 */ "Socket type not supported",
  /* 107 */ "Operation not supported on socket",
  /* 108 */ "Protocol family not supported",
  /* 109 */ "Address family not supported by protocol family",
  /* 110 */ "Address already in use",
  /* 111 */ "Can't assign requested address",
  /* 112 */ "Network is down",
  /* 113 */ "Network is unreachable",
  /* 114 */ "Network dropped connection on reset",
  /* 115 */ "Software caused connection abort",
  /* 116 */ "Connection reset by peer",
  /* 117 */ "No buffer space available",
  /* 118 */ "Socket is already connected",
  /* 119 */ "Socket is not connected",
  /* 120 */ "Can't send after socket shutdown",
  /* 121 */ "Connection timed out",
  /* 122 */ "Connection refused",
  /* 123 */ "Networking subsystem not started",
  /* 124 */ "No route to host",
  /* 125 */ "Operation would block",
  /* 126 */ "Operation now in progress",
  /* 127 */ "Operation already in progress",
  /* 128 */ "Library/driver version mismatch",
  /* 129 */ "Invalid argument to sockets call",
  /* 130 */ "Too many open sockets",
  /* 131 */ "Bad address to sockets call",

  /* additional DJGPP error messages */
  /* 132 */ "Unable to execute MS Client procedure",
  /* 133 */ "Unable to allocate dos memory",  
  /* 134 */ "Unable to free dos memory",
  /* 135 */ "Transfer buffer too small",
};

static const int def_nErrors = TABLE_SIZE(aErrors);

static const word def_nParamOffset = 64; // offset in transfer buffer for command parameters (first 64 bytes are used for the command itself)
static const word def_nBufSize = 1024 * 10; // transfer buffer size for command parameters

/****************************************************************/
inline int Bytes2Paragraphs(int i_nSize)
{
  return (i_nSize + 15) / 16;
}

/****************************************************************/
/** Returns MS Client error description
 * \return error description pointer
 */
char *sock_strerror(int i_nErrNo)
{
  int nNo = i_nErrNo - 100;
  if(nNo < 0 || nNo > def_nErrors)
    return strerror(errno);
  return (char *) aErrors[nNo];
}

/****************************************************************/  
/** Socket initialization
 * \return error (<0)
 */
int socket(int i_nAf, int i_nType, int i_nProtocol)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;

  sCmd.yCmd = eCmd_Socket;
  sCmd.wParam1 = i_nAf;
  sCmd.wParam2 = i_nType;
  sCmd.wParam3 = i_nProtocol;

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 

  return nResult;
}

/****************************************************************/
/** Close socket
 * \return error (<0)
 */
int close(int i_nSocket)
{
  return close_socket(i_nSocket);
}

int close_socket(int i_nSocket)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;

  sCmd.yCmd = eCmd_Close;
  sCmd.wParam1 = i_nSocket;

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo;     

  return nResult;
}

/****************************************************************/
/** Select - check client queue 
 * \return error (<0)
 */
int select(int i_nFdsNum, 
           fd_set * i_pReadFds, 
           fd_set * i_pWriteFds, 
           fd_set * i_pExceptFds,
           struct timeval * i_pTimeout) 
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if( sizeof(fd_set) * 3 + sizeof(struct timeval) > def_nBufSize )
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  memset(&sCmd, 0, sizeof (struct SCommand));

  if(i_pReadFds != NULL)
    {
      dosmemput(i_pReadFds, sizeof(fd_set), dwBuf); 
      sCmd.wParam2 = wOffset; 
      sCmd.wParam3 = wSegment;
    }
  if(i_pWriteFds != NULL)
    {
      dosmemput(i_pWriteFds, sizeof(fd_set), dwBuf + sizeof(fd_set) );   
      sCmd.wParam4 = wOffset + sizeof(fd_set); 
      sCmd.wParam5 = wSegment;
    }
  if(i_pExceptFds != NULL)
    {
      dosmemput(i_pExceptFds, sizeof(fd_set), dwBuf + (sizeof(fd_set) * 2) );   
      sCmd.wParam6 = wOffset + sizeof(fd_set) * 2; 
      sCmd.wParam7 = wSegment;
    }
  if(i_pTimeout != NULL)
    {
      dosmemput(i_pTimeout, sizeof(struct timeval), dwBuf + (sizeof(fd_set) * 3) );   
      sCmd.wParam8 = wOffset + sizeof(fd_set) * 3; 
      sCmd.wParam9 = wSegment;
    }

  // command
  sCmd.yCmd = eCmd_Select;
  sCmd.wParam1 = i_nFdsNum;
  
  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  else
    {
      if(i_pReadFds != NULL)
        dosmemget(dwBuf, sizeof(fd_set), i_pReadFds);
      if(i_pWriteFds != NULL)
        dosmemget(dwBuf + sizeof(fd_set), sizeof(fd_set), i_pWriteFds);   
      if(i_pExceptFds != NULL)
        dosmemget(dwBuf + (sizeof(fd_set) * 2), sizeof(fd_set), i_pExceptFds);   
    }

  return nResult;
}

/****************************************************************/
/** Ioctl
 * \return error (<0)
 */
int ioctl(int i_nSocket, int i_nRequest, int *io_pParam)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(sizeof(word) > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  _farpokew(_dos_ds, dwBuf, *io_pParam);

  // command
  sCmd.yCmd = eCmd_Ioctl;
  sCmd.wParam1 = i_nSocket;
  sCmd.wParam2 = i_nRequest;
  sCmd.wParam3 = wOffset; 
  sCmd.wParam4 = wSegment; 

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  else
    *io_pParam = _farpeekw(_dos_ds, dwBuf);

  return nResult;
}

/****************************************************************/
/** @todo unfinished
 */
struct servent* getservbyname(const char *i_pServName, const char *i_pProtName)
{
  if(strlen(i_pServName)==0)
    return NULL;

  SetServent();
}

/****************************************************************/
/** Get host name 
 * \return error (<0)
 */
int gethostname(char *o_pHost, int i_nHostMaxLen)
{
  int nRes;
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(i_nHostMaxLen > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  // command
  sCmd.yCmd = eCmd_GetHostName;
  sCmd.wParam1 = wOffset;
  sCmd.wParam2 = wSegment;
  sCmd.wParam3 = i_nHostMaxLen;

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  else
    dosmemget(dwBuf, i_nHostMaxLen, o_pHost);

  return nResult;
}

/****************************************************************/
/** Bind socket
 * \return error (<0)
 */
int bind(int i_nSocket, struct sockaddr_in * i_pAddr, int i_nAddrSize)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(i_nAddrSize > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  // command
  sCmd.yCmd = eCmd_Bind;
  sCmd.wParam1 = i_nSocket;
  sCmd.wParam2 = wOffset;
  sCmd.wParam3 = wSegment;
  sCmd.wParam4 = i_nAddrSize;

  dosmemput(i_pAddr, i_nAddrSize, dwBuf);

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 

  return nResult;
}

/****************************************************************/
/** Listen 
 * \return error (<0)
 */
int listen(int i_nSocket, int i_nMaxConn)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;

  sCmd.yCmd = eCmd_Listen;
  sCmd.wParam1 = i_nSocket;
  sCmd.wParam2 = i_nMaxConn;

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  
  return nResult;
}

/****************************************************************/
/** Connect
 * \return error (<0)
 */
int connect(int i_nSocket, struct sockaddr * i_pAddr, int i_nAddrSize)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(i_nAddrSize > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  // command
  sCmd.yCmd = eCmd_Connect;
  sCmd.wParam1 = i_nSocket;
  sCmd.wParam2 = wOffset;
  sCmd.wParam3 = wSegment;
  sCmd.wParam4 = i_nAddrSize;

  dosmemput(i_pAddr, i_nAddrSize, dwBuf);
  
  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 

  return nResult;
}

/****************************************************************/
/** Send data in stream mode (TCP)
 * \return error (<0)
 */
int send(int i_nSocket, char *i_pMessage, int i_nMsgSize, int i_nFlags)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(i_nMsgSize > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  // command
  sCmd.yCmd = eCmd_Send;
  sCmd.wParam1 = i_nSocket;
  sCmd.wParam2 = wOffset;
  sCmd.wParam3 = wSegment;
  sCmd.wParam4 = i_nMsgSize;
  sCmd.wParam5 = i_nFlags;
  sCmd.wParam6 = 0;
  sCmd.wParam7 = 0;
  sCmd.wParam8 = 0x10;
  sCmd.wParam9 = 0;

  dosmemput(i_pMessage, i_nMsgSize, dwBuf);
  
  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 

  return nResult;
}

/****************************************************************/
/** \brief Send data in datagram mode (UDP)
 * \return error (<0)
 */
int sendto(int i_nSocket, 
           char *i_pMessage, 
           int i_nMsgSize, 
           int i_nFlags, 
           struct sockaddr *i_pAddr, 
           int i_nAddrSize)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  int nMsgOffset = i_nAddrSize;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(i_nMsgSize + i_nAddrSize > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  dosmemput(i_pMessage, i_nMsgSize, dwBuf + nMsgOffset);
  dosmemput(i_pAddr, i_nAddrSize, dwBuf);

  // command
  sCmd.yCmd = eCmd_Send;
  sCmd.wParam1 = i_nSocket;
  
  // msg
  sCmd.wParam2 = wOffset + nMsgOffset;
  sCmd.wParam3 = wSegment;
  sCmd.wParam4 = i_nMsgSize;

  sCmd.wParam5 = i_nFlags;
  // addr
  sCmd.wParam6 = wOffset;
  sCmd.wParam7 = wSegment;
  sCmd.wParam8 = i_nAddrSize;
  sCmd.yParam9 = 1;
  
  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 

  return nResult;
}

/****************************************************************/
/** Driver exit
 * \return error (<0)
 */
int driver_exit(int i_nTmp)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;

  sCmd.yCmd = eCmd_Exit;

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  else
    DriverExit();

  return nResult;
}

/****************************************************************/
/** Accept
 * \return error (<0)
 */
int accept(int i_nSocket, struct sockaddr *i_pAddr, int *i_pAddrSize)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(*i_pAddrSize + sizeof(word) > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  // command
  sCmd.yCmd = eCmd_Accept;
  sCmd.wParam1 = i_nSocket;

  sCmd.wParam2 = wOffset; 
  sCmd.wParam3 = wSegment;

  sCmd.wParam4 = wOffset + *i_pAddrSize; 
  sCmd.wParam5 = wSegment; 

  dosmemput(i_pAddr, *i_pAddrSize, dwBuf);
  _farpokew(_dos_ds, dwBuf + *i_pAddrSize, *i_pAddrSize);

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  else
    dosmemget(dwBuf, *i_pAddrSize, i_pAddr);

  return nResult;
}

/****************************************************************/
/** Receive (stream socket)
 * \return error (<0)
 */
int recv(int i_nSocket, char *o_pMsg, int i_nMsgSize, int i_nFlags)
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(i_nMsgSize > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  // command
  sCmd.yCmd = eCmd_Recv;
  sCmd.wParam1 = i_nSocket;
  sCmd.wParam2 = wOffset; 
  sCmd.wParam3 = wSegment;
  sCmd.wParam4 = i_nMsgSize;
  sCmd.wParam5 = i_nFlags;
  sCmd.wParam6 = 0;
  sCmd.wParam7 = 0;
  sCmd.wParam8 = 0;
  sCmd.wParam9 = 0;
  // mode
  sCmd.yParam10 = 2;

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  else
    dosmemget(dwBuf, i_nMsgSize, o_pMsg);

  return nResult;
}

/****************************************************************/
/** Receive from (datagram socket)
 * \return error (<0)
 */
int recvfrom(int i_nSocket, 
             char *o_pMessage, 
             int i_nMsgSize, 
             int i_nFlags, 
             struct sockaddr *i_pAddr, 
             int *i_pAddrSize
             )
{
  sword nResult = 0;
  word wErrNo = 0;
  struct SCommand sCmd;
  int nMsgOffset = *i_pAddrSize + sizeof(word);
  const word wOffset = __tb_offset + def_nParamOffset;
  const word wSegment = __tb_segment;
  const dword dwBuf = __tb + def_nParamOffset;

  if(i_nMsgSize + *i_pAddrSize + sizeof(word) > def_nBufSize)
    {
      errno = eErrNo_TransferBufferTooSmall;
      return -1;
    }

  _farpokew(_dos_ds, dwBuf, *i_pAddrSize);
  dosmemput(i_pAddr, *i_pAddrSize, dwBuf + sizeof(word));

  // command
  sCmd.yCmd = eCmd_Recv;
  sCmd.wParam1 = i_nSocket;
  
  // buffer
  sCmd.wParam2 = wOffset + nMsgOffset;
  sCmd.wParam3 = wSegment;
  sCmd.wParam4 = i_nMsgSize;

  sCmd.wParam5 = i_nFlags;

  // addr
  sCmd.wParam6 = wOffset + sizeof(word);
  sCmd.wParam7 = wSegment;

  // addr size
  sCmd.wParam8 = wOffset;
  sCmd.wParam9 = wSegment;

  // mode
  sCmd.yParam10 = 3;

  nResult = DriverEntry(&sCmd, &wErrNo);
  if(nResult < 0)
    errno = wErrNo; 
  else
    dosmemget(dwBuf + nMsgOffset, i_nMsgSize, o_pMessage);

  return nResult;
}
