/* precomp.c -- example program: how to generate pre-compressed data

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1997 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with the LZO library; see the file COPYING.
   If not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


/*************************************************************************
// This program shows how to generate pre-compressed data.
//
// Please study LZO.FAQ and simple.c first.
//
// We will be trying both LZO1X-999 and LZO1Y-999 and choose
// the algorithm that achieves the best compression ratio.
**************************************************************************/

#include <lzoconf.h>
#if defined(LZO_999_UNSUPPORTED)
#  error "compression level 999 not supported on this 16bit system"
#endif

#include <lzo1x.h>
#include <lzo1y.h>


/*************************************************************************
//
**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

int main(int argc, char *argv[])
{
	int r;

	lzo_byte *in;
	lzo_uint in_len;

	lzo_byte *out;
	lzo_uint out_len;

	lzo_byte *wrkmem;
	lzo_uint wrk_len;

	lzo_uint best_len;
	int best_compress = -1;

	lzo_uint orig_len;
	lzo_uint32 uncompressed_checksum;
	lzo_uint32 compressed_checksum;

	FILE *f;
	const char *progname = 0;
	const char *in_name = 0;
	const char *out_name = 0;


	progname = argv[0];
	if (argc < 2 || argc > 3)
	{
		printf("usage: %s file [output-file]\n", progname);
		exit(1);
	}
 	in_name = argv[1];
 	out_name = argc == 3 ? argv[2] : 0;

/*
 * Step 1: initialize the LZO library
 */
	if (lzo_init() != LZO_E_OK)
	{
		printf("lzo_init() failed !!!\n");
		return 3;
	}

/*
 * Step 2: allocate the work-memory
 */
 	wrk_len = LZO1X_999_MEM_COMPRESS;
 	if (LZO1Y_999_MEM_COMPRESS > wrk_len)
	 	wrk_len = LZO1Y_999_MEM_COMPRESS;
	wrkmem = (lzo_byte *) malloc(wrk_len);
	if (wrkmem == 0)
	{
		printf("%s: out of memory\n", progname);
		exit(1);
	}

/*
 * Step 3: open the input file
 */
 	f = fopen(in_name,"rb");
	if (f == 0)
	{
		printf("%s: cannot open file %s\n", progname, in_name);
		exit(1);
	}
	fseek(f,0,SEEK_END);
	in_len = ftell(f);
	best_len = in_len;
	fseek(f,0,SEEK_SET);

/*
 * Step 4: allocate compression buffers and read the file
 */
 	out_len = in_len + in_len / 64 + 16 + 3;
	in = (lzo_byte *) malloc(in_len);
	out = (lzo_byte *) malloc(out_len);
	if (in == 0 || out == 0)
	{
		printf("%s: out of memory\n", progname);
		exit(1);
	}
	in_len = fread(in,1,in_len,f);
	printf("%s: loaded file %s: %ld bytes\n", progname, in_name, (long) in_len);
	fclose(f);

/*
 * Step 5: compute a checksum of the uncompressed data
 */
 	uncompressed_checksum = lzo_adler32(0,NULL,0);
 	uncompressed_checksum = lzo_adler32(uncompressed_checksum,in,in_len);

/*
 * Step 6a: compress from `in' to `out' with LZO1X-999
 */
		r = lzo1x_999_compress(in,in_len,out,&out_len,wrkmem);
		if (r != LZO_E_OK)
		{
			/* this should NEVER happen */
			printf("internal error - compression failed: %d\n", r);
			exit(1);
		}
		printf("LZO1X-999: %8lu -> %8lu\n", (long) in_len, (long) out_len);
		if (out_len < best_len)
		{
			best_len = out_len;
			best_compress = 1;		/* LZO1X-999 */
		}

/*
 * Step 6b: compress from `in' to `out' with LZO1Y-999
 */
		r = lzo1y_999_compress(in,in_len,out,&out_len,wrkmem);
		if (r != LZO_E_OK)
		{
			/* this should NEVER happen */
			printf("internal error - compression failed: %d\n", r);
			exit(1);
		}
		printf("LZO1Y-999: %8lu -> %8lu\n", (long) in_len, (long) out_len);
		if (out_len < best_len)
		{
			best_len = out_len;
			best_compress = 2;		/* LZO1Y-999 */
		}

/*
 * Step 7: check if compressible
 */
 	if (best_len >= in_len)
	{
		printf("This file contains incompressible data.\n");
		return 0;
	}

/*
 * Step 8: compress data again using the best compressor found
 */
	if (best_compress == 1)
		r = lzo1x_999_compress(in,in_len,out,&out_len,wrkmem);
	else if (best_compress == 2)
		r = lzo1y_999_compress(in,in_len,out,&out_len,wrkmem);
	else
		r = -100;
	assert(r == LZO_E_OK);
	assert(out_len == best_len);

/*
 * Step 9: optimize compressed data (compressed data is in `out' buffer)
 */
 	orig_len = in_len;
	if (best_compress == 1)
		r = lzo1x_optimize(out,out_len,in,&orig_len,NULL);
	else if (best_compress == 2)
		r = lzo1y_optimize(out,out_len,in,&orig_len,NULL);
	else
		r = -100;
	if (r != LZO_E_OK || orig_len != in_len)
	{
		/* this should NEVER happen */
		printf("internal error - optimization failed: %d\n", r);
		exit(1);
	}

/*
 * Step 10: compute a checksum of the compressed data
 */
 	compressed_checksum = lzo_adler32(0,NULL,0);
 	compressed_checksum = lzo_adler32(compressed_checksum,out,out_len);

/*
 * Step 11: write compressed data to a file
 */
	printf("%s: %s: %ld -> %ld, checksum 0x%08lx 0x%08lx\n",
	        progname, in_name, (long) in_len, (long) out_len,
	        (long) uncompressed_checksum, (long) compressed_checksum);

	if (out_name && out_name[0])
	{
		printf("%s: writing to file %s\n", progname, out_name);
		f = fopen(out_name,"wb");
		if (f == 0)
		{
			printf("%s: cannot open output file %s\n", progname, out_name);
			exit(1);
		}
		if (fwrite(out,1,out_len,f) != out_len || fclose(f) != 0)
		{
			printf("%s: write error !!\n", progname);
			exit(1);
		}
	}


	return 0;
}

/*
vi:ts=4
*/

