/*  GNU SED, a batch stream editor.
    Copyright (C) 1999
    Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include "config.h"

#include <ctype.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifndef NULL
# include <stdio.h>
#endif

#include "basicdefs.h"
#include "utils.h"
#include "sed.h"
#include "regex-gnu.h"
#ifndef NULL
# define NULL CAST(VOID *)0
#endif

#ifdef gettext_noop
# define N_(String) gettext_noop(String)
#else
# define N_(String) (String)
#endif

extern flagT use_extended_syntax_p;

#define TRANSLATE_LENGTH 256 /*XXX shouldn't this be (UCHAR_MAX+1)?*/

static const char NO_REGEX[] = N_("No previous regular expression");



regex_t *
compile_regex(b, ignore_case, nosub)
  struct buffer *b;
  flagT ignore_case;
  flagT nosub;
{
  /* Keep track of the last regexp read.  We keep track the source
     form of the regular expression in case different modifiers are
     specified. */
  static char *last_re = NULL;
  static size_t last_re_len;
  regex_t *new_regex;
  const char * error;
  reg_syntax_t syntax = RE_SYNTAX_POSIX_BASIC;

  if (!b)
    {
      FREE(last_re);
      last_re = NULL;
      return NULL;
    }

  /* My reading of IEEE Std 1003.2-1992 is that // means the empty RE.
     But historical and common practice is that // "matches the last RE";
     thus this use of POSIXLY_CORRECT. */
  if (size_buffer(b) > 0 || POSIXLY_CORRECT)
    {
      size_t sz;
      FREE(last_re);
      sz = size_buffer(b);
      last_re = ck_memdup(get_buffer(b), sz);
      last_re_len = normalize_text(last_re, sz);
    }
  else if (!last_re)
    bad_prog(_(NO_REGEX));

  new_regex = MALLOC(1, regex_t);
  new_regex->buffer = NULL;
  new_regex->allocated = 0;
  new_regex->used = 0;
  new_regex->syntax = RE_SYNTAX_POSIX_BASIC;
  new_regex->fastmap = NULL;
  new_regex->translate = NULL;
  new_regex->regs_allocated = REGS_FIXED;
  new_regex->no_sub = !!nosub;
  new_regex->not_bol = 0;
  new_regex->not_eol = 0;
  new_regex->newline_anchor = 0;

  if (ignore_case)
    {
      static char *xlate = NULL;
      if (!xlate)
	{
	  int i;
	  /* Map uppercase characters to corresponding lowercase ones.  */
	  xlate = MALLOC(TRANSLATE_LENGTH, char);
	  for (i=0; i<TRANSLATE_LENGTH; ++i)
	    xlate[i] = ISUPPER(i) ? tolower(i) : i;
	}
      new_regex->translate = xlate;
    }

  if (use_extended_syntax_p)
      syntax = RE_SYNTAX_POSIX_EXTENDED;
  if (!POSIXLY_CORRECT)
    syntax &= ~RE_DOT_NOT_NULL;	/*XXX Is this really needed?*/
  syntax &= ~RE_HAT_LISTS_NOT_NEWLINE;
  syntax |= RE_DOT_NEWLINE;
  re_set_syntax(syntax);

  error = re_compile_pattern(last_re, last_re_len, new_regex);
  if (error)
    bad_prog(error);

  /* @#$*! re_compile_pattern() tweaks these inappropriately... */
  new_regex->regs_allocated = REGS_FIXED;
  new_regex->newline_anchor = 0;

  return new_regex;
}


int
match_regex(regex, buf, buflen, buf_start_offset, regarray)
  regex_t *regex;
  char *buf;
  size_t buflen;
  size_t buf_start_offset;
  struct re_registers *regarray;
{
  int ret =
    re_search_2(regex, NULL, 0,
		buf, CAST(int)buflen,
		CAST(int)buf_start_offset, CAST(int)buflen,
		regarray,
		CAST(int)buflen);
  return (ret >= 0);
}


#ifdef DEBUG_LEAKS
void
release_regex(regex)
  regex_t *regex;
{
    regfree(regex);
    FREE(regex);
}
#endif /*DEBUG_LEAKS*/
