/* Copyright (C) 1997,1998,1999,2000 by Salvador E. Tropea (SET),
   see copying file for details */
/**[txh]********************************************************************

  Module: Math 4
  Comments:
  This module is for ultra-fast/good accuracy co/sine waveforms.
  It uses 256 values to exploit the 8 bits wrap. In addition it uses some
fixed point math to allow increments of 0.0055 degrees.
  This module have a hardcoded table with double values and two tables
for integer values. The double's table is used to generate the other tables.
  Why two tables? because these tables are calculated with a customized
amplitud, offset and scale factors so you can re-use the tables.

***************************************************************************/

#include "math4.h"

// 360+90 degrees
int XSin256t[256+256/4];
int YSin256t[256+256/4];

/* Co/Sine table
Generated with:

int main(void)
{
 int i;
 printf("double DSin256t[256+256/4]={\n");
 for (i=0; i<320; )
    {
     printf("%17.15f",sin(i*2*PI/256));
     if (i!=319)
        printf(",");
     else
        printf(" };");
     i++;
     if (i%4==0)
        printf("\n");
    }
 return 0;
}

*/
double DSin256t[256+256/4]={
0.000000000000000,0.024541228522912,0.049067674327418,0.073564563599667,
0.098017140329561,0.122410675199216,0.146730474455362,0.170961888760301,
0.195090322016128,0.219101240156870,0.242980179903264,0.266712757474898,
0.290284677254462,0.313681740398892,0.336889853392220,0.359895036534988,
0.382683432365090,0.405241314004990,0.427555093430282,0.449611329654607,
0.471396736825998,0.492898192229784,0.514102744193222,0.534997619887097,
0.555570233019602,0.575808191417845,0.595699304492433,0.615231590580627,
0.634393284163645,0.653172842953777,0.671558954847018,0.689540544737067,
0.707106781186547,0.724247082951467,0.740951125354959,0.757208846506484,
0.773010453362737,0.788346427626606,0.803207531480645,0.817584813151584,
0.831469612302545,0.844853565249707,0.857728610000272,0.870086991108711,
0.881921264348355,0.893224301195515,0.903989293123443,0.914209755703531,
0.923879532511287,0.932992798834739,0.941544065183021,0.949528180593037,
0.956940335732209,0.963776065795440,0.970031253194544,0.975702130038529,
0.980785280403230,0.985277642388941,0.989176509964781,0.992479534598710,
0.995184726672197,0.997290456678690,0.998795456205172,0.999698818696204,
1.000000000000000,0.999698818696204,0.998795456205172,0.997290456678690,
0.995184726672197,0.992479534598710,0.989176509964781,0.985277642388941,
0.980785280403230,0.975702130038529,0.970031253194544,0.963776065795440,
0.956940335732209,0.949528180593037,0.941544065183021,0.932992798834739,
0.923879532511287,0.914209755703531,0.903989293123443,0.893224301195515,
0.881921264348355,0.870086991108711,0.857728610000272,0.844853565249707,
0.831469612302545,0.817584813151584,0.803207531480645,0.788346427626606,
0.773010453362737,0.757208846506485,0.740951125354959,0.724247082951467,
0.707106781186548,0.689540544737067,0.671558954847019,0.653172842953777,
0.634393284163645,0.615231590580627,0.595699304492433,0.575808191417845,
0.555570233019602,0.534997619887097,0.514102744193222,0.492898192229784,
0.471396736825998,0.449611329654607,0.427555093430282,0.405241314004990,
0.382683432365090,0.359895036534988,0.336889853392220,0.313681740398891,
0.290284677254462,0.266712757474898,0.242980179903264,0.219101240156870,
0.195090322016129,0.170961888760301,0.146730474455362,0.122410675199216,
0.098017140329561,0.073564563599668,0.049067674327418,0.024541228522912,
0.000000000000000,-0.024541228522912,-0.049067674327418,-0.073564563599667,
-0.098017140329561,-0.122410675199216,-0.146730474455362,-0.170961888760301,
-0.195090322016128,-0.219101240156870,-0.242980179903264,-0.266712757474898,
-0.290284677254462,-0.313681740398891,-0.336889853392220,-0.359895036534988,
-0.382683432365090,-0.405241314004990,-0.427555093430282,-0.449611329654607,
-0.471396736825998,-0.492898192229784,-0.514102744193222,-0.534997619887097,
-0.555570233019602,-0.575808191417845,-0.595699304492433,-0.615231590580627,
-0.634393284163645,-0.653172842953777,-0.671558954847018,-0.689540544737067,
-0.707106781186547,-0.724247082951467,-0.740951125354959,-0.757208846506484,
-0.773010453362737,-0.788346427626606,-0.803207531480645,-0.817584813151584,
-0.831469612302545,-0.844853565249707,-0.857728610000272,-0.870086991108711,
-0.881921264348355,-0.893224301195515,-0.903989293123443,-0.914209755703530,
-0.923879532511287,-0.932992798834739,-0.941544065183021,-0.949528180593037,
-0.956940335732209,-0.963776065795440,-0.970031253194544,-0.975702130038528,
-0.980785280403230,-0.985277642388941,-0.989176509964781,-0.992479534598710,
-0.995184726672197,-0.997290456678690,-0.998795456205172,-0.999698818696204,
-1.000000000000000,-0.999698818696204,-0.998795456205172,-0.997290456678690,
-0.995184726672197,-0.992479534598710,-0.989176509964781,-0.985277642388941,
-0.980785280403230,-0.975702130038529,-0.970031253194544,-0.963776065795440,
-0.956940335732209,-0.949528180593037,-0.941544065183021,-0.932992798834739,
-0.923879532511287,-0.914209755703531,-0.903989293123443,-0.893224301195515,
-0.881921264348355,-0.870086991108711,-0.857728610000272,-0.844853565249707,
-0.831469612302545,-0.817584813151584,-0.803207531480645,-0.788346427626606,
-0.773010453362737,-0.757208846506485,-0.740951125354959,-0.724247082951467,
-0.707106781186548,-0.689540544737067,-0.671558954847019,-0.653172842953777,
-0.634393284163646,-0.615231590580627,-0.595699304492433,-0.575808191417845,
-0.555570233019602,-0.534997619887097,-0.514102744193222,-0.492898192229784,
-0.471396736825998,-0.449611329654607,-0.427555093430283,-0.405241314004990,
-0.382683432365090,-0.359895036534988,-0.336889853392220,-0.313681740398892,
-0.290284677254462,-0.266712757474899,-0.242980179903264,-0.219101240156870,
-0.195090322016129,-0.170961888760302,-0.146730474455362,-0.122410675199216,
-0.098017140329561,-0.073564563599667,-0.049067674327418,-0.024541228522912,
0.000000000000000,0.024541228522912,0.049067674327418,0.073564563599667,
0.098017140329560,0.122410675199216,0.146730474455362,0.170961888760301,
0.195090322016128,0.219101240156870,0.242980179903264,0.266712757474898,
0.290284677254462,0.313681740398891,0.336889853392220,0.359895036534988,
0.382683432365090,0.405241314004990,0.427555093430282,0.449611329654607,
0.471396736825998,0.492898192229784,0.514102744193221,0.534997619887097,
0.555570233019602,0.575808191417845,0.595699304492433,0.615231590580627,
0.634393284163646,0.653172842953777,0.671558954847018,0.689540544737067,
0.707106781186547,0.724247082951467,0.740951125354959,0.757208846506484,
0.773010453362737,0.788346427626606,0.803207531480645,0.817584813151584,
0.831469612302545,0.844853565249707,0.857728610000272,0.870086991108711,
0.881921264348355,0.893224301195515,0.903989293123443,0.914209755703530,
0.923879532511287,0.932992798834739,0.941544065183021,0.949528180593037,
0.956940335732209,0.963776065795440,0.970031253194544,0.975702130038528,
0.980785280403230,0.985277642388941,0.989176509964781,0.992479534598710,
0.995184726672197,0.997290456678690,0.998795456205172,0.999698818696204 };

/**[txh]********************************************************************

  Include: math4.h
  Description:
  This function fills one of the two tables with the values provided.@p
  The equation is: (Amp*sin(x)+Off)*Scale.@p
  Scale is used to calculate Y offsets.@p
  The flags are:@*
@*
MA4NoScale  Don't use scale.@*
MA4Scale    Use scale (default).@*
MA4XSin     Fill the X table.@*
MA4YSin     Fill the Y table (default).@*

***************************************************************************/

void MA4_FillTable(double Amp, double Off, double Scale, int Flags)
{
 int i;

 if (Flags & MA4NoScale)
   {
    if (Flags & MA4XSin)
       for (i=0; i<320; i++)
           XSin256t[i]=Amp*DSin256t[i]+Off;
    else
       for (i=0; i<320; i++)
           YSin256t[i]=Amp*DSin256t[i]+Off;
   }
 else
   {
    if (Flags & MA4XSin)
       for (i=0; i<320; i++)
           XSin256t[i]=(Amp*DSin256t[i]+Off)*Scale;
    else
       for (i=0; i<320; i++)
           YSin256t[i]=(Amp*DSin256t[i]+Off)*Scale;
   }
}

/**[txh]********************************************************************

  Description:
  That's the integer version of @x{MA4_FillTable}.

***************************************************************************/

void MA4_FillTableI(int Amp, int Off, int Scale, int Flags)
{
 int i;

 if (Flags & MA4NoScale)
   {
    if (Flags & MA4XSin)
       for (i=0; i<320; i++)
           XSin256t[i]=(int)(Amp*DSin256t[i])+Off;
    else
       for (i=0; i<320; i++)
           YSin256t[i]=(int)(Amp*DSin256t[i])+Off;
   }
 else
   {
    if (Flags & MA4XSin)
       for (i=0; i<320; i++)
           XSin256t[i]=((int)(Amp*DSin256t[i]+Off))*Scale;
    else
       for (i=0; i<320; i++)
           YSin256t[i]=((int)(Amp*DSin256t[i]+Off))*Scale;
   }
}

/**[txh]********************************************************************

  Description:
  That's the integer version of @x{MA4_FillTable}. Additionally it forces
the values multiples of four.

***************************************************************************/

void MA4_FillTableIW(int Amp, int Off, int Scale, int Flags)
{
 int i;

 if (Flags & MA4NoScale)
   {
    if (Flags & MA4XSin)
       for (i=0; i<320; i++)
           XSin256t[i]=((int)(Amp*DSin256t[i])+Off) & 0xFFFFFFFC;
    else
       for (i=0; i<320; i++)
           YSin256t[i]=((int)(Amp*DSin256t[i])+Off) & 0xFFFFFFFC;
   }
 else
   {
    if (Flags & MA4XSin)
       for (i=0; i<320; i++)
           XSin256t[i]=(((int)(Amp*DSin256t[i]+Off))*Scale) & 0xFFFFFFFC;
    else
       for (i=0; i<320; i++)
           YSin256t[i]=(((int)(Amp*DSin256t[i]+Off))*Scale) & 0xFFFFFFFC;
   }
}


