/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains functions for timing and stop-watches.

*/
#ifndef s_timer_c
#define s_timer_c

#include "s_timer.h"

#include "s_defs.h"
#include "s_math.h"
#include "s_global.h"
#include "s_fade.h"

#include <allegro.h>

#include <math.h>



/********************************************
****                                     ****
**** timer_start, timer_read, timer_stop ****
****                                     ****
*********************************************
Call timer_start. Then call timer_read any number of times to get the time
since last call to timer_start. You can also call timer_start any number of
times to reset the counter. Call timer_pause to pause the timer (it also
returns current time). When paused, you can still use timer_read to get the
current time (but it will be the same all the time that the timer is paused).
Call timer_continue to continue the timing. Do as many pause/continues as you
want. Finally, call timer_stop to remove the timer.
timer_start and timer_continue return the same as install_int and the others
return the time. If timer_read or timer_stop is called before timer_start,
they return 0. If you call timer_continue when timer is off, it returns 1.
*/

//Some variables used by the timer functions.

//Stores the current time as the number of 1193181-parts of a second.
static volatile ulonglong  timer_counter=0;
//0=off, 1=paused, 2=on.
int            timer_mode=0;
//Tells wether the timer variables have been locked.
static int     timer_has_locked=FALSE;
//The numbers of 1193181-parts of a second between calls to the interrupt.
static ulong   timer_interval=0;

//Callback for timer.
void timer_callback(void)
{
timer_counter+=timer_interval;
}
END_OF_FUNCTION(timer_callback);
//Starts and resets timer.
int timer_start(ulong frequence)
{
int ret=0;
if(frequence==0)
	return 1;
timer_counter=0;
timer_interval=urounddiv(TIMERS_PER_SECOND,frequence);
if((timer_mode==0) || (timer_mode==1)){//If no timer is on, we install one.
	if(!timer_has_locked){
		LOCK_FUNCTION(timer_callback);
		LOCK_VARIABLE(timer_counter);
		timer_has_locked=TRUE;
	}
	ret=install_int_ex(timer_callback,timer_interval);
	if(ret)//Non-zero on error.
		timer_mode=0;
	else
		timer_mode=2;
}else{
	timer_mode=2;
	ret=0;
}
return ret;
}
//Returns time since last call to timer_start, as 1193181-parts of a second.
ulonglong timer_read_ex(void)
{
if((timer_mode==1)||(timer_mode==2))
	return timer_counter;
else
	return 0;
}
//Stops timer and returns time since last timer_start.
ulonglong timer_stop_ex(void)
{
ulonglong ret=timer_counter;
if(timer_mode==2)
	remove_int(timer_callback);
if(timer_mode==0)
	ret=0;
timer_mode=0;
timer_counter=0;
return ret;
}
//Pauses the timer.
ulonglong timer_pause_ex(void)
{
ulonglong ret=timer_counter;
switch(timer_mode){
	case 2:
		remove_int(timer_callback);
		timer_mode=1;
		return timer_counter=ret;
	break;
	case 1:
		return ret;
	break;
	default://Timer off.
		timer_mode=1;
		timer_counter=0;
		return 0;
	break;
}
}
//Unpauses the timer.
int timer_unpause(void)
{
int ret;
if(timer_mode==0){//Pretend timer is paused.
	timer_interval=urounddiv(TIMERS_PER_SECOND,1000);
	timer_counter=0;
	timer_mode=1;
}
if(timer_mode==1){
	if(!timer_has_locked){
		LOCK_FUNCTION(timer_callback);
		LOCK_VARIABLE(timer_counter);
		timer_has_locked=TRUE;
	}
	timer_mode=2;
	ret=install_int_ex(timer_callback,timer_interval);
	if(ret)//Non-zero on error.
		timer_mode=0;
	return ret;
}else
	return 0;
}
inline ulong timer_read(void)
{
return ullrounddiv(timer_read_ex() * 1000, TIMERS_PER_SECOND);
}
inline ulong timer_stop(void)
{
return ullrounddiv(timer_stop_ex() * 1000, TIMERS_PER_SECOND);
}
inline ulong timer_pause(void)
{
return ullrounddiv(timer_pause_ex() * 1000, TIMERS_PER_SECOND);
}


/**********************
****               ****
**** callback_time ****
****               ****
***********************
Function that callbacks another function as often as possible for a time
given by the parameter milliseconds. Returns negative on error, and number of
milliseconds counted on success.
*/

int callback_time(void(*callback)(int progress), int milliseconds)
{
int start=salad_timecount;
while((salad_timecount-start)<milliseconds)
	callback(salad_timecount-start);
//Do the final callback. There is a point in doing this, since we call when
//100% is done.
callback(milliseconds);
return salad_timecount-start;
}



/********************************
****                         ****
**** callback_time_nonlinear ****
****                         ****
*********************************
The same as callback_time, but the progress parameter given to the function
is fixed so that it goes slow in the beginning and fast in the end (or vice
versa, depending on the displacement parameter).
*/
int callback_time_nonlinear(void (*callback)(int progress), int milliseconds,
 int displacement)
{
int start=salad_timecount;
while((salad_timecount-start)<milliseconds)
	callback(make_nonlinear_progress(salad_timecount-start,
	 milliseconds, milliseconds, displacement));
callback(milliseconds);
return salad_timecount-start;
}




#endif
