//----------------------------------------------------------------------------
//
// DEGUI - C++ Objects for Allegro's gui 1.1 beta
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------
/*
INI-File class     Header-File
-------------------------------------------------------------
(c) 1997 M.Baas
*/

#ifndef inifile_h
#define inifile_h

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>

#include <stdlib.h>

// A single node
class node
   {
	public:
		node* succ;
		node* pred;

	public:
		node() : succ(NULL),pred(NULL) {}
		virtual ~node() {}
   };


// The list class that manages the nodes
class list
   {
	public:
		node* first;
		node* last;

	public:
		list() : first(NULL), last(NULL) {}
		~list() { free_list(); }

		void insert(node *n);
		void insert_first(node *n);
		void insert_last(node *n);
		void insert_after(node *n, node *p);
		void delete_node(node *n);
		void delete_list();
		void free_node(node *n);
		void free_list();
   };


// One row from the INI file
class row : public node
   {
	public:
		enum rowType { section, parameter, comment };

		char *rowBuffer;
		int  bufferSize;
// This should be the line number in the INI file
		int  rowNumber;

	public:
		row(int num=0) : rowBuffer(NULL),bufferSize(0),rowNumber(num) {}
		virtual ~row() { free(rowBuffer); }

// Copy s into rowBuffer
		virtual bool    set_string(char *s);
// Returns row type (section, parameter, comment)
		virtual rowType row_type() = 0;
// Returns row identifier (Section name or parameter name)
		virtual char    *get_identifier() { return rowBuffer; }
		virtual void    write(ostream &os) { os<<rowBuffer<<"\n"; }
   };


class lookupTable
   {
	protected:
		row** table;
		int   tableSize;
		int   number;           // Number of elements (always less or equal tableSize)

	public:
		lookupTable() : table(NULL),tableSize(0),number(0) {}
		~lookupTable() { clear(); };

		void clear() { if (table!=NULL) delete [] table; table = NULL; tableSize=0; number=0; }
		bool insert_at(row* r,int index);
		bool add(row* r);
		bool index_of(char* k, int &index, int &n);
		void sort();
		row* operator[](int index);
   };


// Comment
class commentRow : public row
   {
	public:
		commentRow(int num=0) : row(num) {}

		virtual rowType row_type() { return comment; }
		bool is_comment(char *s) { return *s=='#'; };
   };


// Parameter
class parameterRow : public row
   {
	private:
		char *param;
		char *value;

		char *param_end;        // position of the parameter names' 0 byte
		char dummy_char;        // original char at param_end

		int  maxvaluesize_without_allocating;
		char *newvalue;
		int  newvalue_size;
		bool use_newvalue;      // Should the allocated value be used or the original one?

	public:
		parameterRow(int num=0) : row(num), param(NULL), value(NULL),param_end(NULL),
		dummy_char(' '),maxvaluesize_without_allocating(0),newvalue(NULL),
		newvalue_size(-1),use_newvalue(false) {}
		virtual ~parameterRow() { if (newvalue!=NULL) free(newvalue); };

		virtual bool    set_string(char *s);
		virtual rowType row_type() { return parameter; }
		virtual char    *get_identifier() { return param; }
		virtual void    write(ostream &os);
		char *get_value() { return value; };
		bool set_value(char *v);
		bool is_parameter(char *s);
   };


// Section
class sectionRow : public row
   {
	public:
		parameterRow*  firstParam;
		parameterRow*  lastParam;
		char*          sectionName;
                              // Position of the 0-byte that was a ']'
			char*          sectionEnd;
		lookupTable    params;

	public:
		sectionRow(int num=0) : row(num),firstParam(NULL),lastParam(NULL),
		sectionName(NULL),sectionEnd(NULL) {}

		virtual bool    set_string(char *s);
		virtual rowType row_type() { return section; }
		virtual char    *get_identifier() { return sectionName; }
		virtual void    write(ostream &os);
		bool  is_section(char *s);
		bool  setup();
		char* get(char *paramname);
		int   set(char *paramname,char *value,bool create,list *lst);
   };


// INI file class
class INIfile
   {

	protected:
		char*       name;       // INI file name (absolute!)
		bool        changed;    // Were there parameter changes
		bool        ini_is_open;// Flag, if the object is connected to a file.

// Internal functions:
		bool   readRows();
		bool   createEmptySection();
		bool   fexpand(char *f, char *ef);


	private:
		list        rows;
		lookupTable sections;

	public:
      // Conctructors and destroctor
		INIfile(void) : name(NULL), changed(false), ini_is_open(false), errorCode("") {};
		INIfile(char* n) : name(NULL), changed(false), ini_is_open(false), errorCode("") { open(n,true); };
		virtual ~INIfile() { close(); }

      // Open an existing INI file, possibly creating
		bool  open(char *filename, bool create_flag=false);
      // Create a new INI file
		bool  create(char *filename);
      // Close the file (changes will be written now, but errors won't be reported)
		void  close();
      // Write changes
		bool  flush();

      // Query section name
		char* get_section_name(int section_nr);

		// Query parameter name
		char* get_name(int section_nr, int param_nr);


      // Write values of specific types
	   bool   set(char *section, char *param, char *value);
	   bool   set(char *section, char *param, const int i);
      bool   set(char *section, char *param, const double d);
	   bool   set(char *section, char *param, const bool b);

      // Read values of specific types
      char*  get(char *section, char *param, char *defaultparam = NULL);
	   int    get(char *section, char *param, const int i_default);
	   double get(char *section, char *param, const double d_default);
	   bool   get(char *section, char *param, const bool b_default);

      // Same as get()
      char*  operator()(char *section, char *param)           { return get(section,param);   }
	   char*  operator()(char *section, char *param, char *c)  { return get(section,param,c); }
	   int    operator()(char *section, char *param, int i)    { return get(section,param,i); }
	   double operator()(char *section, char *param, double d) { return get(section,param,d); }
	   bool   operator()(char *section, char *param, bool b)   { return get(section,param,b); }

      // Check this for an error about the last operation
		const char *errorCode;
   };


#endif



