/* pbmtonokia.c - convert a portable bitmap to Nokia Smart Messaging Formats (NOL, NGG, HEX)

** Copyright (C)2001 OMS Open Media System GmbH, Tim Rhsen <tim.ruehsen@openmediasystem.de>.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.

Notes:
  - limited to rows<=255 and columns<=255
  - supports only b/w graphics, not animated

Testing:
  Testing was done with SwissCom SMSC (Switzerland) and IC3S SMSC (Germany).
  The data was send with EMI/UCP protocol over TCP/IP.

  - 7.6.2001: tested with Nokia 3210: 72x14 Operator Logo
  - 7.6.2001: tested with Nokia 6210: 72x14 Operator Logo and 72x14 Group Graphic

Todo:
  - more testing
  - sendsms compatibility ?
  - are -fmt NOL and -fmt NGG working ok?
*/

#include "pbm.h"
#include "string.h"

#define FMT_HEX_NOL   1
#define FMT_HEX_NGG   2
#define FMT_NOL       3
#define FMT_NGG       4

void usage()
{
    pm_message("Copyright (C)2001 OMS GmbH\n");
    pm_message("contact: Tim Ruehsen <tim.ruehsen@openmediasystem.de>\n");
    pm_usage("[-fmt HEX_NOL|HEX_NGG|NOL|NGG] [pbmfile]\ndefault: fmt=HEX_NOL");
    exit(1); // just to be shure
}

int main(int argc, char *argv[])
{
    FILE    *fp;
    bit **image;
    unsigned int    c;
    int argpos, output=FMT_HEX_NOL, rows, cols, row, col, p;
    char    header[32];

    pbm_init(&argc, argv);

    for(argpos=1;argpos<argc;argpos++) {
        if (argv[argpos][0]=='-') {
            if (argv[argpos][1]=='-') {
                if (argc>argpos+1 && isdigit(argv[argpos+1][0]))
                    {argpos++;break;}
            } else if (!strcmp(argv[argpos],"-fmt") && argc>argpos+1) {
                ++argpos;
                if (!strcasecmp(argv[argpos],"HEX_NOL")) output=FMT_HEX_NOL;
                else if (!strcasecmp(argv[argpos],"HEX_NGG")) output=FMT_HEX_NGG;
                else if (!strcasecmp(argv[argpos],"NOL")) output=FMT_NOL;
                else if (!strcasecmp(argv[argpos],"NGG")) output=FMT_NGG;
                else usage();
            }
            else usage();
        } else break;
    }

    if (argpos==argc) {
        image = pbm_readpbm(stdin, &cols, &rows);
    } else {
        fp=pm_openr(argv[argpos]);
        image = pbm_readpbm(fp, &cols, &rows);
        pm_close(fp);
    }

    memset(header,0,sizeof(header));

    switch (output) {
    case FMT_HEX_NOL:
        // header
        printf("0605041582000062F21000%02X%02X01",cols,rows);

        // image
        for (row=0;row<rows;row++) {
            for (p=c=col=0;col<cols;col++) {
                if (image[row][col]==PBM_BLACK) c|=0x80>>p;
                if (++p==8) {
                    printf("%02X",c);
                    p=c=0;
                }
            }
            if (p) printf("%02X",c);
        }
        break;
    case FMT_HEX_NGG:
        // header
        printf("0605041583000000%02X%02X01",cols,rows);

        // image
        for (row=0;row<rows;row++) {
            for (p=c=col=0;col<cols;col++) {
                if (image[row][col]==PBM_BLACK) c|=0x80>>p;
                if (++p==8) {
                    printf("%02X",c);
                    p=c=0;
                }
            }
            if (p) printf("%02X",c);
        }
        break;
    case FMT_NOL:
        // header - this is a hack
        header[1]=header[4]=header[5]=header[11]=header[13]=1;
        header[3]=4;
        header[7]=cols;
        header[9]=rows;
        header[15]=0x53;
        fwrite(header,17,1,stdout);

        // image
        for (row=0;row<rows;row++) {
            for (p=c=col=0;col<cols;col++) {
                if (image[row][col]==PBM_BLACK) putchar('1');
                else putchar('0');
            }
        }
        break;
    case FMT_NGG:
        // header - this is a hack
        header[1]=header[7]=header[9]=1;
        header[3]=cols;
        header[5]=rows;
        header[11]=0x4a;
        fwrite(header,13,1,stdout);

        // image
        for (row=0;row<rows;row++) {
            for (p=c=col=0;col<cols;col++) {
                if (image[row][col]==PBM_BLACK) putchar('1');
                else putchar('0');
            }
        }
        break;
    default:
        pm_error("Output format %d not implemented!\n"
                 "Contact Tim Ruehsen <tim.ruehsen@openmediasystem.de>\n",
                 output);
    }

    if (isatty(STDOUT_FILENO)) printf("\n");

    return 0;
}
