## Copyright (c) 2014-2020 Michael Hirsch, Ph.D.
## Copyright (c) 2013-2020, Felipe Geremia Nievinski
## Copyright (C) 2019-2020 Philip Nienhuis
##
## Redistribution and use in source and binary forms, with or without 
## modification, are permitted provided that the following conditions are met:
## 1. Redistributions of source code must retain the above copyright notice, 
##    this list of conditions and the following disclaimer.
## 2. Redistributions in binary form must reproduce the above copyright notice, 
##    this list of conditions and the following disclaimer in the documentation 
##    and/or other materials provided with the distribution.
## THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
## AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
## THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
## ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
## LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
## CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
## SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
## OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
## WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
## (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
## SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

## -*- texinfo -*-
## @deftypefn {Function File} {@var{az}, @var{el}, @var{slantrange} =} enu2aer (@var{east}, @var{n}, @var{up})
## @deftypefnx {Function File} {@var{az}, @var{el}, @var{slantrange} =} enu2aer (@dots{...}, @var{angleUnit})
## Converts ENU coordinates to azimuth, elevation, slantrange
##
## Inputs:
## @itemize
## @item
## @var{east}, @var{n}, @var{up}:  East, North, Up coordinates of test points (meters)
##
## @item
## (Optional) angleUnit: string for angular units ('radians' or 'degrees', 
## case-insensitive, only first character is significant).
## Default is 'd': degrees
## @end itemize
##
## Outputs:
## @itemize
## @item
## @var{az}, @var{el}, @var{slantrange}: look angles and distance to point
## under test (degrees, degrees, meters)
## @end itemize
##
## Example:
## @example
## [az, el, slantrange] = enu2aer (186.28, 286.84, 939.69)
## az =  33.001
## el =  70.000
## slantrange = 1000.00
## @end example
##
## With radians
## @example
## [az, el, slantrange] = enu2aer (353.55, 353.55, 866.03,"r")
## az =  0.78540
## el =  1.0472
## slantrange =  1000.0
## @end example
##
## @end deftypefn

## Function adapted by anonymous contributor, see:
## https://savannah.gnu.org/patch/index.php?8377

function [az, el, slantrange] = enu2aer (east, n, up, angleUnit = "degrees")
  
  if (nargin < 3 || nargin > 4)
    print_usage();
  endif 
  
  if (! isnumeric (east)  || ! isreal (east) || ...
      ! isnumeric (n)     || ! isreal (n) || ...
      ! isnumeric (up)    || ! isreal (up))
    error ("enu2aer.m : numeric values expected");
  endif 

  if (! all (size (east) == size (n)) || ! all (size (n) == size (up)))
    error ("enu2aer.m: non-matching dimensions of inputs.");
  endif

  if (! ischar (angleUnit))
    error ("enu2ear.m: character value expected for 'angleUnit'");
  elseif (! strncmpi (angleUnit, "degrees", length (angleUnit)) &&
          ! strncmpi (angleUnit, "radians", length (angleUnit)))
    error ("enu2aer.m: illegal input for 'angleUnit'");
  endif

  r = hypot (east, n);
  slantrange = hypot (r, up);
  ## Radians
  el = atan2 (up, r);
  az = mod (atan2 (east, n), 2 * atan2 (0, -1));
  
  if (nargin == 3) 
    az = rad2deg (az);
    el = rad2deg (el);
  endif 
  
endfunction


%!test
%! [az, el, slantrange] = enu2aer (186.277521, 286.84222, 939.69262);
%! assert ([az, el, slantrange], [33, 70, 1e3], 10e-6)
%! [az, el, slantrange] = enu2aer (186.277521, 286.84222, 939.69262, "rad");
%! assert ([az, el, slantrange], [0.57595865, 1.221730476, 1e3], 10e-6)

%!error <numeric> enu2aer ("s", 25, 1e3)
%!error <numeric> enu2aer (3i, 25, 1e3)
%!error <numeric> enu2aer (33, "s", 1e3)
%!error <numeric> enu2aer (33, 3i, 1e3)
%!error <numeric> enu2aer (33, 25, "s")
%!error <numeric> enu2aer (33, 25, 3i)
%!error <non-matching> enu2aer ([1 1], [2 2]', [4 5])
%!error <non-matching> enu2aer ([1 1], [2 2], [4 5 6])
%!error <illegal> enu2aer (33, 70, 1e3, "f");
%!error <illegal> enu2aer (33, 70, 1e3, "radianf");
