/**
 * @file
 * Test code for mutt_path_tidy()
 *
 * @authors
 * Copyright (C) 2019-2023 Richard Russon <rich@flatcap.org>
 * Copyright (C) 2023 Dennis Schön <mail@dennis-schoen.de>
 *
 * @copyright
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define TEST_NO_MAIN
#include "config.h"
#include "acutest.h"
#include <stdbool.h>
#include <stddef.h>
#include "mutt/lib.h"
#include "test_common.h"

void test_mutt_path_tidy(void)
{
  // bool mutt_path_tidy(struct Buffer *path);

  static const char *tests[][2] = {
    // clang-format off
    // This struct is VERY wide, keep reading...
    { "/..apple/./../////./banana/banana/./banana/..apple/./banana/..apple/banana///banana/..apple/banana/..apple/banana/banana/..apple",                                     "/banana/banana/banana/..apple/banana/..apple/banana/banana/..apple/banana/..apple/banana/banana/..apple",                                      },
    { "/../../banana///..apple///..apple///banana///banana/banana/banana/..apple/banana/banana/banana/./banana/banana/banana/..apple/banana",                                 "/banana/..apple/..apple/banana/banana/banana/banana/..apple/banana/banana/banana/banana/banana/banana/..apple/banana",                         },
    { "///banana/banana/banana/./..apple/../banana/..apple/../..apple/./banana/./..apple",                                                                                    "/banana/banana/banana/banana/..apple/banana/..apple",                                                                                          },
    { "/./banana/banana/../banana/banana/.///banana/..apple/..apple",                                                                                                         "/banana/banana/banana/banana/..apple/..apple",                                                                                                 },
    { "/../banana/banana/banana/banana///..apple///..apple/banana/banana/////./..apple/./../.././banana/banana///banana/banana",                                              "/banana/banana/banana/banana/..apple/..apple/banana/banana/banana/banana/banana",                                                              },
    { "/banana/banana/./././..apple/banana///./banana/banana/banana/banana/banana/banana/../////banana/banana/banana/./..apple/..apple/..///..apple",                         "/banana/banana/..apple/banana/banana/banana/banana/banana/banana/banana/banana/banana/..apple/..apple",                                        },
    { "/banana///..apple///../banana/banana/banana///////banana/banana/./..apple/..apple/./..apple/..apple/banana",                                                           "/banana/banana/banana/banana/banana/banana/..apple/..apple/..apple/..apple/banana",                                                            },
    { "/banana/..apple/..apple/..apple/..apple/banana///../..apple///banana/banana/banana/banana///./../..apple/../banana/..apple/../banana/banana/./..apple",                "/banana/..apple/..apple/..apple/..apple/..apple/banana/banana/banana/banana/banana/banana/..apple",                                            },
    { "/banana/banana/..///../banana/../banana/banana/..apple/./../banana/../../banana/.",                                                                                    "/banana/banana",                                                                                                                               },
    { "/banana/banana/../..apple/banana/././banana///banana/banana",                                                                                                          "/banana/..apple/banana/banana/banana/banana",                                                                                                  },
    { "/////banana/banana/banana///..apple/./banana/..apple/./banana/banana",                                                                                                 "/banana/banana/banana/..apple/banana/..apple/banana/banana",                                                                                   },
    { "/..apple/..apple/banana///banana/././//.///./banana///./banana/..apple/./banana",                                                                                      "/..apple/..apple/banana/banana/banana/banana/..apple/banana",                                                                                  },
    { "///./..apple/banana/./../banana/././..apple///./../../../////banana/banana/../..apple/banana/banana/../banana/banana/../.",                                            "/banana/..apple/banana/banana",                                                                                                                },
    { "/banana/./../././../..apple/banana/banana/..///../.",                                                                                                                  "/..apple",                                                                                                                                     },
    { "/./..apple/banana///./banana/..///../banana//",                                                                                                                        "/..apple/banana",                                                                                                                              },
    { "/.///banana///..apple/banana/banana/../.././banana/../..apple///banana/banana/./banana/banana/..//",                                                                   "/banana/..apple/..apple/banana/banana/banana",                                                                                                 },
    { "/..apple/..apple/../banana/banana/..apple/./banana/../banana///banana",                                                                                                "/..apple/banana/banana/..apple/banana/banana",                                                                                                 },
    { "/banana/banana/../././banana/banana/banana///./.././//banana/banana/banana/.././banana///..apple/banana//",                                                            "/banana/banana/banana/banana/banana/banana/..apple/banana",                                                                                    },
    { "/banana/banana/../banana/./banana/banana/banana/..apple/../banana/.///banana/////../..apple/banana/banana/../..apple/banana/banana/banana///banana",                   "/banana/banana/banana/banana/banana/banana/..apple/banana/..apple/banana/banana/banana/banana",                                                },
    { "/./..apple/./banana///banana/./banana/..apple/banana///.///././banana",                                                                                                "/..apple/banana/banana/banana/..apple/banana/banana",                                                                                          },
    { "/./banana/..apple/banana/banana/.././.././..apple/banana/banana/..apple/.///..apple/.///banana/banana/..",                                                             "/banana/..apple/..apple/banana/banana/..apple/..apple/banana",                                                                                 },
    { "///./../..apple/banana/../banana///banana///..///..apple/../banana/../../banana/..apple/./banana/..apple/banana/..apple/banana//",                                     "/..apple/banana/..apple/banana/..apple/banana/..apple/banana",                                                                                 },
    { "/banana/../..apple/banana///////banana/banana/..apple/../banana/../..",                                                                                                "/..apple/banana/banana",                                                                                                                       },
    { "/../banana/..apple///banana/banana/..apple/..apple///banana/banana/banana///..apple/banana///../././banana/banana/banana/banana/banana/banana",                        "/banana/..apple/banana/banana/..apple/..apple/banana/banana/banana/..apple/banana/banana/banana/banana/banana/banana",                         },
    { "///..apple///.././banana/./..apple///..apple/..",                                                                                                                      "/banana/..apple",                                                                                                                              },
    { "///../..apple/./../..apple/banana/banana///..apple/banana///../banana/banana",                                                                                         "/..apple/banana/banana/..apple/banana/banana",                                                                                                 },
    { "/../banana/banana/banana/./banana/banana/banana///banana/banana/./banana/.",                                                                                           "/banana/banana/banana/banana/banana/banana/banana/banana/banana",                                                                              },
    { "/././..apple/./..apple/../banana/./..apple/banana///.././banana/banana/..",                                                                                            "/..apple/banana/..apple/banana",                                                                                                               },
    { "/..apple/..apple///banana/banana/..apple/////banana/banana/..apple///./../banana/banana/banana///banana/..apple/banana/..apple////",                                   "/..apple/..apple/banana/banana/..apple/banana/banana/banana/banana/banana/banana/..apple/banana/..apple",                                      },
    { "/..apple/banana/./banana/banana/banana/./banana/banana/../banana/../..///..apple/banana/./.././..///././../..apple/../banana/banana//",                                "/..apple/banana/banana/banana/banana/banana",                                                                                                  },
    { "/banana///../banana/../././..apple/..apple///.///banana/./banana/banana///banana/..apple/.",                                                                           "/..apple/..apple/banana/banana/banana/banana/..apple",                                                                                         },
    { "/////..apple/banana/banana/..apple/banana///banana//",                                                                                                                 "/..apple/banana/banana/..apple/banana/banana",                                                                                                 },
    { "/..apple///./banana///../../../..apple/..apple/..apple/./banana/banana",                                                                                               "/..apple/..apple/..apple/banana/banana",                                                                                                       },
    { "///banana///././..apple/banana/banana/././..apple/..apple/..apple/banana///././banana/././banana/..apple/banana/banana/../banana/./banana",                            "/banana/..apple/banana/banana/..apple/..apple/..apple/banana/banana/banana/..apple/banana/banana/banana",                                      },
    { "/banana///./banana/banana/..///./banana//",                                                                                                                            "/banana/banana/banana",                                                                                                                        },
    { "/banana/////banana/banana/..apple/..apple/////.///..///..apple/banana/banana/..apple/..apple///./banana",                                                              "/banana/banana/banana/..apple/..apple/banana/banana/..apple/..apple/banana",                                                                   },
    { "/..apple/banana///../..apple/////./..apple/./././banana/..apple",                                                                                                      "/..apple/..apple/..apple/banana/..apple",                                                                                                      },
    { "/banana/banana///banana/../../../..apple/banana///..apple/..apple/../.././banana/..apple/..apple/..///../../..",                                                       "/..apple",                                                                                                                                     },
    { "/..apple/./././../banana/..apple/banana/banana/////./..//",                                                                                                            "/banana/..apple/banana",                                                                                                                       },
    { "/../..apple/banana/..apple/banana/.././////banana/../banana/banana/..apple/..apple/banana/banana",                                                                     "/..apple/banana/..apple/banana/banana/..apple/..apple/banana/banana",                                                                          },
    { "/..apple/..apple/..apple///banana/banana/../banana/banana/banana/banana/banana/banana/..apple/.///./banana/./..apple/..apple/./..apple/banana/banana/banana/banana/.", "/..apple/..apple/..apple/banana/banana/banana/banana/banana/banana/banana/..apple/banana/..apple/..apple/..apple/banana/banana/banana/banana", },
    { "///..///banana///../..apple/..apple/.///banana/banana/..apple/..apple/banana/././..///banana",                                                                         "/..apple/..apple/banana/banana/..apple/..apple/banana",                                                                                        },
    { "/banana///banana/..apple/banana/..///.././..apple/banana///banana/banana/..apple///./..apple",                                                                         "/banana/banana/..apple/banana/banana/banana/..apple/..apple",                                                                                  },
    { "/banana/banana///.././banana/./banana/..apple/.././banana/../banana/////../banana/./banana/../..apple/banana/../banana/./..",                                          "/banana/banana/banana/banana/..apple",                                                                                                         },
    { "/banana/..apple/..apple/.././//banana/banana///.////",                                                                                                                 "/banana/..apple/banana/banana",                                                                                                                },
    { "/banana/.././banana/banana/banana/.///../banana/..",                                                                                                                   "/banana/banana",                                                                                                                               },
    { "/banana/.///..apple/../banana/banana/banana/../..apple///./banana/banana///./.",                                                                                       "/banana/banana/banana/..apple/banana/banana",                                                                                                  },
    { "/..apple/..apple///../..apple/..apple/banana/banana/////../banana/banana/////../banana/./.././banana/..apple",                                                         "/..apple/..apple/..apple/banana/banana/banana/..apple",                                                                                        },
    { "/./../banana/banana///banana/////./..apple/./..apple/../././..apple///banana",                                                                                         "/banana/banana/banana/..apple/..apple/banana",                                                                                                 },
    { "/..///banana/../banana/./..apple/..apple///././banana",                                                                                                                "/banana/..apple/..apple/banana",                                                                                                               },
    { "/banana/banana/banana/banana/banana/banana/banana/../banana/banana/banana/banana/banana/banana/..apple/../..apple/..apple",                                            "/banana/banana/banana/banana/banana/banana/banana/banana/banana/banana/banana/banana/..apple/..apple",                                         },
    { "/banana/.././banana/..///banana/..apple/banana/banana/..apple",                                                                                                        "/banana/..apple/banana/banana/..apple",                                                                                                        },
    { "/../banana/banana/../..///..apple/banana/..apple/../../..apple/banana/..apple/../banana/..apple/banana/..apple///../banana/banana/banana/../banana/..apple/banana/.",  "/..apple/..apple/banana/banana/..apple/banana/banana/banana/banana/..apple/banana",                                                            },
    { "/banana/banana/..apple/./banana/./././banana/..apple/////..apple/banana/banana/banana////",                                                                            "/banana/banana/..apple/banana/banana/..apple/..apple/banana/banana/banana",                                                                    },
    { "/..apple/banana/banana/../banana/banana/../..apple/banana/banana/./..",                                                                                                "/..apple/banana/banana/..apple/banana",                                                                                                        },
    { "/.///..apple/banana/banana/banana/../banana/banana///banana/banana///banana/banana/./..apple/..///banana/..apple/banana/banana///../banana/..apple/banana",            "/..apple/banana/banana/banana/banana/banana/banana/banana/banana/banana/..apple/banana/banana/..apple/banana",                                 },
    { "/.///./../../banana/../banana///banana/banana///banana///banana///banana",                                                                                             "/banana/banana/banana/banana/banana/banana",                                                                                                   },
    { "/banana/banana/./banana/../../../banana/././..apple/.././banana///..apple/../.",                                                                                       "/banana/banana",                                                                                                                               },
    { "///./../.././../../..apple/banana/..apple/..apple/banana///banana/..apple///../banana/../banana/././..apple/../..apple/./banana/.",                                    "/..apple/banana/..apple/..apple/banana/banana/banana/..apple/banana",                                                                          },
    { "/./../banana/banana///../banana/..apple/../../banana/banana/banana/banana/banana/../////banana/./banana//",                                                            "/banana/banana/banana/banana/banana/banana/banana",                                                                                            },
    { "/banana/./../.././../../banana/../../..apple///.///banana/banana/..apple/./banana/banana/banana/./banana/..apple/banana/..apple",                                      "/..apple/banana/banana/..apple/banana/banana/banana/banana/..apple/banana/..apple",                                                            },
    { "/..apple/.././banana/banana/banana/../../././//../../..apple/banana///../..apple/banana/././..apple///././banana",                                                     "/..apple/..apple/banana/..apple/banana",                                                                                                       },
    { "///../banana/.././banana/../..apple///banana/./../../..apple",                                                                                                         "/..apple",                                                                                                                                     },
    { "/banana/banana/banana/////../..apple/banana/////./banana///banana/..apple/banana/..apple/banana/.///banana/../../..",                                                  "/banana/banana/..apple/banana/banana/banana/..apple/banana",                                                                                   },
    { "///banana/banana/banana/..apple/banana/./..apple///./..apple/.",                                                                                                       "/banana/banana/banana/..apple/banana/..apple/..apple",                                                                                         },
    { "/./././banana/././banana///../////../banana/./../////../banana///..apple///..apple/./.././banana/..apple//",                                                           "/banana/..apple/banana/..apple",                                                                                                               },
    { "/banana/..apple/./../..apple/..apple/banana///./.././banana/./../..apple/banana/banana",                                                                               "/banana/..apple/..apple/..apple/banana/banana",                                                                                                },
    { "/..apple/..apple/..apple///////banana/banana/banana/banana/////./banana/banana/./banana///../.",                                                                       "/..apple/..apple/..apple/banana/banana/banana/banana/banana/banana",                                                                           },
    { "/..apple/../..apple///////banana/./..apple/./banana/../..apple/../../banana/banana///banana/banana/./..///.././..",                                                    "/..apple/banana/banana",                                                                                                                       },
    { "/./.././////banana/banana/..apple/././banana/banana/banana///./.",                                                                                                     "/banana/banana/..apple/banana/banana/banana",                                                                                                  },
    { "/banana/./../banana///././..apple/////banana///..///banana/banana///..apple",                                                                                          "/banana/..apple/banana/banana/..apple",                                                                                                        },
    { "/banana/../banana/../////..apple/banana///./////banana/./..apple/..apple///banana///banana/../banana///banana/..apple",                                                "/..apple/banana/banana/..apple/..apple/banana/banana/banana/..apple",                                                                          },
    { "/banana/banana/..apple/banana/./banana/banana/../banana///.",                                                                                                          "/banana/banana/..apple/banana/banana/banana",                                                                                                  },
    { "/..apple/..apple///./banana/./..apple/../..apple/./../banana/banana/..apple/././banana/..apple/////../../banana",                                                      "/..apple/..apple/banana/banana/banana/..apple/banana",                                                                                         },
    { "/..apple/..///banana///..apple/../banana/../..",                                                                                                                       "/",                                                                                                                                            },
    { "/banana///banana/banana/./banana/../../..apple/./banana/banana/.././//banana/..apple/..apple/banana/banana/.///banana/./banana/..///../..",                            "/banana/banana/..apple/banana/banana/..apple/..apple/banana",                                                                                  },
    { "/..apple/banana/..apple/.././//./..///banana///banana///../..///banana///..apple///.././../banana/../../.",                                                            "/",                                                                                                                                            },
    { "/./banana/..apple/banana/..///./banana/../../.././../../banana/banana/banana/../..apple/banana/banana/..apple/banana/banana/.",                                        "/banana/banana/..apple/banana/banana/..apple/banana/banana",                                                                                   },
    { "/../banana/banana/banana/..apple/..///./banana/..apple///../..apple/././../..apple/banana/./.././..//",                                                                "/banana/banana/banana/banana",                                                                                                                 },
    { "///banana///../../banana///.././//../banana/banana/..apple/banana///banana/banana/banana/..apple/..",                                                                  "/banana/banana/..apple/banana/banana/banana/banana",                                                                                           },
    { "/banana/../banana/././banana/..apple/./..apple///../..apple/.././////banana/./..apple/./banana",                                                                       "/banana/banana/..apple/banana/..apple/banana",                                                                                                 },
    { "/banana/./..apple/../..apple/./banana/..apple/../banana/banana/banana/banana/banana/banana/banana",                                                                    "/banana/..apple/banana/banana/banana/banana/banana/banana/banana/banana",                                                                      },
    { "/.././..apple///banana///..apple///banana/banana/banana/..apple/banana/./banana/.././banana/././/",                                                                    "/..apple/banana/..apple/banana/banana/banana/..apple/banana/banana",                                                                           },
    { "///././../banana/./../../..apple/banana/banana/..apple/banana/../..apple/..apple/./banana/./banana/..apple///banana/./..apple/banana///banana",                        "/..apple/banana/banana/..apple/..apple/..apple/banana/banana/..apple/banana/..apple/banana/banana",                                            },
    { "/..apple/banana/banana/banana///banana/..///./..apple/banana/banana/..apple/banana///.///../banana/..apple",                                                           "/..apple/banana/banana/banana/..apple/banana/banana/..apple/banana/..apple",                                                                   },
    { "/../..apple/banana/../banana/banana/banana/banana///..apple/./..apple/../..apple/..",                                                                                  "/..apple/banana/banana/banana/banana/..apple",                                                                                                 },
    { "/../banana/banana/banana/..apple/banana/../banana/banana/../../../..apple///banana/../banana",                                                                         "/banana/banana/banana/..apple/banana",                                                                                                         },
    { "/banana/..apple/..apple/../banana/banana/////../././banana/banana/..apple/..apple/.",                                                                                  "/banana/..apple/banana/banana/banana/..apple/..apple",                                                                                         },
    { "/././//banana/banana/..apple/./banana/./banana///..apple/..",                                                                                                          "/banana/banana/..apple/banana/banana",                                                                                                         },
    { "/../banana/banana///./..apple/banana/banana///.././banana/banana/.///./banana/banana/banana/banana",                                                                   "/banana/banana/..apple/banana/banana/banana/banana/banana/banana/banana",                                                                      },
    { "/banana/banana/banana/..apple/./././../..apple/banana/..apple/..apple/.///.././..",                                                                                    "/banana/banana/banana/..apple/banana",                                                                                                         },
    { "///..apple/./..apple/..apple/banana/banana/banana/../////.//",                                                                                                         "/..apple/..apple/..apple/banana/banana",                                                                                                       },
    { "/../banana/../../..apple/..apple///..apple/././banana/./banana/..apple///./..apple/./banana/banana/banana/./.././banana/../..",                                        "/..apple/..apple/..apple/banana/banana/..apple/..apple/banana",                                                                                },
    { "/..apple/..apple/banana///..apple///..apple/..apple/banana/.././banana/..apple/././..apple/../..apple///..apple///..apple/banana/../banana/..apple/////banana",        "/..apple/..apple/banana/..apple/..apple/..apple/banana/..apple/..apple/..apple/..apple/banana/..apple/banana",                                 },
    { "/../..apple/././banana///../..apple/banana/../.././////banana/banana/../..apple",                                                                                      "/..apple/banana/..apple",                                                                                                                      },
    { "/banana/..apple/banana/banana///..apple/banana/../banana/.././/",                                                                                                      "/banana/..apple/banana/banana/..apple",                                                                                                        },
    { "/..apple/banana/banana/banana/./banana/../banana/banana///..apple/banana/..///..///.",                                                                                 "/..apple/banana/banana/banana/banana/banana",                                                                                                  },
    { "/..apple/banana/banana/.././banana/..apple/banana/..apple/..apple/../..///..apple///banana/banana/banana///banana/..apple/banana/banana",                              "/..apple/banana/banana/..apple/banana/..apple/banana/banana/banana/banana/..apple/banana/banana",                                              },
    { "/./banana///../banana/banana/./../..apple/banana/../../banana///banana/..apple/..apple/////..",                                                                        "/banana/banana/banana/..apple",                                                                                                                },
    { "/banana/..apple/banana///banana///./..apple/banana/banana/banana/..apple/banana/banana//",                                                                             "/banana/..apple/banana/banana/..apple/banana/banana/banana/..apple/banana/banana",                                                             },
    // clang-format on
  };

  {
    TEST_CHECK(!mutt_path_tidy(NULL, true));
  }

  {
    struct Buffer *path = buf_pool_get();
    for (size_t i = 0; i < mutt_array_size(tests); i++)
    {
      TEST_CASE(tests[i][0]);

      buf_strcpy(path, tests[i][0]);
      mutt_path_tidy(path, true);
      TEST_CHECK_STR_EQ(buf_string(path), tests[i][1]);
    }
    buf_pool_release(&path);
  }
}
